﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Device Tree のプロパティ情報取得用 API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Result.h>
#include <nn/dt/dt_Types.h>
#include <nn/dt/detail/dt_PropertyApi.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace dt {

//! @name プロパティ情報取得用 API
//! @{

/**
 * @brief       プロパティの有無を取得します。
 *
 * @param[in]   pNode ノード
 * @param[in]   name  プロパティの名前
 *
 * @return      プロパティが存在すれば true
 *
 * @details     name で指定した名前を持つ pNode のプロパティが存在すれば、true を返します。
 *              存在しなければ、false を返します。
 *
 * @internal
 * NN_DETAIL_NO_C_IMPLEMENTATION
 * @endinternal
 *
 */
bool IsPropertyExist(const Node* pNode, const char* name) NN_NOEXCEPT;

/**
 * @brief       プロパティの有無を取得します。
 *
 * @param[out]  pOut  プロパティが存在すれば true
 * @param[in]   pNode ノード
 * @param[in]   name  プロパティの名前
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 * @endretresult
 *
 * @details     name で指定した名前を持つ pNode のプロパティが存在すれば、pOut に true を格納します。
 *              存在しなければ、pOut に false を格納します。
 *
 */
nn::Result IsPropertyExist(bool *pOut, const Node* pNode, const char* name) NN_NOEXCEPT;

/**
 * @brief       プロパティのサイズを取得します。
 *
 * @param[out]  pOutSize  プロパティのサイズ
 * @param[in]   pNode     ノード
 * @param[in]   name      プロパティの名前
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyNotExist}
 * @endretresult
 *
 * @details     name で指定した名前を持つ pNode のプロパティを取得し、そのサイズを pOutSize に格納します。
 *
 */
nn::Result GetPropertySize(size_t* pOutSize, const Node* pNode, const char* name) NN_NOEXCEPT;

/**
 * @brief       プロパティを整数型 T の配列とみなしたときの配列長を取得します。
 *
 * @tparam      T         プロパティの型
 * @param[out]  pOutCount 配列長
 * @param[in]   pNode     ノード
 * @param[in]   name      プロパティの名前
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyNotExist}
 *   @handleresult{nn::dt::ResultPropertyTypeError}
 * @endretresult
 *
 * @details     name で指定した名前を持つ pNode のプロパティを取得し、それを整数型 T の配列であるとみなして、
 *              配列長を pOutCount に格納します。
 *              T のサイズは 1 バイト / 2 バイト / 4 バイト / 8 バイト のいずれかである必要があります。
 *              プロパティが整数型 T の配列ではないと判明した場合には ResultPropertyTypeError を返します。
 *              ただし、型チェックは完璧ではなく、実際の型が T の配列でなくても ResultSuccess を返す場合があるので注意してください。
 *
 */
template <typename T>
nn::Result GetPropertyCount(int *pOutCount, const Node* pNode, const char* name) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(detail::IsSupportedPropertySize(sizeof(T)));

    return detail::GetPropertyCount(pOutCount, pNode, name, sizeof(T));
}

/**
 * @brief       プロパティに格納されている整数型 T の値を取得します。
 *
 * @tparam      T         プロパティの型
 * @param[out]  pOutValue T 型の値
 * @param[in]   pNode     ノード
 * @param[in]   name      プロパティの名前
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyNotExist}
 *   @handleresult{nn::dt::ResultPropertyTypeError}
 * @endretresult
 *
 * @details     name で指定した名前を持つ pNode のプロパティを取得し、それを整数型 T であるとみなして、
 *              その値を pOutValue に格納します。
 *              T のサイズは 1 バイト / 2 バイト / 4 バイト / 8 バイト のいずれかである必要があります。
 *              プロパティが整数型 T でないと判明した場合には ResultPropertyTypeError を返します。
 *              ただし、型チェックは完璧ではなく、実際の型が T 以外でも ResultSuccess を返す場合があるので注意してください。
 *
 */
template <typename T>
nn::Result GetProperty(T* pOutValue, const Node* pNode, const char* name) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(detail::IsSupportedPropertySize(sizeof(T)));

    return detail::GetProperty(reinterpret_cast<char*>(pOutValue), pNode, name, sizeof(T));
}

/**
 * @brief       プロパティに格納されている整数型 T の値を取得します。
 *
 * @tparam      T         プロパティの型
 * @param[out]  pOutValue T 型の値
 * @param[in]   pNode     ノード
 * @param[in]   name      プロパティの名前
 * @param[in]   index     T の配列における、取得する値のインデックス
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyNotExist}
 *   @handleresult{nn::dt::ResultPropertyTypeError}
 * @endretresult
 *
 * @details     name で指定した名前を持つ pNode のプロパティを取得し、それを整数型 T の配列であるとみなして、
 *              index で指定したインデックスに格納されている値を pOutValue に格納します。
 *              T のサイズは 1 バイト / 2 バイト / 4 バイト / 8 バイト のいずれかである必要があります。
 *              プロパティが整数型 T の配列でないと判明した場合には ResultPropertyTypeError を返します。
 *              ただし、型チェックは完璧ではなく、実際の型が T の配列でなくても ResultSuccess を返す場合があるので注意してください。
 *
 */
template <typename T>
nn::Result GetProperty(T* pOutValue, const Node* pNode, const char* name, int index) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(detail::IsSupportedPropertySize(sizeof(T)));

    return detail::GetProperty(reinterpret_cast<char*>(pOutValue), pNode, name, index, sizeof(T));
}

/**
 * @brief       プロパティに格納されている整数型 T の配列を取得します。
 *
 * @tparam      T         プロパティの型
 * @param[out]  pOutArray 整数型 T の配列
 * @param[in]   count     配列の長さ
 * @param[in]   pNode     ノード
 * @param[in]   name      プロパティの名前
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyNotExist}
 *   @handleresult{nn::dt::ResultPropertyTypeError}
 *   @handleresult{nn::dt::ResultSizeIncorrect}
 * @endretresult
 *
 * @details     name で指定した名前を持つ pNode のプロパティを取得し、それを整数型 T の配列であるとみなして、
 *              pOutArray に格納します。
 *              T のサイズは 1 バイト / 2 バイト / 4 バイト / 8 バイト のいずれかである必要があります。
 *              プロパティが整数型 T の配列でないと判明した場合には ResultPropertyTypeError を返します。
 *              ただし、型チェックは完璧ではなく、実際の型が T の配列でなくても ResultSuccess を返す場合があるので注意してください。
 *              プロパティの配列長が count と一致しなかった場合には ResultSizeIncorrect を返します。
 *              count との一致比較を行いたくない場合は、@ref nn::dt::GetPropertyList() を使用してください。
 *
 */
template <typename T>
nn::Result GetPropertyArray(T* pOutArray, int count, const Node* pNode, const char* name) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(detail::IsSupportedPropertySize(sizeof(T)));

    return detail::GetPropertyArray(reinterpret_cast<char*>(pOutArray), count, pNode, name, sizeof(T));
}

/**
 * @brief       プロパティに格納されている整数型 T の配列を取得します。
 *
 * @tparam      T         プロパティの型
 * @param[out]  pOutList  整数型 T の配列
 * @param[in]   pOutCount 配列の長さ
 * @param[in]   maxCount  pOutArray に格納できるノードの最大数
 * @param[in]   pNode     ノード
 * @param[in]   name      プロパティの名前
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyNotExist}
 *   @handleresult{nn::dt::ResultPropertyTypeError}
 * @endretresult
 *
 * @details     基本的な仕様は @ref nn::dt::GetPropertyArray() と同等ですが、配列長の一致比較を行わない点が異なります。
 *              プロパティの配列長が maxCount と一致しなかった場合には、可能なだけ pOutList に配列をコピーします。
 *
 */
template <typename T>
nn::Result GetPropertyList(T* pOutList, int* pOutCount, int maxCount, const Node* pNode, const char* name) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(detail::IsSupportedPropertySize(sizeof(T)));

    return detail::GetPropertyList(reinterpret_cast<char*>(pOutList), pOutCount, maxCount, pNode, name, sizeof(T));
}

/**
 * @brief       プロパティに格納されている文字列または stringlist を取得します。
 *
 * @param[out]  pOutSize    取得したプロパティのサイズ
 * @param[out]  pOutBuffer  文字列または stringlist を格納するバッファ
 * @param[in]   bufferSize  pOutBuffer のサイズ
 * @param[in]   pNode       ノード
 * @param[in]   name        プロパティの名前
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultBufferTooSmall}
 *   @handleresult{nn::dt::ResultPropertyNotExist}
 *   @handleresult{nn::dt::ResultPropertyTypeError}
 * @endretresult
 *
 * @details     name で指定した名前を持つ pNode のプロパティを取得し、それを文字列または stringlist であるとみなして、
 *              pOutBuffer に格納します。
 *              また、終端文字を含むプロパティ全体のサイズを pOutSize に格納します。
 *              プロパティが文字列または stringlist でないと判明した場合には ResultPropertyTypeError を返します。
 *              ただし、型チェックは完璧ではなく、実際の型が文字列でなくても ResultSuccess を返す場合があるので注意してください。
 *
 */
nn::Result GetPropertyString(size_t* pOutSize, char* pOutBuffer, size_t bufferSize, const Node* pNode, const char* name) NN_NOEXCEPT;

//! @}

}}
