﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   安全なダウンキャストのための型情報を扱う機能に関する宣言
 */

#pragma once

#include <nn/nn_Common.h>

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

#define NN_DETAIL_DDSF_CAST_SAFE_DEFINE_ROOT(className) \
    const ::nn::ddsf::detail::TypeTag className::NnDetailDdTypeTagInstance_(#className);

#else

// TypeTag コンストラクタ側でも className を渡されても Release ビルドでは使用しないようになっているが、
// マクロ側もこの実装分岐をしておくことでクラス名文字列が生成されないことを保証
#define NN_DETAIL_DDSF_CAST_SAFE_DEFINE_ROOT(className) \
    const ::nn::ddsf::detail::TypeTag className::NnDetailDdTypeTagInstance_;

#endif

namespace nn { namespace ddsf { namespace detail {

class TypeTag
{
public:

#if !(defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)) // クラス名を渡さないコンストラクタは Release ビルドでのみ有効
    TypeTag() NN_NOEXCEPT :
        m_ClassName(nullptr),
        m_pBase(nullptr)
    {
        NN_UNUSED(m_ClassName); // Release ビルド対策
    }
    explicit TypeTag(const TypeTag& base) NN_NOEXCEPT :
        m_ClassName(nullptr),
        m_pBase(&base)
    {
    }
#endif

    explicit TypeTag(const char* className) NN_NOEXCEPT :
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
        m_ClassName(className),
#else
        m_ClassName(nullptr),
#endif
        m_pBase(nullptr)
    {
        NN_UNUSED(className); // Release ビルド対策
    }
    TypeTag(const char* className, const TypeTag& base) NN_NOEXCEPT :
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
        m_ClassName(className),
#else
        m_ClassName(nullptr),
#endif
        m_pBase(&base)
    {
        NN_UNUSED(className); // Release ビルド対策
        NN_SDK_ASSERT(this != &base, "Cannot set itself as base object.\n");
    }

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    const char* GetClassName() const NN_NOEXCEPT
    {
        return m_ClassName;
    }
#endif
    bool IsExactly(const TypeTag& rhs) const NN_NOEXCEPT
    {
        return (this == &rhs);
    }
    bool DerivesFrom(const TypeTag& rhs) const NN_NOEXCEPT
    {
        const TypeTag* pCompare = this;
        while ( pCompare )
        {
            if ( pCompare == &rhs )
            {
                return true;
            }
            pCompare = pCompare->m_pBase;
        }
        return false;
    }
private:
    const char* m_ClassName; // Release ビルドでは使わないが、翻訳単位間でビルドタイプが異なる場合を考慮し保持
    const TypeTag* m_pBase{ nullptr };
};

}}} // namespace nn::ddsf::detail
