﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief デバイスコードエントリに関する宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_DeviceCode.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/util/util_TypedStorage.h>

namespace nn { namespace ddsf {

class IDevice;

//! @name デバイスコードエントリ関連 API
//! @{

/**
* @brief        デバイスコードエントリです。
*
* @detail
*   デバイスコードエントリを表すオブジェクトです。
*/
class DeviceCodeEntry
{
    NN_DISALLOW_COPY(DeviceCodeEntry);
    NN_DISALLOW_MOVE(DeviceCodeEntry);

public:
    /**
    * @brief    パラメータの設定を同時に行うコンストラクタです。
    *
    * @param[in]    deviceCode  デバイスコード
    * @param[in]    pDevice     デバイスオブジェクト
    *
    * @pre
    *   - pDevice != nullptr
    *
    * @detail
    *   パラメータの設定を同時に行うコンストラクタです。
    */
    DeviceCodeEntry(nn::DeviceCode deviceCode, IDevice* pDevice) NN_NOEXCEPT :
        m_DeviceCode(deviceCode),
        m_pDevice(pDevice)
    {
        NN_SDK_REQUIRES_NOT_NULL(pDevice);
    }

    /**
    * @brief    デバイスコードを取得します。
    *
    * @return   デバイスコード
    *
    * @detail
    *   デバイスコードを取得します。
    */
    nn::DeviceCode GetDeviceCode() const NN_NOEXCEPT
    {
        return m_DeviceCode;
    }

    /**
    * @brief    関連付けられているデバイスの参照を取得します。
    *
    * @return   デバイスオブジェクトの参照
    *
    * @detail
    *   関連付けられているデバイスの参照を取得します。
    */
    IDevice& GetDevice() NN_NOEXCEPT
    {
        return *m_pDevice;
    }

    /**
    * @brief    関連付けられているデバイスへの const 参照を取得します。
    *
    * @return   デバイスオブジェクトの const 参照
    *
    * @detail
    *   関連付けられているデバイスの const 参照を取得します。 @n
    *   const メンバ関数内から呼び出すことができます。
    */
    const IDevice& GetDevice() const NN_NOEXCEPT
    {
        return *m_pDevice;
    }

private:
    nn::DeviceCode m_DeviceCode{ nn::DeviceCode::GetInvalidCode() };
    IDevice* m_pDevice{ nullptr };
};

/**
* @brief        デバイスコードエントリの構築と管理のためのホルダオブジェクトです。
*
* @detail
*   デバイスコードエントリの構築と管理のためのホルダオブジェクトです。 @n
*   DeviceCodeEntryManager などのリソース管理クラスで使用することを想定しています。
*/
class DeviceCodeEntryHolder
{
    NN_DISALLOW_COPY(DeviceCodeEntryHolder);
    NN_DISALLOW_MOVE(DeviceCodeEntryHolder);

private:
    nn::util::IntrusiveListNode m_ListNode;
    friend class nn::util::IntrusiveListMemberNodeTraits<DeviceCodeEntryHolder, &DeviceCodeEntryHolder::m_ListNode>;

public:
    typedef nn::util::IntrusiveList<
        DeviceCodeEntryHolder, nn::util::IntrusiveListMemberNodeTraits<DeviceCodeEntryHolder, &DeviceCodeEntryHolder::m_ListNode>
    > List;

public:
    /**
    * @brief    デフォルトコンストラクタです。
    */
    DeviceCodeEntryHolder() NN_NOEXCEPT
    {
    }

    /**
    * @brief    デストラクタです。
    */
    ~DeviceCodeEntryHolder() NN_NOEXCEPT
    {
        if ( IsBuilt() )
        {
            Destroy();
        }
    }

    /**
    * @brief    本オブジェクトを渡されたリストに追加します。
    *
    * @pre
    *   - pList != nullptr
    *
    * @param[in,out]    pList   操作対象のリスト
    */
    void AddTo(List* pList) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pList);
        pList->push_back(*this);
    }

    /**
    * @brief    本オブジェクトを渡されたリストから削除します。
    *
    * @pre
    *   - pList != nullptr
    *
    * @param[in,out]    pList   操作対象のリスト
    */
    void RemoveFrom(List* pList) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pList);
        pList->erase(pList->iterator_to(*this));
    }

    /**
    * @brief    本オブジェクトがリストに追加された状態かどうかを返します。
    *
    * @return   リストに追加された状態かどうか
    */
    bool IsLinkedToList() const NN_NOEXCEPT
    {
        return m_ListNode.IsLinked();
    }

    /**
    * @brief    デバイスコードエントリを構築します。
    *
    * @param[in]    deviceCode  デバイスコード
    * @param[in]    pDevice     デバイスオブジェクト
    *
    * @return   構築済のデバイスコードエントリ
    *
    * @pre
    *   - IsBuilt() == false
    *   - pDevice != nullptr
    *
    * @detail
    *   デバイスコードエントリのパラメータを設定します。
    */
    DeviceCodeEntry& Build(nn::DeviceCode deviceCode, IDevice* pDevice) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsBuilt());
        auto pBuiltEntry = new (&m_Storage) DeviceCodeEntry(deviceCode, pDevice);
        m_IsBuilt = true;
        return *pBuiltEntry;
    }

    /**
    * @brief    デバイスコードエントリの構築状態を返します。
    *
    * @return   構築済かどうか
    */
    bool IsBuilt() const NN_NOEXCEPT
    {
        return m_IsBuilt;
    }

    /**
    * @brief    デバイスコードエントリを破棄します。
    *
    * @pre
    *   - IsBuilt() == true
    */
    void Destroy() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBuilt());
        nn::util::Get(m_Storage).~DeviceCodeEntry();
        m_IsBuilt = false;
    }

    /**
    * @brief    構築済のデバイスコードエントリを取得します。
    *
    * @return   構築済のデバイスコードエントリ
    *
    * @pre
    *   - IsBuilt() == true
    *
    * @detail
    *   構築済のデバイスコードエントリを取得します。事前に Build() により構築済である必要があります。
    */
    DeviceCodeEntry& Get() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBuilt());
        return nn::util::Get(m_Storage);
    }

    /**
    * @brief    構築済のデバイスコードエントリを取得します。 (const)
    *
    * @return   構築済のデバイスコードエントリ
    *
    * @pre
    *   - IsBuilt() == true
    *
    * @detail
    *   構築済のデバイスコードエントリを取得します。事前に Build() により構築済である必要があります。 @n
    *   const メンバ関数内から呼び出すことができます。
    */
    const DeviceCodeEntry& Get() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBuilt());
        return nn::util::Get(m_Storage);
    }

private:
    nn::util::TypedStorage<DeviceCodeEntry, sizeof(DeviceCodeEntry), NN_ALIGNOF(DeviceCodeEntry)> m_Storage;
    bool m_IsBuilt{ false };
};

//! @}

}} // nn::ddsf
