﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   デバイスドライバ向けデバイスアドレス空間に関する公開ヘッダファイル
 */

#pragma once

#include <nn/TargetConfigs/build_Os.h>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/dd/dd_Types.h>
#include <nn/dd/dd_DeviceAddressSpaceCommon.h>
#include <nn/dd/dd_DeviceAddressSpaceTypes.h>
#include <nn/os/os_NativeHandleTypes.h>

namespace nn { namespace dd {

//! @name   デバイスアドレス空間関連 API
//! @{

//-----------------------------------------------------------------------------
/**
 * @brief   デバイスアドレス空間オブジェクトを作成します。
 *
 * @param[in] deviceSpace   デバイスアドレス空間オブジェクトへのポインタ
 * @param[in] address       デバイスアドレス空間の開始アドレス
 * @param[in] size          デバイスアドレス空間のサイズ
 *
 * @retresult
 *   @handleresult{nn::dd::ResultOutOfMemory}
 *   @handleresult{nn::dd::ResultOutOfResource}
 *   @handleresult{nn::dd::ResultNotSupported}
 * @endretresult
 *
 * @pre
 *  - deviceSpace が未初期化状態もしくは初期化されていない状態である
 *  - address が 4KB アライメントされている
 *  - size > 0 かつ、size が 4KB アライメントされている
 *
 * @post
 *  - deviceSpace が初期化状態である
 *
 * @details
 *  デバイス用のデバイスアドレス空間ハンドルを新規に作成し、
 *  デバイスアドレス空間オブジェクトを初期化します。
 *  デバイスアドレス空間は address で指定した場所から開始し、size で指定したサイズのアドレス空間となります。
 *
 *  なお、DeviceAddressSpaceType オブジェクトは、オブジェクトの状態とは別に、
 *  bool 型のハンドル管理フラグを保持しています。ハンドル管理フラグは、
 *  DestroyDeviceAddressSpace() 時にハンドルを自動的にシステムに返却するか否か
 *  を示しており、本 API では必ず true に設定されます。
 *
 *  初期化した deviceSpace が指すオブジェクトは、以下のいずれかの処理を
 *  順不同で行なうことが出来ます。
 *
 *  1) AttachDeviceAddressSpace() により、
 *     deviceSpace が指すデバイスアドレス空間オブジェクトを、
 *     実際のデバイスグループに関連付けます。
 *
 *  2) MapDeviceAddressSpaceAligned() および
 *     MapNextDeviceAddressSpaceRegion() により、
 *     指定されたプロセス空間上のメモリ領域を、
 *     deviceSpace が指すデバイスアドレス空間上にマップします。
 *
 *  上記 1) と 2) の両方の操作を行なうと、deviceSpace に関連付けられた
 *  メモリ領域に対して、特定のデバイスからアクセスできるようになります。
 *
 *  このメモリアクセスは、各デバイス向けに用意されたアドレス変換テーブルを
 *  介してアクセスされるため、CPU から見えるアドレス空間とは異なっていますので
 *  ご注意ください。
 *
 *  デバイスアドレス空間の開始アドレスを address で指定することが出来ますが、
 *  指定できる値はプラットフォームによって異なります。
 *  詳細については SoC のマニュアルを参照して下さい。
 *
 *  デバイスアドレス空間のサイズを size で指定することが出来ますが、
 *  これは各デバイス毎に定義された規定のアドレス空間サイズに合わせて設定する
 *  という目的に加えて、1 つのデバイスアドレス空間オブジェクトを複数のデバイス
 *  にアタッチして利用するときにも重要なパラメータとなります。
 *
 *  具体的には、各デバイスによって利用可能なデバイスアドレス空間の最大サイズが
 *  異なっている場合に、size を空間サイズの最も小さいデバイスに合わせて
 *  初期化しておきます。これにより、これら空間サイズの異なる複数のデバイスに
 *  対しても同一のデバイスアドレス空間オブジェクトをマップして利用することが
 *  可能になります。各デバイスのアドレス空間サイズについては LSI 仕様書なども
 *  合わせて確認して下さい。
 *
 *  なお、対象ターゲットがデバイスアドレス空間機能をサポートしていない場合、
 *  nn::dd::ResultNotSupported を返し、deviceSpace は未初期化状態となります。
 *
 */
Result CreateDeviceAddressSpace(DeviceAddressSpaceType* deviceSpace, uint64_t address, uint64_t size) NN_NOEXCEPT;

//-----------------------------------------------------------------------------
/**
 * @brief   この API は従来との互換性のために存在しています。
 *
 * @param[in] deviceSpace   デバイスアドレス空間オブジェクトへのポインタ
 * @param[in] size          デバイスアドレス空間のサイズ
 *
 * @retresult
 *   @handleresult{nn::dd::ResultOutOfMemory}
 *   @handleresult{nn::dd::ResultOutOfResource}
 *   @handleresult{nn::dd::ResultNotSupported}
 * @endretresult
 *
 * @pre
 *  - deviceSpace が未初期化状態もしくは初期化されていない状態である
 *  - size > 0 かつ、size が 4KB アライメントされている
 *
 * @post
 *  - deviceSpace が初期化状態である
 *
 * @details
 *  本 API は従来との互換性維持のために存在しています。
 *  本 API は CreateDeviceAddressSpace(DeviceAddressSpaceType*, uint64_t, uint64_t) の第 2 引数のアドレスに 0 を指定したものと同じです。
 *
 */
Result CreateDeviceAddressSpace(DeviceAddressSpaceType* deviceSpace, uint64_t size) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 * @brief   既存のデバイスアドレス空間ハンドルを DeviceAddressSpaceType オブジェクトに関連付けします。
 *
 * @param[in] deviceSpace   デバイスアドレス空間オブジェクトへのポインタ
 * @param[in] handle        デバイスアドレス空間のハンドル
 * @param[in] managed       デバイスアドレス空間のハンドル管理フラグ
 *
 * @pre
 *  - deviceSpace が未初期化状態もしくは初期化されていない状態である
 *  - handle が有効なハンドル値である
 *
 * @post
 *  - deviceSpace が初期化状態である
 *
 * @details
 *  指定されたデバイスアドレス空間ハンドルを使って、
 *  デバイスアドレス空間オブジェクトを初期化します。
 *
 *  本 API で初期化したデバイスアドレス空間オブジェクトは、
 *  DestroyDeviceAddressSpace() にて破棄することが出来ます。
 *
 *  また、 DestroyDeviceAddressSpace() 時に自動的にハンドルをシステムに
 *  返却するかどうかを managed で指定しておきます。@n
 *  managed に true を指定すると DestroyDeviceAddressSpace() 時にハンドルを
 *  システムに返却し、false を指定すると DestroyDeviceAddressSpace() 時に
 *  ハンドルを返却しません。後者の場合、このハンドルは引き続き有効なままです。
 *
 *  本 API の動作中は、対象デバイスアドレス空間オブジェクトに対する操作を
 *  行なわないで下さい。
 *
 */
void AttachDeviceAddressSpaceHandle(DeviceAddressSpaceType* deviceSpace, nn::os::NativeHandle handle, bool managed) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   デバイスアドレス空間オブジェクトを破棄します。
 *
 * @param[in] deviceSpace     デバイスアドレス空間オブジェクトへのポインタ
 *
 * @pre
 *  - deviceSpace が初期化状態である
 *
 * @post
 *  - deviceSpace が未初期化状態である
 *
 * @details
 *  デバイスアドレス空間オブジェクトを破棄します。
 *
 *  同時に、CreateDeviceAddressSpace() 時および
 *  AttachDeviceAddressSpaceHandle() 時に設定されたハンドル管理フラグが
 *  true ならば、デバイスアドレス空間ハンドルをシステムに返却します。
 *  ハンドル管理フラグが false の場合は、ハンドルは引き続き有効なままです。
 *
 *  本 API の動作中は、対象 DeviceAddressSpaceType オブジェクトに対する操作を
 *  行なわないで下さい。
 *
 */
void DestroyDeviceAddressSpace(DeviceAddressSpaceType* deviceSpace) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   デバイスアドレス空間オブジェクトの ハンドル を取得します。
 *
 * @param[in] deviceSpace     デバイスアドレス空間オブジェクトへのポインタ
 *
 * @return  デバイスアドレス空間のハンドルを返します。
 *
 * @pre
 *  - deviceSpace が初期化状態である
 *
 * @details
 *  デバイスアドレス空間オブジェクトが管理している デバイスアドレス空間ハンドル
 *  を返します。ハンドル管理フラグは変更されません。
 *
 */
nn::os::NativeHandle GetDeviceAddressSpaceHandle(DeviceAddressSpaceType* deviceSpace) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   デバイスアドレス空間にメモリ領域をマップします。
 *
 * @param[in] deviceSpace       デバイスアドレス空間オブジェクトへのポインタ
 * @param[in] processHandle     マップ元のプロセスを指すハンドル
 * @param[in] processAddress    マップ元のプロセスのメモリアドレス
 * @param[in] size              マップするメモリサイズ
 * @param[in] deviceAddress     マップ先のデバイスアドレス空間上のアドレス
 * @param[in] devicePermission  マップ先のデバイスアドレス空間上のアクセス権限
 *
 * @retresult
 *   @handleresult{nn::dd::ResultOutOfMemory}
 *   @handleresult{nn::dd::ResultOutOfResource}
 *   @handleresult{nn::dd::ResultInvalidHandle}
 *   @handleresult{nn::dd::ResultInvalidMemoryState}
 * @endretresult
 *
 * @pre
 *  - deviceSpace が初期化状態である
 *  - processAddress が 4KB アライメントされている
 *  - deviceAddress  が 4KB アライメントされている
 *  - (processAddress & 0x3fffff) == (deviceAddress & 0x3fffff) である
 *  - size > 0 かつ、size が 4KB アライメントされている
 *  - processHandle, processAddress, size が指すメモリ領域が以下の条件を満たす
 *    - nn::os::MemoryPermission_ReadOnly もしくは nn::os::MemoryPermission_ReadWrite である
 *    - devicePermission 以上のアクセス権限である
 *  - devicePermission が以下のいずれかである
 *    - nn::dd::MemoryPermission_ReadOnly
 *    - nn::dd::MemoryPermission_WriteOnly
 *    - nn::dd::MemoryPermission_ReadWrite
 *
 * @details
 *  processHandle が指すプロセス上の processAddress から size 分のメモリ領域を、
 *  deviceSpace が指すデバイスアドレス空間上の deviceAddress が指すアドレスに
 *  マップします。
 *
 *  processAddress および size は @ref DeviceAddressSpaceMemoryRegionAlignment
 *  が示す定数値でアライメントされている必要があります。この定数値は 4096 です。
 *  そのようなバッファ領域を配列などで定義する場合には
 *  @ref NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY マクロをご利用下さい。
 *
 *  また、processAddress と deviceAddress の下位 22 bit は一致していなければ
 *  なりません。一致していない場合、事前条件違反でアボートします。
 *
 *  deviceSpace が指すデバイスアドレス空間オブジェクトが、
 *  AttachDeviceAddressSpace() によって既に特定のデバイスグループに関連付け
 *  られている場合、即座に対象のデバイスアドレス空間上に当該メモリ領域が
 *  マップされます。
 *
 *  マップ元のメモリ領域のアクセスパーミションは変更されず、
 *  マップ先のメモリ領域は devicePermission でアクセスできるようになります。
 *
 *  nn::os::MapAliasMemory() によってマップされたメモリ領域をマップ
 *  しようとした場合には、nn::dd::ResultInvalidMemoryState が返ります。
 *
 *  本 API でマップしたメモリをアンマップする場合には、
 *  UnmapDeviceAddressSpace() をご使用下さい。
 *
 */
Result MapDeviceAddressSpaceAligned(DeviceAddressSpaceType* deviceSpace, ProcessHandle processHandle, uint64_t processAddress, size_t size, DeviceVirtualAddress deviceAddress, MemoryPermission devicePermission) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   デバイスアドレス空間にメモリ領域をマップします。
 *
 * @param[in] deviceSpace       デバイスアドレス空間オブジェクトへのポインタ
 * @param[in] processHandle     マップ元のプロセスを指すハンドル
 * @param[in] processAddress    マップ元のプロセスのメモリアドレス
 * @param[in] size              マップするメモリサイズ
 * @param[in] deviceAddress     マップ先のデバイスアドレス空間上のアドレス
 * @param[in] devicePermission  マップ先のデバイスアドレス空間上のアクセス権限
 *
 * @retresult
 *   @handleresult{nn::dd::ResultOutOfMemory}
 *   @handleresult{nn::dd::ResultOutOfResource}
 *   @handleresult{nn::dd::ResultInvalidHandle}
 *   @handleresult{nn::dd::ResultInvalidMemoryState}
 * @endretresult
 *
 * @pre
 *  - deviceSpace が初期化状態である
 *  - processAddress が 4KB アライメントされている
 *  - deviceAddress  が 4KB アライメントされている
 *  - size > 0 かつ、size が 4KB アライメントされている
 *  - processHandle, processAddress, size が指すメモリ領域が以下の条件を満たす
 *    - nn::os::MemoryPermission_ReadOnly もしくは nn::os::MemoryPermission_ReadWrite である
 *    - devicePermission 以上のアクセス権限である
 *  - devicePermission が以下のいずれかである
 *    - nn::dd::MemoryPermission_ReadOnly
 *    - nn::dd::MemoryPermission_WriteOnly
 *    - nn::dd::MemoryPermission_ReadWrite
 *
 * @details
 *  processHandle が指すプロセス上の processAddress から size 分のメモリ領域を、
 *  deviceSpace が指すデバイスアドレス空間上の deviceAddress が指すアドレスに
 *  マップします。
 *
 *  本機能は MapDeviceAddressSpaceAligned() と機能的には同じであるため、
 *  詳細はそちらのリファレンスを参照して下さい。唯一の違いは、
 *  processAddress と deviceAddress の下位 22 bit が一致していなくても
 *  マップが可能であるという点です。
 *
 *  ただし、本関数は特別に許可されたシステムプロセスのみが使用できます。
 *  許可されていないプロセスで本関数を呼び出した場合は、
 *  関数の内部でアボートします。
 *
 *  本 API でマップしたメモリをアンマップする場合には、
 *  UnmapDeviceAddressSpace() をご使用下さい。
 *
 */
Result MapDeviceAddressSpaceNotAligned(DeviceAddressSpaceType* deviceSpace, ProcessHandle processHandle, uint64_t processAddress, size_t size, DeviceVirtualAddress deviceAddress, MemoryPermission devicePermission) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   デバイスアドレス空間にマップしていたメモリ領域をアンマップします。
 *
 * @param[in] deviceSpace       デバイスアドレス空間オブジェクトへのポインタ
 * @param[in] processHandle     マップ元のプロセスを指すハンドル
 * @param[in] processAddress    マップ元のプロセスのメモリアドレス
 * @param[in] size              アンマップするメモリサイズ
 * @param[in] deviceAddress     マップ先のデバイスアドレス空間上のアドレス
 *
 * @pre
 *  - deviceSpace が初期化状態である
 *  - processAddress が 4KB アライメントされている
 *  - deviceAddress  が 4KB アライメントされている
 *  - size > 0 かつ、size が 4KB アライメントされている
 *
 * @details
 *  processHandle が指すプロセス上の processAddress から size 分のメモリ領域を、
 *  deviceSpace が指すデバイスアドレス空間上からアンマップします。
 *
 *  deviceSpace が指すデバイスアドレス空間オブジェクトが、
 *  AttachDeviceAddressSpace() によって既に特定のデバイスグループに関連付け
 *  られている場合、対象のデバイスアドレス空間上の当該メモリ領域への
 *  マップも即座にアンマップされます。
 *
 */
void UnmapDeviceAddressSpace(DeviceAddressSpaceType* deviceSpace, ProcessHandle processHandle, uint64_t processAddress, size_t size, DeviceVirtualAddress deviceAddress) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   デバイスアドレス空間にメモリ領域をインクリメンタルにマップするための準備をします。
 *
 * @param[out] info             DeviceAddressSpaceMapInfo オブジェクトへのポインタ
 * @param[in]  deviceSpace      デバイスアドレス空間オブジェクトへのポインタ
 * @param[in]  processHandle    マップ元のプロセスを指すハンドル
 * @param[in]  processAddress   マップ元のプロセスのメモリアドレス
 * @param[in]  size             マップするメモリサイズ
 * @param[in]  deviceAddress    マップ先のデバイスアドレス空間上のアドレス
 * @param[in]  devicePermission マップ先のデバイスアドレス空間上のアクセス権限
 *
 * @pre
 *  - info が初期化されていない状態である
 *  - deviceSpace が初期化状態である
 *  - processAddress が 4KB アライメントされている
 *  - deviceAddress  が 4KB アライメントされている
 *  - size > 0 かつ、size が 4KB アライメントされている
 *  - processHandle, processAddress, size が指すメモリ領域が以下の条件を満たす
 *    - nn::os::MemoryPermission_ReadOnly もしくは nn::os::MemoryPermission_ReadWrite である
 *    - devicePermission 以上のアクセス権限である
 *  - devicePermission が以下のいずれかである
 *    - nn::dd::MemoryPermission_ReadOnly
 *    - nn::dd::MemoryPermission_WriteOnly
 *    - nn::dd::MemoryPermission_ReadWrite
 *
 * @details
 *  processHandle が指すプロセス上の processAddress から size 分のメモリ領域を、
 *  deviceSpace が指すデバイスアドレス空間上にマップを行なうための準備を行ない、
 *  info オブジェクトにその情報を設定します。
 *
 *  本 API は nn::os::MapAliasMemory() によってエイリアスされたメモリ領域を
 *  デバイスアドレス空間上にマップできる唯一の機能になります。
 *  さらに MapDeviceAddressSpaceAligned() とは異なり、
 *  システムのメモリリソース消費量を抑制するために、指定された size よりも
 *  小さなサイズで部分的なマップとアンマップをインクリメンタルに繰り返すような
 *  使い方を前提としています。
 *
 *  processAddress および size は @ref DeviceAddressSpaceMemoryRegionAlignment
 *  が示す定数値でアライメントされている必要があります。
 *  そのようなバッファ領域を配列などで定義する場合には
 *  @ref NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY マクロをご利用下さい。
 *
 *  マップ先の先頭デバイスアドレスは deviceAddress であり、
 *  マップしたデバイス空間上のメモリアクセス権限は devicePermission になります。
 *
 *  具体的な使用方法は以下のサンプルコードを参照して下さい。
 *  全ての操作が終わった後は、info オブジェクトは自由に破棄して構いません。
 *
 * @code
 *  Result IncrementalProcessing(DeviceAddressSpaceType* pDas, ...)
 *  {
 *      nn::dd::DeviceAddressSpaceMapInfo info;
 *      nn::dd::InitializeDeviceAddressSpaceMapInfo(&info, pDas, processHandle, processAddress, size, deviceAddress, nn::dd::MemoryPermission_ReadWrite);
 *
 *      for (;;)
 *      {
 *          size_t mappedSize;
 *          result = nn::dd::MapNextDeviceAddressSpaceRegion(&mappedSize, &info);
 *          if (!result.IsSuccess())
 *          {
 *              // エラーを検出
 *              return result;
 *          }
 *          if (mappedSize == 0)
 *          {
 *              // 全ての領域のマップ＆アンマップ処理を完了
 *              break;
 *          }
 *
 *          auto processAddress = nn::dd::GetMappedProcessAddress(&info);
 *          auto deviceAddress  = nn::dd::GetMappedDeviceVirtualAddress(&info);
 *          // ここで各デバイスから、デバイスアドレス空間へのアクセスを行なう
 *
 *          nn::dd::UnmapDeviceAddressSpaceRegion(&info);
 *      }
 *      return nn::ResultSuccess();
 *  }
 * @endcode
 *
 */
void InitializeDeviceAddressSpaceMapInfo(DeviceAddressSpaceMapInfo* info, DeviceAddressSpaceType* deviceSpace, ProcessHandle processHandle, uint64_t processAddress, size_t size, DeviceVirtualAddress deviceAddress, MemoryPermission devicePermission) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   デバイスアドレス空間にメモリ領域をインクリメンタルにマップします。
 *
 * @param[out] outMappedSize マップに成功したサイズの格納先アドレス
 * @param[in]  info          DeviceAddressSpaceMapInfo オブジェクトへのポインタ
 *
 * @retresult
 *   @handleresult{nn::dd::ResultOutOfMemory}
 *   @handleresult{nn::dd::ResultOutOfResource}
 *   @handleresult{nn::dd::ResultInvalidHandle}
 *   @handleresult{nn::dd::ResultInvalidMemoryState}
 * @endretresult
 *
 * @pre
 *  - info は InitializeDeviceAddressSpaceMapInfo() で初期化済みである
 *
 * @details
 *  info に設定された情報に基づいて、対象プロセスの指定されたメモリ領域を
 *  対象デバイスアドレス空間上にマップします。
 *  本 API は MapDeviceAddressSpaceAligned() とは異なり、
 *  マップするサイズは、システムのメモリ消費量が小さくなるようなサイズが
 *  自動的に決定され、一度に全てのメモリ領域がマップされるとは限りません。
 *
 *  一回の API 呼出しでマップに成功したサイズは outMappedSize に格納されます。
 *  InitializeDeviceAddressSpaceMapInfo() のリファレンスに記載の
 *  サンプルコードのように、インクリメンタルにマップとアンマップを
 *  繰り返すようにして使用します。
 *
 *  呼出元は本関数の返値である Result を見てマップの成否を判定します。
 *  また、ResultSuccess() が返された場合は、さらに outMappedSize に格納された
 *  サイズを見て、以下のように繰り返しループの終了を判定しなければなりません。
 *
 *  outMappedSize > 0 の場合には、以下の API を使ってマップに成功した
 *  プロセスアドレス、デバイスアドレス、サイズの情報を取得することができます。
 *  また、ここでマップしたメモリ領域は、この時の info オブジェクトを使って
 *  UnmapDeviceAddressSpaceRegion() を呼び出すことでアンマップして下さい。
 *
 *  - nn::dd::GetMappedProcessAddress()
 *  - nn::dd::GetMappedDeviceVirtualAddress()
 *  - nn::dd::GetMappedSize()
 *
 *  一方、outMappedSize == 0 の場合は最初に size で指定した全メモリ領域分の
 *  繰り返しマップ処理が完了したことを表します。この場合、追加のマップ処理は
 *  行なわれていないため、この時の info オブジェクトを使って
 *  UnmapDeviceAddressSpaceRegion() を呼ぶ必要はありません。
 *
 *  deviceSpace が指すデバイスアドレス空間オブジェクトが、
 *  AttachDeviceAddressSpace() によって既に特定のデバイスグループに関連付け
 *  られている場合、本関数によって即座に対象のデバイスアドレス空間上に
 *  当該メモリ領域がされます。
 *
 */
Result MapNextDeviceAddressSpaceRegion(size_t* outMappedSize, DeviceAddressSpaceMapInfo* info) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   デバイスアドレス空間にマップしていたメモリ領域をインクリメンタルにアンマップします。
 *
 * @param[in] info      DeviceAddressSpaceMapInfo オブジェクトへのポインタ
 *
 * @pre
 *  - info は InitializeDeviceAddressSpaceMapInfo() で初期化済みである
 *  - info に MapNextDeviceAddressSpaceRegion() によってマップした情報が保持されている
 *
 * @details
 *  MapNextDeviceAddressSpaceRegion() 呼び出し時に info の中に記録された、
 *  最後にマップした部分メモリ領域をデバイスアドレス空間からアンマップします。
 *
 *  具体的な使用方法は InitializeDeviceAddressSpaceMapInfo() のリファレンスに
 *  記載のサンプルコードを参照して下さい。
 *
 */
void UnmapDeviceAddressSpaceRegion(DeviceAddressSpaceMapInfo* info) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   DeviceAddressSpaceMapInfo に格納されているプロセスアドレス情報を返します。
 *
 * @param[in] info      DeviceAddressSpaceMapInfo オブジェクトへのポインタ
 *
 * @pre
 *  - info は InitializeDeviceAddressSpaceMapInfo() で初期化済みである
 *  - info に MapNextDeviceAddressSpaceRegion() によってマップした情報が保持されている
 *
 * @details
 *  MapNextDeviceAddressSpaceRegion() 呼び出し時に info の中に記録された、
 *  最後にマップした部分メモリ領域のプロセスアドレス値を返します。
 *
 */
uint64_t GetMappedProcessAddress(DeviceAddressSpaceMapInfo* info) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   DeviceAddressSpaceMapInfo に格納されているデバイスアドレス情報を返します。
 *
 * @param[in] info      DeviceAddressSpaceMapInfo オブジェクトへのポインタ
 *
 * @pre
 *  - info は InitializeDeviceAddressSpaceMapInfo() で初期化済みである
 *  - info に MapNextDeviceAddressSpaceRegion() によってマップした情報が保持されている
 *
 * @details
 *  MapNextDeviceAddressSpaceRegion() 呼び出し時に info の中に記録された、
 *  最後にマップした部分メモリ領域のデバイスアドレス値を返します。
 *
 */
DeviceVirtualAddress GetMappedDeviceVirtualAddress(DeviceAddressSpaceMapInfo* info) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   DeviceAddressSpaceMapInfo に格納されているサイズ情報を返します。
 *
 * @param[in] info      DeviceAddressSpaceMapInfo オブジェクトへのポインタ
 *
 * @pre
 *  - info は InitializeDeviceAddressSpaceMapInfo() で初期化済みである
 *
 * @details
 *  MapNextDeviceAddressSpaceRegion() 呼び出し時に info の中に記録された、
 *  最後にマップした部分メモリ領域のサイズ値を返します。
 *  返値が 0 の場合、InitializeDeviceAddressSpaceMapInfo() 時に設定した
 *  全領域サイズ分のマップ処理を完了していることを表します。
 *
 */
size_t GetMappedSize(DeviceAddressSpaceMapInfo* info) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   デバイスアドレス空間オブジェクトをデバイスグループに関連付けます。
 *
 * @param[in] deviceSpace       デバイスアドレス空間オブジェクトへのポインタ
 * @param[in] deviceName        デバイスグループ名
 *
 * @retresult
 *   @handleresult{nn::dd::ResultOutOfMemory}
 * @endretresult
 *
 * @pre
 *  - deviceSpace が初期化状態である
 *
 * @post
 *  - deviceSpace が deviceName のデバイスグループに関連付けられる
 *
 * @details
 *  deviceSpace で示されるデバイスアドレス空間オブジェクトを、
 *  deviceName で表されるデバイスグループに関連付けます。
 *
 *  deviceSpace が指すデバイスアドレス空間オブジェクトに既にメモリがマップ
 *  されている場合は、deviceName で示されるデバイスのアドレス空間上に、
 *  当該メモリ領域が即座にマップされます。
 *
 */
Result AttachDeviceAddressSpace(DeviceAddressSpaceType* deviceSpace, DeviceName deviceName) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   デバイスアドレス空間オブジェクトへのデバイスグループの関連付けを解除します。
 *
 * @param[in] deviceSpace       デバイスアドレス空間オブジェクトへのポインタ
 * @param[in] deviceName        デバイスグループ名
 *
 * @pre
 *  - deviceSpace が初期化状態である
 *
 * @post
 *  - deviceSpace が deviceName のデバイスグループから切り離される
 *
 * @details
 *  deviceSpace で示されるデバイスアドレス空間オブジェクトを、
 *  deviceName で表されるデバイスグループから切り離します。
 *
 *  deviceSpace が指すデバイスアドレス空間オブジェクトに既にメモリがマップ
 *  されている場合は、deviceName で示されるデバイスのアドレス空間上からも、
 *  当該メモリ領域へのマップが解除されます。
 *
 */
void DetachDeviceAddressSpace(DeviceAddressSpaceType* deviceSpace, DeviceName deviceName) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   自プロセスを示す ProcessHandle を取得します。
 *
 * @return  自プロセスを示すハンドルを返します。
 *
 * @details
 *  自プロセスを示すハンドルを ProcessHandle 型で返します。
 *
 *  MapDeviceAddressSpaceAligned() および MapNextDeviceAddressSpaceRegion()
 *  において、他プロセスのメモリをデバイスアドレス空間にマップする際には、
 *  対象プロセス上で ProcessHandle を取得し、そのハンドル値を実際に
 *  マップ操作を行なうプロセスへ渡して下さい。
 *
 */
ProcessHandle GetCurrentProcessHandle() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   指定されたプロセスハンドルをクローズします。
 *
 * @param[in] handle    クローズするプロセスハンドル
 *
 * @details
 *  指定されたプロセスハンドルをクローズします。
 *
 *  この機能は、 GetCurrentProcessHandle() で取得したプロセスハンドルを
 *  プロセス間通信で他のプロセスに渡した場合に、受け取った側のプロセスが
 *  使用済みになったプロセスハンドルをクローズするために使用します。
 *
 */
void CloseProcessHandle(ProcessHandle handle) NN_NOEXCEPT;


//-----------------------------------------------------------------------------

//! @}

}}  // namespace nn::dd

