﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief BSD Configuration Client API
 *
 * @details
 */
namespace nn { namespace bsdsocket { namespace cfg {


/**
 * @brief Administatively enables the specified interface using
 *        supplied options.
 * @param[in] ifName
 *   Network interface name
 * @param[in] pIfOptionData
 *   Option data
 * @pre
 *   User has not already successfully called SetIfUp() for this ifName,
 *   OR user did and subsequently called SetIfDown().
 * @post
 * @details
 *   This call will block until the specified interface is
 *   successfully enabled, or until an unrecoverable error occurs.
 *   To cancel a call which is in progress, user must call
 *   CancelIf() from another thread.
 * @return Status of the attempted operation is returned.
 * @retval ResultSuccess
 *   Operation completed successfully.
 * @retval ResultIfInvalid
 *   Specified ifName does not exist.
 * @retval ResultMtuSizeError
 *   Core TCP/IP stack did not accept specified MTU value.
 * @retval ResultIfRemoved
 *   The underlying network interface device has detached.
 * @retval ResultRouteError
 *   Unable to configure route using specified IP parameters.
 *   There could be a problem with the specified address, subnet
 *   mask or gateway.
 * @retval ResultInvalidIfAddressMode
 *   The specified address mode (static versus DHCP) was
 *   invalid.
 * @retval ResultInvalidIfAddress
 *   The specified statically assigned IP address or DHCP acquired
 *   address was not accepted by the core TCP/IP stack.
 * @retval ResultMaximumExceeded
 *   The maximum number of simultaneously supported network interfaces
 *   have already been enabled via prior call to SetIfUp().
 * @retval ResultBusy
 *   Specified ifName is in the process of going UP, or is already
 *   UP.
 * @retval ResultCancelled
 *   Attempt cancelled via call to CancelIf().
 * @retval ResultIfSendError
 *   Unable to send packets out low level network interface.
 */
Result SetIfUp(const char *ifName, IfSettings* pIfOptionData);

/**
 * @brief Administatively enables the specified interface using
 *        supplied options. Also return a system event, to allow
 *        caller to know if state of interface changes later.
 *        At this time the only such thing would could later
 *        change is DHCP lease state.
 * @param[in] ifName
 *   Network interface name
 * @param[in] pIfOptionData
 *   Option data
 * @param[out] systemEvent
 *   Provisioned system event for providing asynchronous status
 *   changes after the interface initially transitioned to UP.
 * @param[in] eventClearMode
 *   Mode associated with system event.
 * @pre
 *   User has not already successfully called SetIfUp() for this ifName,
 *   OR user did and subsequently called SetIfDown().
 * @post
 * @details
 *   This call will block until the specified interface is
 *   successfully enabled, or until an unrecoverable error occurs.
 *   To cancel a call which is in progress, user must call
 *   CancelIf() from another thread.
 * @return Status of the attempted operation is returned.
 * @retval ResultSuccess
 *   Operation completed successfully.
 * @retval ResultIfInvalid
 *   Specified ifName does not exist.
 * @retval ResultMtuSizeError
 *   Core TCP/IP stack did not accept specified MTU value.
 * @retval ResultIfRemoved
 *   The underlying network interface device has detached.
 * @retval ResultRouteError
 *   Unable to configure route using specified IP parameters.
 *   There could be a problem with the specified address, subnet
 *   mask or gateway.
 * @retval ResultInvalidIfAddressMode
 *   The specified address mode (static versus DHCP) was
 *   invalid.
 * @retval ResultInvalidIfAddress
 *   The specified statically assigned IP address or DHCP acquired
 *   address was not accepted by the core TCP/IP stack.
 * @retval ResultMaximumExceeded
 *   The maximum number of simultaneously supported network interfaces
 *   have already been enabled via prior call to SetIfUp().
 * @retval ResultBusy
 *   Specified ifName is in the process of going UP, or is already
 *   UP.
 * @retval ResultCancelled
 *   Attempt cancelled via call to CancelIf().
 * @retval ResultIfSendError
 *   Unable to send packets out low level network interface.
 */
Result SetIfUp(const char *ifName, IfSettings *pIfOptionData,
               nn::os::SystemEvent *pSystemEvent, nn::os::EventClearMode eventClearMode);

/**
 * @brief Cancels prior unfinished call to SetIfUp().
 * @param[in] ifName
 *   Network interface name
 * @pre
 *   User has called SetIfUp() for this interface, and that
 *   call may or may not yet have returned.
 * @post
 *   Prior call to SetIfUp(), if still pending, is now
 *   in the process of returning.
 * @details
 *   This call will not block for significant lengths of time.
 * @return Status of the attempted operation is returned.
 * @retval ResultSuccess
 *   The cancel operation successfully caused SetIfUp() operation
 *   to be aborted.
 * @retval ResultIfInvalid
 *   There is no record of prior call to SetIfUp() for this ifName.
 * @retval ResultIfIsUp
 *   Prior call to SetIfUp() has already completed.
 */
Result CancelIf(const char *ifName);

/**
 * @brief Administatively disables the specified interface.
 * @param[in] ifName
 *   Network interface name
 * @pre
 *   User has already called SetIfUp() for this
 *   interface, and that call has returned successfully.
 * @post
 *   Upon successful completion, BSDCFG no longer has internal
 *   record of the specified ifName.
 * @details
 *   This call will not block for significant lengths of time.
 * @return Status of the attempted operation is returned.
 * @retval ResultSuccess
 *   Operation completed successfully.
 * @retval ResultIfInvalid
 *   There is no record of prior call to SetIfUp() for this ifName.
 * @retval ResultBusy
 *   Prior call to SetIfUp() has not yet completed.
 *
 */
Result SetIfDown(const char *ifName);

/**
 * @brief Administatively disables the specified interface.
 * @param[in] ifName
 *   Network interface name
  * @param[in] options
 *   Network interface down options
 * @pre
 *   User has already called SetIfUp() for this
 *   interface, and that call has returned successfully.
 * @post
 *   Upon successful completion, BSDCFG no longer has internal
 *   record of the specified ifName.
 * @details
 *   This call will not block for significant lengths of time.
 * @return Status of the attempted operation is returned.
 * @retval ResultSuccess
 *   Operation completed successfully.
 * @retval ResultIfInvalid
 *   There is no record of prior call to SetIfUp() for this ifName.
 * @retval ResultBusy
 *   Prior call to SetIfUp() has not yet completed.
 *
 */
Result SetIfDown(const char *ifName, uint32_t options);

/**
 * @brief Get current state of specified interface.
 * @param[in] ifName
 *   Network interface name
 * @param[out] pIfState
 *   Returned network interface state data
 * @pre
 * @post
 * @details
 *   This call will not block for significant lengths of time.
 * @return Status of the attempted operation is returned.
 * @retval ResultSuccess
 *   Operation completed successfully.
 * @retval ResultIfInvalid
 *   There is no record of prior call to SetIfUp() for this ifName.
 */
Result GetIfState(const char *ifName, IfState *pIfState);

/**
 * @brief Administatively causes a DHCP renew.
 * @pre
 *   DHCP has a valid lease.
 * @post
 *   DHCP is now in the renewing state.
 * @details
 *   This call will not block for significant lengths of time.
 * @return Status of the attempted operation is returned.
 * @retval ResultSuccess
 *   Operation completed successfully.
 * @retval ResultIfInvalid
 *   There is no record of prior call to SetIfUp() for this ifName.
 * @retval ResultBusy
 *   DHCP does not currently have a properly bound lease.
 * @retval ResultInvalidIfAddressMode
 *   Attempting to request renewal when address mode is not DHCP.
 *
 */
Result DhcpRenew(const char *ifName);

/**
*   @brief      Add a static arp entry for the provided address
*   @details    Adds a static arp entry for the provided address. Will replace any previously existing entry.
*   @pre        There is an active interface.
*
*   @param[in]  ipAddress                Address for which to add an arp entry
*   @param[in]  pInputHardwareAddress    Pointer to the 6 byte hardware address to associate with the provided ip address.
*   @param[in]  hardwareAddressSize      Size of the hardware address buffer (Should be 6 bytes).
*
*   @return     Result of the process is returned
*   @retval     ResultSuccess                      Success
*   @retval     ResultHardwareAddressSizeInvalid   Provided size of the hardware address buffer is not valid (not 6 bytes).
*   @retval     ResultInvalidIfAddress             Passed Ip or hardware address is invalid. (Address is not on subnet or non-standard address)
*   @retval     ResultIfInvalid                    An interface does not exist
*   @retval     ResultMemAllocStackError           Stack ran out of memory
*   @retval     ResultSocketOpenError              Failed to open a socket
*   @retval     ResultUnknownStackError            An unknown error occured
*/
Result AddStaticArpEntry(nn::socket::InAddr ipAddress, const uint8_t* pInputHardwareAddress, size_t hardwareAddressSize);
Result AddStaticArpEntry(in_addr ipAddress, const uint8_t* pInputHardwareAddress, size_t hardwareAddressSize);

/**
*   @brief      Remove an arp entry with the provided address
*   @details    Removes an arp entry with the provided address.
*   @pre        There is an active interface.
*
*   @param[in]  ipAddress                Address of arp entry to remove.
*
*   @return     Result of the process is returned
*   @retval     ResultSuccess                      Success
*   @retval     ResultArpEntryNotFound             Failed to find an arp entry for the provided IP Address
*   @retval     ResultIfInvalid                    An interface does not exist
*   @retval     ResultMemAllocStackError           Stack ran out of memory
*   @retval     ResultSocketOpenError              Failed to open a socket
*   @retval     ResultUnknownStackError            An unknown error occured
*/
Result RemoveArpEntry(nn::socket::InAddr ipAddress);
Result RemoveArpEntry(in_addr ipAddress);

/**
*   @brief      Look up the arp entry with the provided ip address.
*   @details    looks up an arp entry with the provided ip address, and returns the associated hardware address.
*   @pre        There is an active interface.
*
*   @param[out] pOutputHardwareAddress  Pointer to the buffer to fill with the looked up arp hardware address.
*   @param[in]  hardwareAddressSize     Size of the provided hardware address buffer (Should be 6 bytes).
*   @param[in]  ipAddress               Address of arp entry to look up.
*
*   @return     Result of the process is returned
*   @retval     ResultSuccess                      Success
*   @retval     ResultArpEntryNotFound             Failed to find an arp entry for the provided IP Address
*   @retval     ResultIfInvalid                    An interface does not exist
*   @retval     ResultHardwareAddressSizeInvalid   Provided size of the hardware address buffer is not valid (not 6 bytes).
*   @retval     ResultMemAllocStackError           Stack ran out of memory
*   @retval     ResultUnknownStackError            An unknown error occured
*/
Result LookupArpEntry(uint8_t* pOutputHardwareAddress, size_t hardwareAddressSize, nn::socket::InAddr ipAddress);
Result LookupArpEntry(uint8_t* pOutputHardwareAddress, size_t hardwareAddressSize, in_addr ipAddress);

/**
*   @brief      Look up the arp entry with the provided hardware address.
*   @details    looks up an arp entry with the provided hardware address, and returns the associated ip address.
*   @pre        There is an active interface.
*
*   @param[out] pOutputIPAddress    Pointer to the buffer to fill with the looked up arp ip address.
*   @param[in]  ipAddressSize       Size of the provided ip address buffer (Should be 4 bytes).
*   @param[in]  pHardwareAddress    Pointer to the hardware address of the arp entry to look up.
*   @param[in]  hardwareAddressSize Size of the provided hardware address buffer (Should be 6 bytes).
*
*   @return     Result of the process is returned
*   @retval     ResultSuccess                      Success
*   @retval     ResultArpEntryNotFound             Failed to find an arp entry for the provided IP Address
*   @retval     ResultIfInvalid                    An interface does not exist
*   @retval     ResultIpv4AddressSizeInvalid       Provided size of the Ipv4 address buffer is not valid (not 4 bytes).
*   @retval     ResultHardwareAddressSizeInvalid   Provided size of the hardware or ip address buffer is not valid (not 6 bytes).
*   @retval     ResultMemAllocStackError           Stack ran out of memory
*   @retval     ResultUnknownStackError            An unknown error occured
*/
Result LookupArpEntry(nn::socket::InAddr* pOutputIpAddress, size_t ipAddressSize, const uint8_t* pHardwareAddress, size_t hardwareAddressSize);
Result LookupArpEntry(in_addr* pOutputIpAddress, size_t ipAddressSize, const uint8_t* pHardwareAddress, size_t hardwareAddressSize);

/**
*   @brief      Clear all arp entries.
*   @details    Clears all arp entries in the arp table.
*
*   @return     Result of the process is returned
*   @retval     ResultSuccess                      Success
*   @retval     ResultMemAllocStackError           Stack ran out of memory
*   @retval     ResultUnknownStackError            An unknown error occured
*/
Result ClearArpEntries();

/**
*   @brief      Clear all arp entries for a given interface.
*   @details    Clears all arp entries associated with the provided interface.
*
*   @param[in]  pIfName   Pointer to the interface name
*
*   @return     Result of the process is returned
*   @retval     ResultSuccess                      Success
*   @retval     ResultMemAllocStackError           Stack ran out of memory
*   @retval     ResultUnknownStackError            An unknown error occured
*/
Result ClearArpEntries(const char * pIfName);

/**
*   @brief      Prints all arp entries.
*   @details    Prints all arp entries in the arp table.
*
*   @return     Result of the process is returned
*   @retval     ResultSuccess                      Success
*   @retval     ResultMemAllocStackError           Stack ran out of memory
*   @retval     ResultUnknownStackError            An unknown error occured
*/
Result PrintArpEntries();


} // namespace cfg
} // namespace bsdsocket
} // namespace nn





