﻿using LibGit2Sharp;
using Nintendo.Nact.FileSystem;
using Nintendo.Nact.Utilities;
using SigloNact.Utilities;
using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using Xunit;

namespace SigloNact.Test.Utilities
{
    public class GitUtilitiesTest
    {
        [Fact]
        void GetGitRevisionInfoTest()
        {
            const string testBranchName = nameof(GetGitRevisionInfoTest);
            string sha;
            string treeSha;
            var datetime = DateTimeOffset.FromUnixTimeSeconds(1514764800);

            var testRoot = FilePath.CreateLocalFileSystemPath(Environment.CurrentDirectory)
                .Combine(nameof(GetGitRevisionInfoTest));
            using (var testDir = new TemporaryDirectory(testRoot))
            {
                // コミットするダミーのファイル生成
                File.WriteAllText(testDir.Path.Combine("test.txt").PathString, "test");
                Repository.Init(testDir.Path.PathString);
                using (var repo = new Repository(testDir.Path.PathString))
                {
                    // ステージングとコミット
                    Commands.Stage(repo, "*");
                    var committer = new Signature("SigloNact", "SigloNact@nintendo.co.jp", datetime);
                    repo.Commit("test commit", committer, committer);
                    sha = repo.Head.Tip.Sha;
                    treeSha = repo.Head.Tip.Tree.Sha;

                    // ブランチ生成とチェックアウト
                    var branch = repo.CreateBranch(testBranchName);
                    Commands.Checkout(repo, branch);
                }

                var revInfo = GitUtilities.GetGitRevisionInfo(testDir.Path.PathString, "HEAD");

                Assert.Equal(sha, revInfo.CommitHash);
                Assert.Equal(testBranchName, revInfo.BranchName);
                Assert.Equal(1, revInfo.CommitCount);
                Assert.Equal(datetime, revInfo.CommitterDate);

                // コミット以外のハッシュが渡されたら例外を投げる
                Assert.Throws<ArgumentException>(() => GitUtilities.GetGitRevisionInfo(testDir.Path.PathString, treeSha));
            }
        }
    }
}
