﻿using Nintendo.Nact;
using Nintendo.Nact.ClassSystem;
using Nintendo.Nact.Evaluation;
using Nintendo.Nact.FileSystem;
using Nintendo.Nact.Foundation;
using System;
using System.Collections.Generic;
using System.Linq;
using static Nintendo.Nact.Extensions.NactValueExtensions;

namespace SigloNact
{
    class SubRootDefinition
    {
        public FilePath RootPath { get; }
        public FilePath IntermediatesDirectory { get; }
        public FilePath LibrariesDirectory { get; }
        public FilePath OutputsDirectory { get; }
        public IEnumerable<FilePath> IncludeDirectories { get; }
        public FilePath ResourcesDirectory { get; }

        private SubRootDefinition(
            FilePath rootPath,
            FilePath intermediatesDirectory,
            FilePath librariesDirectory,
            FilePath outputsDirectory,
            IEnumerable<FilePath> includeDirectories,
            FilePath resourcesDirectory)
        {
            RootPath = rootPath;
            IntermediatesDirectory = intermediatesDirectory;
            LibrariesDirectory = librariesDirectory;
            OutputsDirectory = outputsDirectory;
            IncludeDirectories = includeDirectories;
            ResourcesDirectory = resourcesDirectory;
        }

        private static readonly Symbol SymbolSubRootPath = SymbolRepository.GetSymbol("SubRootPath");
        private static readonly Symbol SymbolIntermediatesDirectoryPath = SymbolRepository.GetSymbol("IntermediatesDirectoryPath");
        private static readonly Symbol SymbolLibrariesDirectoryPath = SymbolRepository.GetSymbol("LibrariesDirectoryPath");
        private static readonly Symbol SymbolOutputsDirectoryPath = SymbolRepository.GetSymbol("OutputsDirectoryPath");
        private static readonly Symbol SymbolIncludeDirectoryPaths = SymbolRepository.GetSymbol("IncludeDirectoryPaths");
        private static readonly Symbol SymbolResourcesDirectoryPath = SymbolRepository.GetSymbol("ResourcesDirectoryPath");

        public static IEnumerable<SubRootDefinition> GetSubRootDefinitions(EvaluationState evaluationState, BuildTaskTree buildTaskTree)
        {
            // DefaultSubRootAccessor が定義された BuildNode を探す (BuildNode のパスはルール側で定義される)
            BuildNode definitionNode;
            {
                BuildNode rootNode;
                if (!buildTaskTree.BuildNodeCollection.TryGetNodeByPath(buildTaskTree.RootRulePath, out rootNode))
                {
                    throw new InternalLogicErrorException("Root build node cannot be found");
                }

                FilePath definitionNodePath;
                try
                {
                    var ev = new Evaluator(evaluationState, rootNode, rootNode);
                    definitionNodePath = ev.EvaluateVariableReferenceSourceCode("SigloNactSubRootDefinitionNodePath").CheckedCast<FilePath>();
                }
                catch (ErrorException)
                {
                    // TORIAEZU: TestRunner が生成した nact ルールを与えてもエラーにならないように回避
                    // throw new InternalLogicErrorException("Unexpected definition of SigloNactSubRootDefinitionNodePath", ex);
                    return Array.Empty<SubRootDefinition>();
                }

                if (!buildTaskTree.BuildNodeCollection.TryGetNodeByPath(definitionNodePath, out definitionNode))
                {
                    throw new InternalLogicErrorException("SubRoot definition build node cannot be found");
                }
            }

            // DefaultSubRootAccessor を評価してサブルート情報を取得
            var subRootDefinitions = new List<SubRootDefinition>();
            try
            {
                var ev = new Evaluator(evaluationState, definitionNode, definitionNode);
                var knownSubRootInfos = ev.EvaluateVariableReferenceSourceCode("DefaultSubRootAccessor.KnownSubRootInfos").NactToArray<IInstance>();

                foreach (var ins in knownSubRootInfos)
                {
                    subRootDefinitions.Add(new SubRootDefinition(
                         ev.EvaluateMember(ins, SymbolSubRootPath).CheckedCast<FilePath>(),
                         ev.EvaluateMember(ins, SymbolIntermediatesDirectoryPath).CheckedCast<FilePath>(),
                         ev.EvaluateMember(ins, SymbolLibrariesDirectoryPath).CheckedCast<FilePath>(),
                         ev.EvaluateMember(ins, SymbolOutputsDirectoryPath).CheckedCast<FilePath>(),
                         ev.EvaluateMember(ins, SymbolIncludeDirectoryPaths).NactToArray<FilePath>().ToArray(),
                         ev.EvaluateMember(ins, SymbolResourcesDirectoryPath).CheckedCast<FilePath>()));
                }
            }
            catch (ErrorException ex)
            {
                throw new InternalLogicErrorException("Unexpected definition of SubRootDefinition class", ex);
            }

            return subRootDefinitions;
        }
    }
}
