﻿using Nintendo.Nact;
using Nintendo.Nact.BuildEngine;
using Nintendo.Nact.Evaluation;
using Nintendo.Nact.FileSystem;
using Nintendo.Nact.Utilities;
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;

namespace SigloNact
{
    public class SigloWildcardCopyChecker
    {
        private SubRootDefinition[] m_SubRootDefinitions;
        private TaskResult[] m_TaskResults;

        public SigloWildcardCopyChecker(EvaluationState evaluationState,
            BuildTaskTree buildTaskTree,
            BuildResult buildResult)
        {
            m_SubRootDefinitions = SubRootDefinition.GetSubRootDefinitions(evaluationState, buildTaskTree).ToArray();
            m_TaskResults = buildResult.TaskResults.ToArray();
        }

        public void Check()
        {
            var outputDirectories = m_SubRootDefinitions.Select(x => x.OutputsDirectory);

            // ワイルドカード指定のファイルコピー検出
            AggregateTask.ParallelConvert(
                m_TaskResults.Where(x => x.CommandLines.Count > 0),
                x => CheckWildcardCopyCommandLine(x.CommandLines, x.Task.RootSourceInfo.Path, outputDirectories));
        }

        private int CheckWildcardCopyCommandLine(IEnumerable<string> commandLines, FilePath rulePath, IEnumerable<FilePath> outputDirectories)
        {
            foreach (var commandLine in commandLines)
            {
                var commandLineComponents = commandLine.Split(new[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                if (commandLineComponents.Length == 0)
                {
                    continue;
                }

                string command = commandLineComponents[0];
                FilePath[] copySourceFilePaths;
                try
                {
                    if (command.Equals("copy", StringComparison.OrdinalIgnoreCase))
                    {
                        // copy コマンドのチェック
                        string[] commandLineArgs = commandLineComponents.Skip(1)
                            .Where(x => !x.StartsWith("/", StringComparison.Ordinal))
                            .ToArray();
                        // オプションを除き、最後のコマンドライン引数以外はコピー元
                        // ただし、相対パスの場合はコマンドラインの情報だけでは解釈できないため無視
                        copySourceFilePaths = commandLineArgs.Take(commandLineArgs.Length - 1)
                            .Select(x => CreateMaybeRelativeFilePathFromCommandLineArg(x))
                            .Where(x => x.IsRooted)
                            .Select(x => FilePath.CreateLocalFileSystemPath(x.PathString))
                            .ToArray();
                    }
                    else if (command.Equals("xcopy", StringComparison.OrdinalIgnoreCase))
                    {
                        // xcopy コマンドのチェック
                        string[] commandLineArgs = commandLineComponents.Skip(1)
                            .Where(x => !x.StartsWith("/", StringComparison.Ordinal))
                            .ToArray();
                        // オプションを除いた最初のコマンドライン引数がコピー元
                        // ただし、相対パスの場合はコマンドラインの情報だけでは解釈できないため無視
                        var maybeRelativeFilePath = CreateMaybeRelativeFilePathFromCommandLineArg(commandLineArgs[0]);
                        copySourceFilePaths = maybeRelativeFilePath.IsRooted ?
                            new[] { FilePath.CreateLocalFileSystemPath(maybeRelativeFilePath.PathString) } :
                            Array.Empty<FilePath>();
                    }
                    else
                    {
                        continue;
                    }
                }
                catch (Exception e)
                {
                    throw new InternalLogicErrorException(
                        $"SigloWildcardCopyChecker: Failed to parse commandline '{commandLine}'. (Rule at '{rulePath}')", e);
                }

                // コピー元に出力ディレクトリ以下かつワイルドカードを使用したパスが含まれていれば警告
                if (copySourceFilePaths.Any(path => path.PathString.Contains("*")
                        && outputDirectories.Any(outDir => outDir.IsSameOrAncestorOf(path))))
                {
                    Log.DiagnosticMessage(DiagnosticsSeverity.Warning,
                        string.Format(CultureInfo.CurrentCulture, Strings.CommandLineCheck_WildcardCopy, rulePath),
                        commandLine);
                }
            }

            return 0;
        }

        static private MaybeRelativeFilePath CreateMaybeRelativeFilePathFromCommandLineArg(string path)
        {
            return MaybeRelativeFilePath.Create(
                path.Trim()
                    .Replace("\"", string.Empty)
                    .Replace(@"\\", @"\"));
        }
    }
}
