﻿using Nintendo.Nact;
using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.Execution;
using Nintendo.Nact.FileSystem;
using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Xml.Linq;

namespace SigloNact.BuiltIns.Target
{
    [NactConstants]
    public class VersionMetaData : INactObject
    {
        public ImmutableArray<object> NactObjectCreationArguments { get; }

        [NactObjectCreator]
        public VersionMetaData()
        {
            this.NactObjectCreationArguments = ImmutableArray.Create<object>();

            // バージョン値の取得は Action である MakeMetaXml 内部で行う
        }
    }

    [NactActionFunctionContainer]
    public static class MakeMetaXmlContainer
    {
        [NactActionFunction]
        public static NactActionResult MakeMetaXml(
            INactActionContext executionState,
            FilePath outputPath,
            IReadOnlyDictionary<string, object> metaData,
            FilePath versionXmlPath,
            bool removesEmptyMap
            )
        {
            var xElement = ConvertNactMapToXml(metaData, versionXmlPath, removesEmptyMap);

            using (var sw = new StreamWriter(outputPath.PathString, false, new UTF8Encoding(true)))
            {
                xElement.Save(sw);
            }

            return NactActionResult.CreateSuccess(
                new[] { versionXmlPath },
                new[] { outputPath });
        }

        private static UInt32 ParseVersionXml(FilePath versionXmlPath)
        {
            UInt32 version;
            try
            {
                using (var sr = new StreamReader(versionXmlPath.PathString, true))
                {
                    var element = XDocument.Load(sr);
                    try
                    {
                        var major = UInt32.Parse(element.Element("Product").Element("Version").Element("Major").Value);
                        var minor = UInt32.Parse(element.Element("Product").Element("Version").Element("Minor").Value);
                        var micro = UInt32.Parse(element.Element("Product").Element("Version").Element("Micro").Value);
                        var majorRelstep = UInt32.Parse(element.Element("Product").Element("Version").Element("MajorRelstep").Value);
                        var minorRelstep = UInt32.Parse(element.Element("Product").Element("Version").Element("MinorRelstep").Value);

                        version = major << 26 | minor << 20 | micro << 16 | majorRelstep << 8 | minorRelstep;
                    }
                    catch (NullReferenceException e)
                    {
                        throw new ErrorException(string.Format("Failed to get a version value from '{0}'.", versionXmlPath.PathString), e.Message, e);
                    }
                }
            }
            catch (IOException e)
            {
                throw new ErrorException(string.Format("Failed to load '{0}'.", versionXmlPath.PathString), e.Message, e);
            }

            return version;
        }

        private static XElement ConvertNactMapToXml(IReadOnlyDictionary<string, object> map, FilePath versionXmlPath, bool removesEmptyMap)
        {
            var kv = default(KeyValuePair<string, object>);

            try
            {
                kv = map.Single();
            }
            catch (InvalidOperationException)
            {
                if (map.Count < 1)
                {
                    throw new ErrorException("Missing root element for an xml.");
                }
                else
                {
                    throw new ErrorException("There is more than one root element for an xml.");
                }
            }

            return new XElement(kv.Key, ConvertNactValueToXmlRecursively(kv.Value, versionXmlPath, removesEmptyMap));
        }

        private static object ConvertNactValueToXmlRecursively(object value, FilePath versionXmlPath, bool removesEmptyMap)
        {
            if (value == null)
            {
                return null;
            }

            if (value is VersionMetaData)
            {
                return ParseVersionXml(versionXmlPath);
            }
            else if (value is IReadOnlyDictionary<string, object>)
            {
                var dic = (IReadOnlyDictionary<string, object>)value;

                var ret = dic.SelectMany(kv =>
                {
                    var vals = default(IEnumerable<object>);

                    if (kv.Value is IEnumerable<object>)
                    {
                        vals = (IEnumerable<object>)kv.Value;
                    }
                    else
                    {
                        vals = new object[] { kv.Value };
                    }
                    return vals.Select(v => ConvertNactValueToXmlRecursively(v, versionXmlPath, removesEmptyMap))
                               .Where(v => v != null)
                               .Select(v => new XElement(kv.Key, v));
                });

                if (removesEmptyMap && !ret.Any())
                {
                    return null;
                }
                else
                {
                    return ret;
                }
            }
            else if (value is string)
            {
                return value;
            }
            else if (value is double)
            {
                return value;
            }
            else if (value is bool)
            {
                return value;
            }
            else
            {
                throw new ErrorException(string.Format("Unexpected type value detected. Type = {0}", value.GetType().Name));
            }
        }
    }
}
