﻿using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.Execution;
using Nintendo.Nact.FileSystem;
using SigloNact.Utilities;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Xml;

namespace SigloNact.BuiltIns.Target
{
    [NactActionFunctionContainer]
    public static class CreateSystemVersionFileContainer
    {
        [NactActionFunction]
        public static NactActionResult CreateSystemVersionFile(
            INactActionContext context,
            FilePath firmwareVersionFilePath,
            string platform,
            FilePath revisionInfoFile,
            FilePath outputFilePath,
            FilePath digestFilePath)
        {
            var helper = context.Helper;

            var revisionInfo = DataContractSerializationUtil.DeserializeFromBytes<GitRevisionInfo>(helper.ReadAllBytes(revisionInfoFile));

            var document = new XmlDocument();
            using (var fs = helper.OpenRead(firmwareVersionFilePath))
            {
                document.Load(fs);
            }

            Create(helper, outputFilePath, platform, revisionInfo.CommitHash, document);

            CreateDigest(helper, digestFilePath, document);

            return helper.FinishAsSuccess();
        }

        // SystemVersionFile のフォーマットを変えるときは、下記ページの修正を行い、
        // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=173240056
        // 統合デバッグ環境構築チームにその旨を共有する必要がある。
        public static void Create(
            INactActionHelper helper,
            FilePath path,
            string platform,
            string revision,
            XmlDocument document)
        {
            var majorVersion = Parse<byte>(
                document, GetVersionNodePath("Major"), x => byte.Parse(x));

            var minorVersion = Parse<byte>(
                document, GetVersionNodePath("Minor"), x => byte.Parse(x));

            var microVersion = Parse<byte>(
                document, GetVersionNodePath("Micro"), x => byte.Parse(x));

            byte majorRelstep = 0;

            byte minorRelstep = 0;

            if (!Exists(document, GetVersionNodePath("MajorRelstep")))
            {
                majorRelstep = Parse<byte>(
                    document, GetVersionNodePath("Relstep"), x => byte.Parse(x));
            }
            else
            {
                majorRelstep = Parse<byte>(
                    document,
                    GetVersionNodePath("MajorRelstep"), x => byte.Parse(x));

                minorRelstep = Parse<byte>(
                    document,
                    GetVersionNodePath("MinorRelstep"), x => byte.Parse(x));
            }

            var displayVersion = string.Format(
                "{0}.{1}.{2}", majorVersion, minorVersion, microVersion);

            var displayName = new StringBuilder();

            displayName.AppendFormat(
                "NintendoSDK Firmware for {0} {1}-{2}.{3}",
                platform, displayVersion, majorRelstep, minorRelstep);

            if (Exists(document, GetDescriptionNodePath()))
            {
                displayName.AppendFormat(
                    " {0}",
                    document.SelectSingleNode(GetDescriptionNodePath()).InnerText);
            }

            var content = new List<byte>();
            content.AddRange(new byte[]
            {
                majorVersion,
                minorVersion,
                microVersion,
                (byte)0,
                majorRelstep,
                minorRelstep,
                (byte)0,
                (byte)0
            });
            content.AddRange(Encode(platform, 32));
            content.AddRange(Encode(revision, 64));
            content.AddRange(Encode(displayVersion, 24));
            content.AddRange(Encode(displayName.ToString(), 128));

            helper.WriteAllBytes(path, content.ToArray());
        }

        public static void CreateDigest(
            INactActionHelper helper,
            FilePath path,
            XmlDocument document)
        {
            var digest = Parse<string>(
                document, GetVersionNodePath("Digest"), x => x);

            helper.WriteAllBytes(path, Encode(digest, 64));
        }

        static private byte[] Encode(string text, int length)
        {
            byte[] bytes = Encoding.UTF8.GetBytes(text);

            if (bytes.Length >= length)
            {
                bytes = bytes.Take(length - 1).ToArray();
            }

            return bytes.Concat(new byte[length - bytes.Length]).ToArray();
        }

        private static string GetDescriptionNodePath()
        {
            return "/Product/Description";
        }

        private static string GetVersionNodePath(string name)
        {
            return string.Format("/Product/Version/{0}", name);
        }

        private static bool Exists(XmlDocument document, string xpath)
        {
            return document.SelectSingleNode(xpath) != null;
        }

        private static T Parse<T>(
            XmlDocument document, string xpath, Func<string, T> parser)
        {
            XmlNode node = document.SelectSingleNode(xpath);

            if (node == null)
            {
                throw new Exception(string.Format("{0} is not found", xpath));
            }

            if (string.IsNullOrEmpty(node.InnerText))
            {
                throw new Exception(string.Format("{0} has no value", xpath));
            }

            return parser(node.InnerText);
        }
    }
}
