﻿using Nintendo.Nact.Execution;
using Nintendo.Nact.FileSystem;
using Nintendo.Nact.Utilities;
using Nn.ResultTool;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;

namespace SigloNact.BuiltIns.ResultTools
{
    class ResultGenerator
    {
        public static NactActionResult InvokeWithHandleError(FilePath moduleMapFilePath, Func<NactActionResult> action)
        {
            try
            {
                return action();
            }
            catch (ModuleMapException e)
            {
                return NactActionResult.CreateFailure(e.Message, $"モジュール変換マップファイルを確認してください : {moduleMapFilePath}");
            }
            catch (Exception e)
            {
                return NactActionResult.CreateFailure(e.Message, e.StackTrace);
            }
        }

        public static IEnumerable<FilePath> EnumerateScriptPaths(IEnumerable<IReadOnlyDictionary<string, object>> resultScriptPathsAndPackageNames)
        {
            return resultScriptPathsAndPackageNames.Select(x => GetScriptPath(x));
        }

        public static string MakeSkipIfText(IEnumerable<IReadOnlyDictionary<string, object>> resultScriptPathsAndPackageNames)
        {
            var list = from p in resultScriptPathsAndPackageNames
                       let scriptPath = GetScriptPath(p)
                       let packageNames = GetPackageNames(p)
                       orderby scriptPath, packageNames
                       select string.Format("{0}=>{1}", scriptPath, packageNames);
            return string.Join("|", list);
        }

        private static void WriteTextIfModified(string path, Action<TextWriter> f)
        {
            Func<byte[]> makeContents = () =>
            {
                using (var ms = new MemoryStream())
                {
                    using (var w = new StreamWriter(ms, Encoding.UTF8))
                    {
                        f(w);
                        w.Flush();
                    }
                    return ms.ToArray();
                }
            };
            var newContents = makeContents();
            if (!Util.FileHasContents(path, newContents))
            {
                File.WriteAllBytes(path, newContents);
            }
        }

        public static void MakeCppResultFile(
            IEnumerable<IReadOnlyDictionary<string, object>> resultScriptPathsAndPackageNames,
            FilePath outputPath,
            FilePath moduleMapFilePath,
            string packageName,
            FilePath headerTemplatePath)
        {
            var moduleMap = new ModuleMap(moduleMapFilePath.PathString);
            var results = MakeResultRanges(resultScriptPathsAndPackageNames);
            WriteTextIfModified(outputPath.PathString, g =>
            {
                g.WriteLine(File.ReadAllText(headerTemplatePath.PathString));
                ResultDefinitionPrinter.PrintResultRangeForCpp(packageName, results, g, moduleMap);
            });
        }

        public static void MakeJsResultFile(
            IEnumerable<IReadOnlyDictionary<string, object>> resultScriptPathsAndPackageNames,
            FilePath outputPath,
            FilePath moduleMapFilePath,
            string variableName)
        {
            MakeJsResultFileImpl(resultScriptPathsAndPackageNames, outputPath, moduleMapFilePath, variableName, _ => true);
        }

        public static void MakeJsResultFileWithFullNameList(
            IEnumerable<IReadOnlyDictionary<string, object>> resultScriptPathsAndPackageNames,
            FilePath outputPath,
            FilePath moduleMapFilePath,
            string variableName,
            IEnumerable<string> fullNames)
        {
            MakeJsResultFileImpl(resultScriptPathsAndPackageNames, outputPath, moduleMapFilePath, variableName, r => fullNames.Contains(r.FullName));
        }

        public static void MakeXmlResultFile(
            IEnumerable<IReadOnlyDictionary<string, object>> resultScriptPathsAndPackageNames,
            FilePath outputPath,
            FilePath moduleMapFilePath)
        {
            var moduleMap = new ModuleMap(moduleMapFilePath.PathString);
            var results = MakeResultRanges(resultScriptPathsAndPackageNames);
            WriteTextIfModified(outputPath.PathString, g =>
            {
                ResultDefinitionSerializer.WriteAsXml(moduleMap, results, g);
            });
        }

        public static void MakeJsResultAndReferenceInfoFile(
            IEnumerable<IReadOnlyDictionary<string, object>> resultScriptPathsAndPackageNames,
            FilePath outputPath,
            FilePath moduleMapFilePath,
            string variableName,
            FilePath tagPath)
        {
            var moduleMap = new ModuleMap(moduleMapFilePath.PathString);
            var results = MakeResultRanges(resultScriptPathsAndPackageNames);
            var resultFullNames = results.Select(r => r.FullName);
            var xml = new XmlDocument();
            xml.Load(tagPath.PathString);
            var tags = ExtractReferenceCompoundInfo(xml).ToArray();
            var tagNames = tags.Select(x => x.Item1);
            var targets = new HashSet<string>(resultFullNames.Intersect(tagNames));
            var targetResults = results.Where(r => targets.Contains(r.FullName));
            var targetTags = tags.Where(r => targets.Contains(r.Item1));
            var tagMap = new Dictionary<string, string>();
            foreach (var p in targetTags)
            {
                tagMap[p.Item1] = p.Item2;
            }
            var text = ResultDefinitionSerializer.MakeJsonString(moduleMap, targetResults, tagMap);
            WriteTextIfModified(outputPath.PathString, g =>
            {
                g.WriteLine("var {0} = {1};", variableName, text);
            });
        }

        private static FilePath GetScriptPath(IReadOnlyDictionary<string, object> p)
        {
            return (FilePath)p["scriptPath"];
        }

        private static string GetPackageNames(IReadOnlyDictionary<string, object> p)
        {
            return (string)p["packageNames"];
        }

        private static IEnumerable<ResultRange> MakeResultRanges(IEnumerable<IReadOnlyDictionary<string, object>> resultScriptPathsAndPackageNames)
        {
            var list = from p in resultScriptPathsAndPackageNames
                       select new
                       {
                           ScriptPath = GetScriptPath(p),
                           PackageNames = GetPackageNames(p).Split(new[] { ',' }, StringSplitOptions.RemoveEmptyEntries),
                       };
            return from p in list
                   let path = p.ScriptPath
                   from r in ResultProcessor.ReadResultRanges(File.ReadAllText(path.PathString), path.PathString)
                   let packageNames = p.PackageNames
                   where packageNames.Contains("*") || packageNames.Contains(r.Package)
                   select r;
        }

        private static void MakeJsResultFileImpl(
            IEnumerable<IReadOnlyDictionary<string, object>> resultScriptPathsAndPackageNames,
            FilePath outputPath,
            FilePath moduleMapFilePath,
            string variableName,
            Func<ResultRange, bool> filter)
        {
            var moduleMap = new ModuleMap(moduleMapFilePath.PathString);
            var results = MakeResultRanges(resultScriptPathsAndPackageNames).Where(filter);
            var text = ResultDefinitionSerializer.MakeJsonString(moduleMap, results);
            using (var g = new StreamWriter(outputPath.PathString, false, Encoding.UTF8))
            {
                g.WriteLine("var {0} = {1};", variableName, text);
            }
        }

        private static IEnumerable<Tuple<string, string>> ExtractReferenceCompoundInfo(XmlDocument xml)
        {
            return from n in xml.GetElementsByTagName("compound").Cast<XmlNode>()
                   let e = n as XmlElement
                   where e != null
                   let nameTags = e.GetElementsByTagName("name")
                   let filenameTags = e.GetElementsByTagName("filename")
                   where nameTags.Count == 1 && filenameTags.Count == 1
                   let name = nameTags[0].InnerText
                   let filename = filenameTags[0].InnerText
                   select Tuple.Create(name, filename);
        }
    }
}
