﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.Build.Construction;
using Nintendo.MakeVisualStudioProject.Converter;

namespace Nintendo.MakeVisualStudioProject.Generator
{
    internal class ProjectGenerator
    {
        // TODO: コンストラクタと Generate メソッドのシグニチャがこれでいいのか再検討

        private ProjectRootElement m_VcProject;
        private PathUtility m_PathUtility;

        private IEnumerable<ProjectConfiguration> m_PossibleProjectConfigurations;

        public ProjectSetting ProjectSetting { get; private set; }
        public IEnumerable<IPlatformElementConverter> PlatformElementConverters { get; private set; }

        public bool EmbedConfigurationValidityFlag { get; set; }
        public bool EmbedNintendoSdkLibraryProjectDefinition { get; set; }
        public bool EmbedNintendoSdkSampleProjectDefinition { get; set; }
        public bool EmbedNintendoSdkTestProjectDefinition { get; set; }

        public ProjectGenerator(ProjectSetting setting, IEnumerable<IPlatformElementConverter> platformElementConverters)
        {
            foreach (var configuration in setting.TargetSettings)
            {
                var generator = platformElementConverters.SingleOrDefault(x => x.AcceptsBuildTargetSetting(configuration));
                if (generator == null)
                {
                    throw new ArgumentException(string.Format(
                        "不明なプラットフォームコンフィギュレーション '{0}|{1}' が含まれています",
                        configuration.Configuration,
                        configuration.Platform));
                }
            }

            ProjectSetting = setting;
            PlatformElementConverters = platformElementConverters;
        }

        public ProjectRootElement Generate(string projectDirectory, IDictionary<string, string> substitutionPaths)
        {
            m_VcProject = ProjectRootElement.Create();
            m_PathUtility = new PathUtility(projectDirectory, substitutionPaths);
            m_PossibleProjectConfigurations = EnumeratePossibleProjectConfigurations(ProjectSetting.TargetSettings);

            GenerateImpl();

            return m_VcProject;
        }

        private IEnumerable<ProjectConfiguration> EnumeratePossibleProjectConfigurations(IEnumerable<BuildTargetSetting> targetSettings)
        {
            var vcPlatforms = targetSettings.Select(x => x.Platform).Distinct().OrderBy(x => x);
            var vcConfigurations = targetSettings.Select(x => x.Configuration).Distinct().OrderBy(x => x);

            foreach (var vcPlatform in vcPlatforms)
            {
                foreach (var vcConfiguration in vcConfigurations)
                {
                    yield return new ProjectConfiguration(vcConfiguration, vcPlatform);
                }
            }
        }

        private void GenerateImpl()
        {
            // とりあえず XML の要素順に作っていく。メソッド呼び出しの順番を不用意に変えないこと
            AddProjectConfigurations();
            AddSourceFiles();
            AddHeaderFiles();
            AddUserFiles();
            AddConfigurationType();
            ImportProjectConfiguration();
            AddMiscGlobalProperties();
            AddNintendoSdkProperties();
            ImportDefaultDefinitions();
            AddConfigurationPropertyGroup();
            ImportCommonDefinitions();
            AddExtensionSettings();
            AddUserPropertySheets();
            AddUserMacros();
            AddCustomPropertyGroup();
            AddCustomItemDefinitionGroup();
            ImportTargets();
            AddExtensionTargets();
        }

        private void AddProjectConfigurations()
        {
            // 全 Platform で共通

            var projectConfigurations = AddItemGroup();
            projectConfigurations.Label = "ProjectConfigurations";

            foreach (var pc in m_PossibleProjectConfigurations)
            {
                var projectConfiguration = projectConfigurations.AddItem("ProjectConfiguration", string.Format("{0}|{1}", pc.Configuration, pc.Platform));

                projectConfiguration.AddMetadata("Configuration", pc.Configuration);
                projectConfiguration.AddMetadata("Platform", pc.Platform);
            }
        }

        private void AddSourceFiles()
        {
            // Platform 固有
            // ソースファイルと、ソースファイル固有のコンパイラオプションの設定

            // ItemGroup 要素を PlatformConfiguration ごとに作らずひとつにまとめる
            // (そうしないと、Visual Studio でソースファイルのプロパティウインドウを開いたときに表示がおかしくなることがある)

            var allSourceFiles = ProjectSetting.TargetSettings
                .Where(c => c.SourceFiles != null).SelectMany(c => c.SourceFiles).Select(c => c.Path).Distinct();
            if (!allSourceFiles.Any())
            {
                return;
            }

            var itemGroup = AddItemGroup();

            foreach (var sourceFile in allSourceFiles)
            {
                var clCompile = itemGroup.AddItem("ClCompile", ConvertToRelativePath(sourceFile));

                foreach (var pc in m_PossibleProjectConfigurations)
                {
                    var entry = default(SourceFile);

                    var targetSetting = GetBuildTargetSetting(pc);
                    if (targetSetting != null)
                    {
                        try
                        {
                            entry = targetSetting.SourceFiles.Where(c => c.Path == sourceFile).SingleOrDefault();
                        }
                        catch (InvalidOperationException)
                        {
                            throw new ProjectGenerationException(string.Format(
                                "ソースファイル '{0}' が多重登録されているため、適切なプロジェクト設定を生成できません。", sourceFile));
                        }
                    }

                    // 構成設定にソースファイルとして出てこなければ ExcludedFromBuild で除外
                    if (entry == null)
                    {
                        var metadataElement = clCompile.AddMetadata("ExcludedFromBuild", "true");
                        metadataElement.Condition = pc.MakeConfigurationPlatformConditionString();
                    }

                    // ソースファイルごとのコンパイラオプションを追加
                    if (entry != null)
                    {
                        System.Diagnostics.Debug.Assert(targetSetting != null);

                        var platformElementConverter = GetPlatformElementConverter(targetSetting);
                        platformElementConverter.AddCompileOptionMetadatas(clCompile, entry.CompileOptions, targetSetting);
                    }
                }
            }
        }

        private void AddHeaderFiles()
        {
            // 全 Platform で共通

            // ItemGroup 要素を PlatformConfiguration ごとに作らずひとつにまとめる

            var allHeaderFiles = ProjectSetting.TargetSettings.Where(c => c.HeaderFiles != null).SelectMany(c => c.HeaderFiles).Distinct();
            if (!allHeaderFiles.Any())
            {
                return;
            }

            var itemGroup = AddItemGroup();

            foreach (var headerFile in allHeaderFiles)
            {
                itemGroup.AddItem("ClInclude", ConvertToRelativePath(headerFile));
            }
        }

        private void AddUserFiles()
        {
            // 全 Platform で共通

            // TODO: AddHeaderFiles と実装を共通化

            var allNoneFiles = ProjectSetting.TargetSettings.Where(c => c.UserFiles != null).SelectMany(c => c.UserFiles).Distinct();
            if (!allNoneFiles.Any())
            {
                return;
            }

            var itemGroup = AddItemGroup();

            foreach (var noneFile in allNoneFiles)
            {
                itemGroup.AddItem("None", ConvertToRelativePath(noneFile));
            }
        }

        private void AddConfigurationType()
        {
            // 全 Platform で共通

            var globals = AddPropertyGroup();
            globals.Label = "Globals";

            globals.AddProperty("ConfigurationType", Enum.GetName(typeof(VcProjectConfigurationType), ProjectSetting.ConfigurationType));
        }

        private void ImportProjectConfiguration()
        {
            // 全 Platform で共通

            if (!string.IsNullOrEmpty(ProjectSetting.LocalPropertySheetName))
            {
                var importLocalProps = AddImport(ProjectSetting.LocalPropertySheetName);
                importLocalProps.Condition = string.Format("exists('{0}')", ProjectSetting.LocalPropertySheetName);
            }
            if (!string.IsNullOrEmpty(ProjectSetting.ProjectConfigurationPropertySheetName))
            {
                var importProjectConfiguration = AddImport(ProjectSetting.ProjectConfigurationPropertySheetName);
                if (!string.IsNullOrEmpty(ProjectSetting.LocalPropertySheetName))
                {
                    importProjectConfiguration.Condition = string.Format("exists('{0}')", ProjectSetting.LocalPropertySheetName);
                }
            }
        }

        private void AddMiscGlobalProperties()
        {
            // 全 Platform で共通

            // TODO: AddConfigurationType() とまとめてしまっても問題はないかもしれない

            var globals = AddPropertyGroup();
            globals.Label = "Globals";

            // TODO: 成果物名などから自動生成すれば外から与える必要はない
            var guid = ProjectSetting.ProjectGuid;
            if (!guid.StartsWith("{"))
            {
                guid = "{" + guid;
            }
            if (!guid.EndsWith("}"))
            {
                guid = guid + "}";
            }
            globals.AddProperty("ProjectGuid", guid);

            globals.AddProperty("Keyword", ProjectSetting.Keyword);
            globals.AddProperty("RootNamespace", ProjectSetting.RootNamespace);
        }

        private void AddNintendoSdkProperties()
        {
            if (EmbedNintendoSdkLibraryProjectDefinition)
            {
                {
                    var propertyGroup = AddPropertyGroup();

                    propertyGroup.AddProperty(
                        "NintendoSdkRoot", @"$([MSBuild]::GetDirectoryNameOfFileAbove($(MSBuildProjectDirectory), NintendoSdkRootMark))\");

                    var property = propertyGroup.AddProperty(
                        "NintendoSdkSubDevelopmentRoot", @"$([MSBuild]::GetDirectoryNameOfFileAbove($(MSBuildProjectDirectory), NintendoSdkSubDevelopmentRootMark))\");
                    property.Condition = @"'$([MSBuild]::GetDirectoryNameOfFileAbove($(MSBuildProjectDirectory), NintendoSdkSubDevelopmentRootMark))' != ''";
                }
                {
                    var propertyGroup = AddPropertyGroup();
                    propertyGroup.Condition = @"'$(NintendoSdkSubDevelopmentRoot)'!=''";

                    propertyGroup.AddProperty(
                        "NintendoSdkCommonPropertySheetDirectory", @"$(NintendoSdkSubDevelopmentRoot)Build\Vc\");
                    propertyGroup.AddProperty(
                        "NintendoSdkBuildEnvironment", @"Repository");
                }
                {
                    var propertyGroup = AddPropertyGroup();
                    propertyGroup.Condition = @"'$(NintendoSdkSubDevelopmentRoot)'==''";

                    propertyGroup.AddProperty(
                        "NintendoSdkCommonPropertySheetDirectory", @"$(NintendoSdkRoot)Build\Vc\");
                    propertyGroup.AddProperty(
                        "NintendoSdkBuildEnvironment", @"Package");
                }
            }
            if (EmbedNintendoSdkSampleProjectDefinition)
            {
                {
                    var choose = AddChoose();
                    {
                        var when = m_VcProject.CreateWhenElement(
                            @"'$([MSBuild]::GetDirectoryNameOfFileAbove($(MSBuildProjectDirectory), NintendoSdkRootMark))'!=''");
                        choose.AppendChild(when);
                        {
                            var propertyGroup = m_VcProject.CreatePropertyGroupElement();
                            when.AppendChild(propertyGroup);

                            propertyGroup.AddProperty(
                                "NINTENDO_SDK_ROOT", @"$([MSBuild]::GetDirectoryNameOfFileAbove($(MSBuildProjectDirectory), NintendoSdkRootMark))");
                        }
                    }
                    {
                        var when = m_VcProject.CreateWhenElement(
                            @"'$([MSBuild]::GetDirectoryNameOfFileAbove($(MSBuildProjectDirectory), NintendoSdkRootDefinition.props))'!=''");
                        choose.AppendChild(when);
                        {
                            var propertyGroup = m_VcProject.CreatePropertyGroupElement();
                            when.AppendChild(propertyGroup);

                            propertyGroup.AddProperty(
                                "NintendoSdkRootDefinitionFile", @"$([MSBuild]::GetDirectoryNameOfFileAbove($(MSBuildProjectDirectory), NintendoSdkRootDefinition.props))\NintendoSdkRootDefinition.props");
                        }
                    }
                }
                {
                    var import = AddImport(@"$(NintendoSdkRootDefinitionFile)");
                    import.Condition = @"'$(NintendoSdkRootDefinitionFile)'!=''";
                }
                {
                    var propertyGroup = AddPropertyGroup();
                    propertyGroup.Condition = @"'$(NINTENDO_SDK_ROOT)'!=''";

                    propertyGroup.AddProperty("NintendoSdkRoot", @"$(NINTENDO_SDK_ROOT)\");
                }
                {
                    var choose = AddChoose();
                    {
                        var when = m_VcProject.CreateWhenElement(
                            @"exists('$(NintendoSdkRoot)Build\Vc\ForApplication\NintendoSdkVcProjectSettings.props')");
                        choose.AppendChild(when);
                        {
                            var propertyGroup = m_VcProject.CreatePropertyGroupElement();
                            when.AppendChild(propertyGroup);

                            propertyGroup.AddProperty(
                                "NintendoSdkApplicationPropertySheetDirectory", @"$(NintendoSdkRoot)Build\Vc\ForApplication\");
                        }
                    }
                    {
                        var when = m_VcProject.CreateWhenElement(
                            @"exists('$(NintendoSdkRoot)Common\Build\Vc\ForApplication\NintendoSdkVcProjectSettings.props')");
                        choose.AppendChild(when);
                        {
                            var propertyGroup = m_VcProject.CreatePropertyGroupElement();
                            when.AppendChild(propertyGroup);

                            propertyGroup.AddProperty(
                                "NintendoSdkApplicationPropertySheetDirectory", @"$(NintendoSdkRoot)Common\Build\Vc\ForApplication\");
                        }
                    }
                }
                {
                    var choose = AddChoose();
                    {
                        var when = m_VcProject.CreateWhenElement(
                            @"exists('$(NintendoSdkRoot)Samples\Build\Vc\NintendoSdkSampleVcProjectSettings.props')");
                        choose.AppendChild(when);
                        {
                            var propertyGroup = m_VcProject.CreatePropertyGroupElement();
                            when.AppendChild(propertyGroup);

                            propertyGroup.AddProperty(
                                "NintendoSdkSamplePropertySheetDirectory", @"$(NintendoSdkRoot)Samples\Build\Vc\");
                        }
                    }
                }
            }
            if (EmbedNintendoSdkTestProjectDefinition)
            {
                var commonPropertyGroup = AddPropertyGroup();
                commonPropertyGroup.AddProperty(
                    "NintendoSdkRoot", @"$([MSBuild]::GetDirectoryNameOfFileAbove($(MSBuildProjectDirectory), NintendoSdkRootMark))\");
            }
        }

        private void ImportDefaultDefinitions()
        {
            // 全 Platform で共通

            var importMsCppDefaultProps = AddImport(@"$(VCTargetsPath)\Microsoft.Cpp.Default.props");

            if (!string.IsNullOrEmpty(ProjectSetting.DefaultDefinitionsPropertySheetName))
            {
                var importDefaultDefinitions = AddImport(ProjectSetting.DefaultDefinitionsPropertySheetName);
                if (!string.IsNullOrEmpty(ProjectSetting.LocalPropertySheetName))
                {
                    importDefaultDefinitions.Condition = string.Format("exists('{0}')", ProjectSetting.LocalPropertySheetName);
                }
            }
        }

        private void AddConfigurationPropertyGroup()
        {
            // 全 Platform で共通

            foreach (var pc in m_PossibleProjectConfigurations)
            {
                var propertyGroup = AddPropertyGroup();
                propertyGroup.Label = "Configuration";
                propertyGroup.Condition = pc.MakeConfigurationPlatformConditionString();

                var targetSetting = GetBuildTargetSetting(pc);
                if (targetSetting != null)
                {
                    // 成果物名を設定
                    //    * 本来は AddCustomPropertyGroup() で設定されるプロパティだが、このプロパティを IntDir, OutDir で
                    //      使用したいため、ImportCommonDefinitions() の前に設定する
                    //    * Visual Studio では構成ごとに変えられるようになっているので、それに合わせる
                    if (!string.IsNullOrEmpty(ProjectSetting.Name))
                    {
                        propertyGroup.AddProperty("TargetName", ProjectSetting.Name);
                    }

                    foreach (var property in targetSetting.ConfigurationProperties)
                    {
                        propertyGroup.AddProperty(property.Name, property.Value);
                    }
                }
                else
                {
                    if (EmbedConfigurationValidityFlag)
                    {
                        propertyGroup.AddProperty("_NintendoSdkIsUnsupportedProjectConfiguration", "true");
                    }
                }
            }
        }

        private void ImportCommonDefinitions()
        {
            // 全 Platform で共通

            var importMsCppProps = AddImport(@"$(VCTargetsPath)\Microsoft.Cpp.props");

            if (!string.IsNullOrEmpty(ProjectSetting.CommonDefinitionsPropertySheetName))
            {
                var importCommonDefinitions = AddImport(ProjectSetting.CommonDefinitionsPropertySheetName);
                if (!string.IsNullOrEmpty(ProjectSetting.LocalPropertySheetName))
                {
                    importCommonDefinitions.Condition = string.Format("exists('{0}')", ProjectSetting.LocalPropertySheetName);
                }
            }
        }

        private void AddExtensionSettings()
        {
            // 全 Platform で共通

            var extensionSettings = AddImportGroup();
            extensionSettings.Label = "ExtensionSettings";
        }

        private void AddUserPropertySheets()
        {
            // 全 Platform で共通

            foreach (var pc in m_PossibleProjectConfigurations)
            {
                var importGroup = AddImportGroup();
                importGroup.Label = "PropertySheets";
                importGroup.Condition = pc.MakeConfigurationPlatformConditionString();

                var targetSetting = GetBuildTargetSetting(pc);
                if (targetSetting != null)
                {
                    if (targetSetting.PropertySheets != null)
                    {
                        foreach (var propertySheet in targetSetting.PropertySheets)
                        {
                            importGroup.AddImport(ConvertToRelativePath(propertySheet));
                        }
                    }
                }
                else
                {
                    // 有効でないプロジェクト構成でビルドしようとした時に適切なエラーメッセージを表示できるよう適当なプロパティシートをインポートしておく
                    if (EmbedNintendoSdkSampleProjectDefinition)
                    {
                        importGroup.AddImport(string.Format(
                            "$(NintendoSdkSamplePropertySheetDirectory)NintendoSdkSampleSpec_{0}.props",
                            SigloProjectUtility.GetSpec(pc.Configuration)));
                        importGroup.AddImport(string.Format(
                            "$(NintendoSdkSamplePropertySheetDirectory)NintendoSdkSampleBuildType_{0}.props",
                            SigloProjectUtility.GetBuildType(pc.Configuration)));
                        importGroup.AddImport("$(NintendoSdkSamplePropertySheetDirectory)NintendoSdkSampleVcProjectSettings.props");
                    }
                }
            }
        }

        private void AddUserMacros()
        {
            // 全 Platform で共通

            var userMacros = AddPropertyGroup();
            userMacros.Label = "UserMacros";
        }

        private void AddCustomPropertyGroup()
        {
            // Platform 固有
            // ある Platform のビルド全体に共通する設定事項の Property

            foreach (var pc in m_PossibleProjectConfigurations)
            {
                var propertyGroup = AddPropertyGroup();
                propertyGroup.Condition = pc.MakeConfigurationPlatformConditionString();

                var targetSetting = GetBuildTargetSetting(pc);
                if (targetSetting == null)
                {
                    continue;
                }

                var platformElementConverter = GetPlatformElementConverter(targetSetting);
                platformElementConverter.AddCustomProperties(propertyGroup, targetSetting);
            }
        }

        private void AddCustomItemDefinitionGroup()
        {
            // Platform 固有
            // ある Platform のコンパイラ、アーカイバ、リンカの設定事項の ItemDefinition

            foreach (var pc in m_PossibleProjectConfigurations)
            {
                var itemDefinitionGroup = AddItemDefinitionGroup();
                itemDefinitionGroup.Condition = pc.MakeConfigurationPlatformConditionString();

                var targetSetting = GetBuildTargetSetting(pc);
                if (targetSetting == null)
                {
                    continue;
                }

                var platformElementConverter = GetPlatformElementConverter(targetSetting);
                platformElementConverter.AddCustomItemDefinitions(itemDefinitionGroup, ProjectSetting.ConfigurationType, targetSetting);
            }
        }

        private void ImportTargets()
        {
            // 全 Platform で共通

            var importTargets = AddImport(@"$(VCTargetsPath)\Microsoft.Cpp.targets");
        }

        private void AddExtensionTargets()
        {
            // 全 Platform で共通

            var extensionTargets = AddImportGroup();
            extensionTargets.Label = "ExtensionTargets";
        }

        private BuildTargetSetting GetBuildTargetSetting(ProjectConfiguration pc)
        {
            return ProjectSetting.TargetSettings.SingleOrDefault(x => x.Configuration == pc.Configuration && x.Platform == pc.Platform);
        }

        private IPlatformElementConverter GetPlatformElementConverter(BuildTargetSetting targetSetting)
        {
            return PlatformElementConverters.Single(x => x.AcceptsBuildTargetSetting(targetSetting));
        }

        private ProjectItemGroupElement AddItemGroup()
        {
            // Add*** 系のメソッドは XML 要素を先頭に挿入することがあるようなので、必ず Create*** と AppendChild を使う
            var ret = m_VcProject.CreateItemGroupElement();
            m_VcProject.AppendChild(ret);
            return ret;
        }
        private ProjectPropertyGroupElement AddPropertyGroup()
        {
            var ret = m_VcProject.CreatePropertyGroupElement();
            m_VcProject.AppendChild(ret);
            return ret;
        }
        private ProjectImportElement AddImport(string project)
        {
            var ret = m_VcProject.CreateImportElement(project);
            m_VcProject.AppendChild(ret);
            return ret;
        }
        private ProjectImportGroupElement AddImportGroup()
        {
            var ret = m_VcProject.CreateImportGroupElement();
            m_VcProject.AppendChild(ret);
            return ret;
        }
        private ProjectItemDefinitionGroupElement AddItemDefinitionGroup()
        {
            var ret = m_VcProject.CreateItemDefinitionGroupElement();
            m_VcProject.AppendChild(ret);
            return ret;
        }
        private ProjectChooseElement AddChoose()
        {
            var ret = m_VcProject.CreateChooseElement();
            m_VcProject.AppendChild(ret);
            return ret;
        }

        private string ConvertToRelativePath(string path)
        {
            if (m_PathUtility == null)
            {
                return path;
            }
            return m_PathUtility.ConvertToRelativePath(path);
        }
    }
}
