﻿using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.Build.Construction;
using Nintendo.MakeVisualStudioProject.Converter;

namespace Nintendo.MakeVisualStudioProject.Converter
{
    [Export(typeof(IPlatformElementConverter)), ExportMetadata("SupportedPlatform", "Win32")]
    internal class Win32PlatformElementConverter : PlatformElementConverter, IBuiltinOptionDefinitionsAvailable
    {
        private Dictionary<string, Win32OptionConverter> m_ClOptionConverters = new Dictionary<string, Win32OptionConverter>();
        private Dictionary<string, Win32OptionConverter> m_LibOptionConverters = new Dictionary<string, Win32OptionConverter>();
        private Dictionary<string, Win32OptionConverter> m_LinkOptionConverters = new Dictionary<string, Win32OptionConverter>();

        public bool UseBuiltinOptionDefinitions { get; set; }

        public override bool AcceptsBuildTargetSetting(BuildTargetSetting targetSetting)
        {
            return targetSetting.Platform == "Win32";
        }

        protected override void AddPreBuildEventItemDefinition(ProjectItemDefinitionGroupElement itemDefinitionGroup, BuildTargetSetting targetSetting)
        {
            base.AddPreBuildEventItemDefinition(itemDefinitionGroup, targetSetting);

            if (!string.IsNullOrEmpty(targetSetting.ApplicationDataDirectory))
            {
                // コマンド列の先頭に、data ディレクトリを作成するコマンドを入れる
                var commandString = new StringBuilder();
                commandString.AppendLine(string.Format(
                    @"set APPLICATION_DATA_DIRECTORY=""{0}""",
                    ConvertToSubstitutedPathOrRelativePath(targetSetting.ApplicationDataDirectory, targetSetting)));
                commandString.AppendLine(
                    @"(if exist ""$(OutDir)data"" rmdir /S /Q ""$(OutDir)data"") & mklink /j ""$(OutDir)data"" %APPLICATION_DATA_DIRECTORY%");

                var preBuildEvent = itemDefinitionGroup.ItemDefinitions.LastOrDefault(x => x.ItemType == "PreBuildEvent");
                if (preBuildEvent == null)
                {
                    preBuildEvent = itemDefinitionGroup.AddItemDefinition("PreBuildEvent");
                }

                var command = preBuildEvent.Metadata.LastOrDefault(x => x.Name == "Command");
                if (command == null)
                {
                    preBuildEvent.AddMetadata("Command", string.Format("{0}%(Command)", commandString));
                }
                else
                {
                    command.Value = string.Format("{0}{1}{2}", commandString, Environment.NewLine, command.Value);
                }
            }
        }

        protected override IEnumerable<ProjectMetadataElement> AddCompileOptionMetadatasFromOptionStrings(
            ProjectItemElement compileElement, IEnumerable<string> compileOptions, OptionConverter optionConverter)
        {
            var addedMetadatas = base.AddCompileOptionMetadatasFromOptionStrings(compileElement, compileOptions, optionConverter);
            return AddCompileOptionMetadatasFromOptionStringsImpl(compileElement, addedMetadatas);
        }

        protected override IEnumerable<ProjectMetadataElement> AddCompileOptionMetadatasFromOptionStrings(
            ProjectItemDefinitionElement compileElement, IEnumerable<string> compileOptions, OptionConverter optionConverter)
        {
            var addedMetadatas = base.AddCompileOptionMetadatasFromOptionStrings(compileElement, compileOptions, optionConverter);
            return AddCompileOptionMetadatasFromOptionStringsImpl(compileElement, addedMetadatas);
        }

        protected override OptionConverter GetCompileOptionConverter(BuildTargetSetting targetSetting)
        {
            return GetOptionConverter(targetSetting, m_ClOptionConverters, () =>
            {
                var platformToolset = GetPlatformToolset(targetSetting);
                var vcToolsVersion = GetVcToolsVersion(targetSetting);
                var vsVersion = GetMsBuildVsVersion(targetSetting);

                if (UseBuiltinOptionDefinitions)
                {
                    if (platformToolset.EndsWith("_clang_c2"))
                    {
                        return Win32OptionConverter.GetBuiltinClangC2OptionConverter(vcToolsVersion);
                    }
                    else
                    {
                        return Win32OptionConverter.GetBuiltinClOptionConverter(vcToolsVersion);
                    }
                }
                else
                {
                    if (platformToolset.EndsWith("_clang_c2"))
                    {
                        return Win32OptionConverter.GetClangC2OptionConverter(vcToolsVersion, vsVersion);
                    }
                    else
                    {
                        return Win32OptionConverter.GetClOptionConverter(vcToolsVersion, vsVersion);
                    }
                }
            });
        }

        protected override OptionConverter GetArchiveOptionConverter(BuildTargetSetting targetSetting)
        {
            return GetOptionConverter(targetSetting, m_LibOptionConverters, () =>
            {
                var vcToolsVersion = GetVcToolsVersion(targetSetting);
                var vsVersion = GetMsBuildVsVersion(targetSetting);

                if (UseBuiltinOptionDefinitions)
                {
                    return Win32OptionConverter.GetBuiltinLibOptionConverter(vcToolsVersion);
                }
                else
                {
                    return Win32OptionConverter.GetLibOptionConverter(vcToolsVersion, vsVersion);
                }
            });
        }

        protected override OptionConverter GetLinkOptionConverter(BuildTargetSetting targetSetting)
        {
            return GetOptionConverter(targetSetting, m_LinkOptionConverters, () =>
            {
                var vcToolsVersion = GetVcToolsVersion(targetSetting);
                var vsVersion = GetMsBuildVsVersion(targetSetting);

                if (UseBuiltinOptionDefinitions)
                {
                    return Win32OptionConverter.GetBuiltinLinkOptionConverter(vcToolsVersion);
                }
                else
                {
                    return Win32OptionConverter.GetLinkOptionConverter(vcToolsVersion, vsVersion);
                }
            });
        }

        protected override string CompilerName { get { return "cl.exe"; } }

        protected override string ArchiverName { get { return "lib.exe"; } }

        protected override string LinkerName { get { return "link.exe"; } }

        protected override string LibraryFileExtension { get { return ".lib"; } }

        private Win32OptionConverter GetOptionConverter(
            BuildTargetSetting targetSetting, Dictionary<string, Win32OptionConverter> converters, Func<Win32OptionConverter> converterFactory)
        {
            var platformToolset = GetPlatformToolset(targetSetting);

            if (!converters.ContainsKey(platformToolset))
            {
                converters[platformToolset] = converterFactory();
            }
            return converters[platformToolset];
        }

        private IEnumerable<ProjectMetadataElement> AddCompileOptionMetadatasFromOptionStringsImpl(
            dynamic compileElement, IEnumerable<ProjectMetadataElement> addedMetadatas)
        {
            var addedMetadataList = new List<ProjectMetadataElement>(addedMetadatas);

            // 一部のメタデータは強制的に AdditionalOptions として保存する (現状 DisableSpecificWarnings のみ)
            var moreAdditionalOptions = new List<string>();

            var disableSpecificWarnings = addedMetadataList.Where(e => e.Name == "DisableSpecificWarnings").SingleOrDefault();
            if (disableSpecificWarnings != null)
            {
                addedMetadataList.Remove(disableSpecificWarnings);
                compileElement.RemoveChild(disableSpecificWarnings);

                foreach (var warningNumber in disableSpecificWarnings.Value.Split(';'))
                {
                    int number;
                    if (int.TryParse(warningNumber, out number))
                    {
                        moreAdditionalOptions.Add(string.Format("/wd{0}", number));
                    }
                }
            }

            if (moreAdditionalOptions.Any())
            {
                var optionString = string.Join(" ", moreAdditionalOptions);
                var additionalOptionsMetadata = addedMetadataList.SingleOrDefault(x => x.Name == "AdditionalOptions");

                if (additionalOptionsMetadata != null)
                {
                    additionalOptionsMetadata.Value += $" {optionString}";
                }
                else
                {
                    addedMetadataList.Add(
                        // %(AdditionalOptions) は *前に* 書く (/w4 で有効化した警告を /wd で無効化する、などの要求があるため)
                        compileElement.AddMetadata("AdditionalOptions", "%(AdditionalOptions) " + optionString));
                }
            }

            return addedMetadataList;
        }

        private string GetPlatformToolset(BuildTargetSetting targetSetting)
        {
            var platformToolsetProperty = targetSetting.ConfigurationProperties.LastOrDefault(x => x.Name == "PlatformToolset");
            if (platformToolsetProperty == null)
            {
                throw new ConversionException(
                    $"プロジェクト構成 {targetSetting.Configuration}|{targetSetting.Platform} に対して構成プロパティ PlatformToolset が定義されていません。プロジェクト設定ファイルで定義してください");
            }
            else
            {
                return platformToolsetProperty.Value;
            }
        }

        private string GetVcToolsVersion(BuildTargetSetting targetSetting)
        {
            var platformToolset = GetPlatformToolset(targetSetting);

            switch (platformToolset)
            {
                case "v110":
                    return "4.0";
                case "v120":
                    return "12.0";
                case "v140":
                case "v140_clang_c2":
                    return "14.0";
                case "v141":
                case "v141_clang_c2":
                    return "15.0";
                default:
                    throw new ConversionException($"PlatformToolset プロパティの値が無効です: {platformToolset}");
            }
        }

        private string GetMsBuildVsVersion(BuildTargetSetting targetSetting)
        {
            var platformToolset = GetPlatformToolset(targetSetting);

            switch (GetPlatformToolset(targetSetting))
            {
                case "v110":
                    return "110";
                case "v120":
                    return "120";
                case "v140":
                case "v140_clang_c2":
                    return "140";
                case "v141":
                case "v141_clang_c2":
                    return "141";
                default:
                    throw new ConversionException($"PlatformToolset プロパティの値が無効です: {platformToolset}");
            }
        }
    }
}
