﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.MakeVisualStudioProject.Converter
{
    public class Win32OptionDefinitionGenerator
    {
        public static IEnumerable<OptionDefinition> GetClOptionDefinitions(string vcToolsVersion, string vsVersion)
        {
            return GenerateOptionDefinitions("cl.xml", vcToolsVersion, vsVersion);
        }

        public static IEnumerable<OptionDefinition> GetClangC2OptionDefinitions(string vcToolsVersion, string vsVersion)
        {
            return GenerateOptionDefinitions("clang.xml", vcToolsVersion, vsVersion);
        }

        public static IEnumerable<OptionDefinition> GetLibOptionDefinitions(string vcToolsVersion, string vsVersion)
        {
            return GenerateOptionDefinitions("lib.xml", vcToolsVersion, vsVersion);
        }

        public static IEnumerable<OptionDefinition> GetLinkOptionDefinitions(string vcToolsVersion, string vsVersion)
        {
            // link.exe のオプション '/MANIFESTUAC:' はふたつの引数を持つ真偽値型のオプションであり、引数に対応する
            // ふたつの ItemMetadata の生成がデフォルトの実装では行えない
            // とりあえず、単一引数のオプションとして独自に定義しておき、変換を Win32ItemMetadataConverter で対応する
            var ignoreMetadataNames = new string[]
            {
                "EnableUAC", "UACExecutionLevel", "UACUIAccess"
            };

            return GenerateOptionDefinitions("link.xml", vcToolsVersion, vsVersion, ignoreMetadataNames)
                .Concat(new[] {
                    new OptionDefinition("/MANIFESTUAC:", "EnableUAC", "true", true, requiresArg: true),
                    new OptionDefinition("/MANIFESTUAC:NO", "EnableUAC", "false"),
                });
        }

        private static IEnumerable<OptionDefinition> GenerateOptionDefinitions(
            string filename, string vcToolsVersion, string vsVersion, IEnumerable<string> ignoreMetadataNames = null)
        {
            var vcTargetsPath = OptionDefinitionGenerator.GetVCTargetsPath(vcToolsVersion, vsVersion);
            var lcid = System.Globalization.CultureInfo.InstalledUICulture.LCID.ToString();
            var xmlPath = Path.Combine(vcTargetsPath, lcid, filename);

            try
            {
                return OptionDefinitionGenerator.Generate(xmlPath, ignoreMetadataNames);
            }
            catch (Exception ex)
            {
                if (ex is FileNotFoundException || ex is DirectoryNotFoundException)
                {
                    throw new OptionGenerationException(
                        "MSVC ツールチェインのコマンドラインオプション定義を生成できません。適切なバージョンの Visual Studio がインストールされていることを確認してください。", ex);
                }
                else
                {
                    throw;
                }
            }
        }
    }
}
