﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;
using Nintendo.MakeVisualStudioProject;
using VsSolutionLibrary;

namespace Nintendo.MakeSampleSolution
{
    internal class Program
    {
        private static void Main(string[] args)
        {
            AppDomain.CurrentDomain.UnhandledException += UnhandledExceptionHandler;

            CommandLineOption option;
            {
                var parser = new CommandLineParser();

                try
                {
                    var continues = parser.ParseArgs(args, out option);

                    if (!continues)
                    {
                        Environment.Exit(0);
                        return;
                    }
                }
                catch
                {
                    Environment.Exit(1);
                    return;
                }
            }

            IEnumerable<VsSolutionVersion> solutionVersions;
            try
            {
                solutionVersions = CommandLineOption.ToVsSolutionVersions(option.VsVersionString);
            }
            catch (Exception ex)
            {
                Log.Error("コマンドラインオプションの解釈に失敗しました");
                Log.ReportException(ex);
                Environment.Exit(1);
                return;
            }

            var solutionSettings = ReadSolutionSettings(option.InputFile);

            foreach (var solutionSetting in solutionSettings)
            {
                ProcessSolutionSetting(solutionSetting, option.InputFile, option.OutputDirectory, solutionVersions);
            }

            Environment.Exit(0);
        }

        private static IEnumerable<SolutionSetting> ReadSolutionSettings(string path)
        {
            var inputPath = Path.GetFullPath(path);
            var inputFileDirectory = Path.GetDirectoryName(inputPath);

            if (!File.Exists(inputPath))
            {
                Log.Error("入力ファイル {0} が見つかりませんでした。", inputPath);
                Environment.Exit(1);
            }

            IEnumerable<SolutionSetting> solutionSettings = null;

            if (Path.GetExtension(inputPath) == ".yml" || Path.GetExtension(inputPath) == ".yaml")
            {
                try
                {
                    solutionSettings = SolutionSetting.ReadSolutionSettings(inputPath).ToArray();

                    foreach (var project in solutionSettings.SelectMany(x => x.Projects))
                    {
                        if (!Path.IsPathRooted(project.Path))
                        {
                            project.Path = MakeAbsolutePath(project.Path, inputFileDirectory);
                        }
                    }
                }
                catch (Exception ex)
                {
                    Log.Error("ソリューション設定ファイルの読み込みに失敗しました。");
                    Log.ReportException(ex);
                    Environment.Exit(1);
                }
            }
            else
            {
                solutionSettings = new SolutionSetting[]
                {
                    new SolutionSetting(new SolutionSetting.SolutionProject[] { new SolutionSetting.SolutionProject(inputPath) })
                };
            }

            return solutionSettings;
        }

        private static void ProcessSolutionSetting(
            SolutionSetting solutionSetting, string inputPath, string outputDirectory, IEnumerable<VsSolutionVersion> solutionVersions)
        {
            var absOutputDirectory = Path.GetFullPath(outputDirectory);

            var solutionProjects = ReadSolutionProjects(solutionSetting);

            var solutions = CreateSolutions(solutionProjects, solutionSetting, absOutputDirectory, solutionVersions);

            foreach (var solution in solutions)
            {
                if (solutionSetting.SolutionProperties != null)
                {
                    foreach (var kv in solutionSetting.SolutionProperties)
                    {
                        solution.SolutionProperties.Add(kv.Key, kv.Value);
                    }
                }

                string solutionPath;
                if (!string.IsNullOrEmpty(solutionSetting.OutputName))
                {
                    solutionPath = Path.Combine(absOutputDirectory, GetSolutionFileName(solutionSetting.OutputName, solution));
                }
                else
                {
                    solutionPath = Path.Combine(absOutputDirectory, GetSolutionFileName(Path.GetFileNameWithoutExtension(inputPath), solution));
                }

                SaveSolution(solutionPath, solution);
            }
        }

        private static IEnumerable<SolutionProject> ReadSolutionProjects(SolutionSetting solutionSetting)
        {
            try
            {
                return SolutionProject.ReadSolutionProjects(solutionSetting);
            }
            catch (Exception ex)
            {
                Log.Error("プロジェクトファイルの読み込みに失敗しました。");
                Log.ReportException(ex);
                Environment.Exit(1);
                return null;  // 到達しない
            }
        }

        private static IEnumerable<VsSolution> CreateSolutions(
            IEnumerable<SolutionProject> solutionProjects, SolutionSetting solutionSetting, string baseDirectory, IEnumerable<VsSolutionVersion> solutionVerions)
        {
            try
            {
                return solutionVerions
                    .Select(x => CreateSolution(solutionProjects, solutionSetting, baseDirectory, x))
                    .ToArray();
            }
            catch (Exception ex)
            {
                Log.Error("ソリューションファイルの生成に失敗しました。");
                Log.ReportException(ex);
                Environment.Exit(1);
                return null;  // 到達しない
            }
        }

        private static VsSolution CreateSolution(
            IEnumerable<SolutionProject> solutionProjects, SolutionSetting solutionSetting, string baseDirectory, VsSolutionVersion solutionVersion)
        {
            // ソリューションを作る
            var solution = new VsSolution();

            // ソリューションファイルのバージョンを設定
            solution.Version = solutionVersion;

            // プロジェクトファイルのパスと依存関係を設定
            foreach (var solutionProject in solutionProjects)
            {
                var projectFile = solutionProject.ProjectFile;

                solution.AddProject(
                    projectFile.GetGuid(),
                    new VsSolutionProjectProperty(
                        VsProjectType.VisualC,
                        Path.GetFileNameWithoutExtension(projectFile.Path),
                        MakeRelativePath(projectFile.Path, baseDirectory),
                        solutionProject.Dependencies.Select(x => x.ProjectFile.GetGuid())));

                if (solutionProject.SolutionFolderGuid != Guid.Empty)
                {
                    solution.AddNestRelation(new VsNestRelation(solutionProject.SolutionFolderGuid, projectFile.GetGuid()));
                }
            }

            // ソリューションフォルダを追加
            foreach (var folder in solutionSetting.SolutionFolderPaths)
            {
                var name = SolutionSetting.GetSolutionFolderName(folder.Key);
                var folderPath = SolutionSetting.GetSolutionFolderDirectoryName(folder.Key);
                solution.AddProject(folder.Value, new VsSolutionProjectProperty(VsProjectType.SolutionFolder, name, name));
                if (!string.IsNullOrWhiteSpace(folderPath))
                {
                    solution.AddNestRelation(new VsNestRelation(solutionSetting.SolutionFolderPaths[folderPath], folder.Value));
                }
            }

            // ソリューション構成を作成
            IEnumerable<SolutionConfiguration> solutionConfigurations = GetSolutionConfigurations(
                solutionProjects, solutionSetting.SolutionConfigurationPreference, solutionVersion);

            foreach (var solutionConfiguration in solutionConfigurations)
            {
                solution.AddSolutionConfiguration(new VsConfigurationPair(solutionConfiguration.Configuration, solutionConfiguration.Platform));
            }

            // ソリューション構成からプロジェクト構成への参照を設定
            var solutionProjectConfigurations = GetSolutionProjectConfigurations(
                solutionSetting,
                solutionConfigurations,
                solutionProjects);

            foreach (var solutionProjectConfiguration in solutionProjectConfigurations)
            {
                WarnInvalidSolutionProjectConfigurationIfNeeded(solutionProjectConfiguration);

                solution.SetProjectConfiguration(
                    new VsConfigurationPair(
                        solutionProjectConfiguration.SolutionConfiguration.Configuration,
                        solutionProjectConfiguration.SolutionConfiguration.Platform),
                    solutionProjectConfiguration.ProjectFile.GetGuid(),
                    new VsProjectConfiguration(
                        new VsConfigurationPair(
                            solutionProjectConfiguration.ProjectConfiguration.Configuration,
                            solutionProjectConfiguration.ProjectConfiguration.Platform),
                        solutionProjectConfiguration.DoBuild));
            }

            return solution;
        }

        private static string GetSolutionFileName(string baseName, VsSolution solution)
        {
            string vsVersion;
            try
            {
                vsVersion = ToVsVersionString(solution.Version).ToLowerInvariant();
            }
            catch (ArgumentException)
            {
                Log.Error("不明な Visual Studio バージョン {0} 用のソリューションファイルを作成しようとしました。", Enum.GetName(typeof(VsSolutionVersion), solution.Version));
                Environment.Exit(1);
                return null;  // 到達しない
            }

            return string.Format("{0}.{1}.sln", baseName, vsVersion.ToLower());
        }

        private static void SaveSolution(string solutionPath, VsSolution solution)
        {
            try
            {
                File.WriteAllText(solutionPath, solution.ToString(), Encoding.UTF8);
            }
            catch (Exception ex)
            {
                Log.Error("ソリューションファイルの書き出しに失敗しました。");
                Log.ReportException(ex);
                Environment.Exit(1);
            }
        }

        private static string MakeAbsolutePath(string path, string baseDirectory)
        {
            return Path.GetFullPath(Path.Combine(baseDirectory, path));
        }

        private static IEnumerable<SolutionConfiguration> GetSolutionConfigurations(
            IEnumerable<SolutionProject> solutionProjects,
            SolutionConfigurationPreference preference,
            VsSolutionVersion solutionVersion)
        {
            var projectConfigurations = solutionProjects.SelectMany(x => x.ProjectFile.GetProjectConfigurations());

            // スペックが揃っているか確認
            string spec = null;
            {
                var specs = projectConfigurations.Select(x => SigloProjectUtility.GetSpec(x)).Distinct();

                if (specs.Count() == 0)
                {
                    // プロジェクトが 1 つも指定されていない
                    System.Diagnostics.Debug.Assert(!projectConfigurations.Any());
                    return Enumerable.Empty<SolutionConfiguration>();
                }
                else if (specs.Count() == 1)
                {
                    // スペックが 1 つに揃っている
                    spec = specs.Single();
                }
                else
                {
                    throw new ArgumentException($"ソリューション内に異なるスペック用のプロジェクトが混在しています。スペック = [{string.Join(", ", specs)}]");
                }
            }

            // ソリューションから選択されるプロジェクト構成だけを洗い出す
            // * Visual Studio バージョンを考慮する (Win32, x64 ならソリューションのバージョンと同じ、そうでなければ常にデフォルトバージョン)
            // * OriginOnly の場合は、対象となるプロジェクトを列挙してからプロジェクト構成を洗い出す
            var projectConfigurationFilter = new Func<ProjectConfiguration, bool>(x =>
            {
                return (IsWindowsPlatform(x) && SigloProjectUtility.GetVsVersion(x) == ToVsVersionString(solutionVersion))
                    || (!IsWindowsPlatform(x) && SigloProjectUtility.GetVsVersion(x) == SigloProjectUtility.DefaultVsVersion);
            });

            var targetProjectConfigurations = default(IEnumerable<ProjectConfiguration>);
            switch (preference)
            {
                case SolutionConfigurationPreference.All:
                    targetProjectConfigurations = projectConfigurations
                        .Where(projectConfigurationFilter);
                    break;
                case SolutionConfigurationPreference.OriginOnly:
                    targetProjectConfigurations = GetOriginSolutionProjects(solutionProjects)
                        .SelectMany(x => x.ProjectFile.GetProjectConfigurations())
                        .Where(projectConfigurationFilter);
                    break;
                default:
                    throw new NotImplementedException();
            }

            // 列挙したプロジェクト構成からプラットフォーム、ビルドタイプ、ツールセットの直積を生成しソリューション構成とする
            //  * スペック、Visual Studio バージョンは考慮しない (ソリューションファイルが分かれる前提のため)
            //  * ソリューション構成に対応するプロジェクト構成が存在しない場合もあるが、ここでは考慮しない (後でソリューション構成の解決を行う)
            var platforms = targetProjectConfigurations.Select(x => x.Platform).Distinct();
            var buildTypes = targetProjectConfigurations.Select(x => SigloProjectUtility.GetBuildType(x)).Distinct();
            var toolsets = targetProjectConfigurations.Select(x => SigloProjectUtility.GetToolset(x)).Distinct();

            var solutionConfigurations = new List<SolutionConfiguration>();
            foreach (var toolset in toolsets)
            {
                foreach (var buildType in buildTypes)
                {
                    foreach (var platform in platforms)
                    {
                        solutionConfigurations.Add(new SolutionConfiguration(buildType, platform, spec, ToVsVersionString(solutionVersion), toolset));
                    }
                }
            }

            return solutionConfigurations;
        }

        private static IEnumerable<SolutionProject> GetOriginSolutionProjects(IEnumerable<SolutionProject> solutionProjects)
        {
            var ret = new HashSet<SolutionProject>(solutionProjects);

            foreach (var solutionProject in solutionProjects)
            {
                foreach (var dependency in solutionProject.Dependencies)
                {
                    ret.Remove(dependency);
                }
            }

            return ret;
        }

        private static IEnumerable<SolutionProjectConfiguration> GetSolutionProjectConfigurations(
            SolutionSetting solutionSetting,
            IEnumerable<SolutionConfiguration> solutionConfigurations,
            IEnumerable<SolutionProject> solutionProjects)
        {
            var resolutionPreference = solutionSetting.SolutionConfigurationResolutionPreference;
            var resolutionRules = Enumerable.Empty<SolutionConfigurationResolver.MapRule>();

            if (solutionSetting.SolutionConfigurationResolutionRules != null)
            {
                resolutionRules = CreateSolutionConfigurationResolverMapRules(solutionSetting.SolutionConfigurationResolutionRules);
            }

            foreach (var solutionProject in solutionProjects)
            {
                var projectFile = solutionProject.ProjectFile;
                var projectConfigurations = projectFile.GetProjectConfigurations().ToArray();

                foreach (var solutionConfiguration in solutionConfigurations)
                {
                    ProjectConfiguration mappedProjectConfiguration = null;
                    try
                    {
                        mappedProjectConfiguration = SolutionConfigurationResolver.Resolve(
                            solutionConfiguration, projectConfigurations, resolutionPreference, resolutionRules);
                    }
                    catch (Exception ex)
                    {
                        throw new ErrorException(
                            $"プロジェクト {projectFile.Path} をソリューション構成 {solutionConfiguration.ToString()} でビルドするための適切なプロジェクト構成が見つかりません",
                            ex);
                    }

                    if (mappedProjectConfiguration != null)
                    {
                        yield return new SolutionProjectConfiguration(projectFile, solutionConfiguration, mappedProjectConfiguration, true);
                    }
                    else
                    {
                        // もし対応するプロジェクト構成が無ければ、代わりのプロジェクト構成を適当に入れる
                        yield return new SolutionProjectConfiguration(projectFile, solutionConfiguration, projectConfigurations.First(), false);
                    }
                }
            }
        }

        private static IEnumerable<SolutionConfigurationResolver.MapRule> CreateSolutionConfigurationResolverMapRules(
            IEnumerable<string> ruleStrings)
        {
            foreach (var ruleString in ruleStrings)
            {
                var split = ruleString.Split(new[] { "=>" }, StringSplitOptions.RemoveEmptyEntries).Select(x => x.Trim()).ToArray();
                if (split.Length != 2)
                {
                    throw new ErrorException($"ソリューション構成解決ルールの書式が不正です: {ruleString}");
                }

                var solutionSplit = split[0].Split('|').Select(x => x.Trim()).ToArray();
                if (solutionSplit.Length != 2)
                {
                    throw new ErrorException($"ソリューション構成解決ルールの書式が不正です: {ruleString}");
                }

                var projectSplit = split[1].Split('|').Select(x => x.Trim()).ToArray();
                if (projectSplit.Length != 2)
                {
                    throw new ErrorException($"ソリューション構成解決ルールの書式が不正です: {ruleString}");
                }

                yield return new SolutionConfigurationResolver.MapRule(
                    solutionSplit[0], solutionSplit[1], projectSplit[0], projectSplit[1]);
            }
        }

        private static void WarnInvalidSolutionProjectConfigurationIfNeeded(SolutionProjectConfiguration solutionProjectConfiguration)
        {
            if (!solutionProjectConfiguration.DoBuild)
            {
                return;
            }

            var solutionConfiguration = solutionProjectConfiguration.SolutionConfiguration;
            var projectConfiguration = solutionProjectConfiguration.ProjectConfiguration;

            if (solutionConfiguration.Spec != SigloProjectUtility.GetSpec(projectConfiguration))
            {
                Log.Warn(
                    $"ソリューションのスペックは '{solutionConfiguration.Spec}' ですが、"
                    + $"プロジェクト '{solutionProjectConfiguration.ProjectFile.Path}' をスペック '{SigloProjectUtility.GetSpec(projectConfiguration)}' の"
                    + $"プロジェクト構成でビルドしようとしています ({solutionConfiguration} => {projectConfiguration})");
            }
            if (solutionConfiguration.VsVersion != SigloProjectUtility.GetVsVersion(projectConfiguration) && IsWindowsPlatform(projectConfiguration))
            {
                Log.Warn(
                    $"ソリューションの Visual Studio バージョンは '{solutionConfiguration.VsVersion}' ですが、"
                    + $"プロジェクト '{solutionProjectConfiguration.ProjectFile.Path}' を Visual Studio バージョン '{SigloProjectUtility.GetVsVersion(projectConfiguration)}' の"
                    + $"プロジェクト構成でビルドしようとしています ({solutionConfiguration} => {projectConfiguration})");
            }
        }

        private static bool IsWindowsPlatform(string platform)
        {
            return platform == "Win32" || platform == "x64";
        }
        private static bool IsWindowsPlatform(ProjectConfiguration projectConfiguration)
        {
            return IsWindowsPlatform(projectConfiguration.Platform);
        }

        private static string ToVsVersionString(VsSolutionVersion solutionVersion)
        {
            switch (solutionVersion)
            {
                case VsSolutionVersion.VisualStudio2012:
                    return "VS2012";
                case VsSolutionVersion.VisualStudio2013:
                    return "VS2013";
                case VsSolutionVersion.VisualStudio2015:
                    return "VS2015";
                case VsSolutionVersion.VisualStudio2017:
                    return "VS2017";
                default:
                    throw new ArgumentException(
                        $"認識できないソリューションバージョンです: {Enum.GetName(typeof(VsSolutionVersion), solutionVersion)}",
                        nameof(solutionVersion));
            }
        }

        private static bool HasProjectConfiguration(ProjectFile projectFile, string configuration, string platform)
        {
            return projectFile.GetProjectConfigurations().Any(x => x.Configuration == configuration && x.Platform == platform);
        }

        private static void UnhandledExceptionHandler(object sender, UnhandledExceptionEventArgs e)
        {
            var ex = e.ExceptionObject as Exception;
            if (ex != null)
            {
                Log.Error(ex.ToString());
            }
            else
            {
                Log.Error(e.ToString());
            }

            Environment.Exit(1);
        }

        #region nact.exe から実装をコピペ

        private static string MakeRelativePath(string path, string baseDirectory)
        {
            var isFromAbs = Path.IsPathRooted(baseDirectory);
            var isTargetAbs = Path.IsPathRooted(path);

            if (isFromAbs == isTargetAbs)
            {
                var normalizedFrom = NormalizePath(baseDirectory);
                var normalizedTarget = NormalizePath(path);

                if (isFromAbs)
                {
                    // from と target が共に絶対パスの場合
                    //   → ドライブが異なるなら target をそのまま返す
                    //      そうでなければドライブを除いて相対パスとして扱う

                    var fromRoot = Path.GetPathRoot(normalizedFrom);
                    var targetRoot = Path.GetPathRoot(normalizedTarget);

                    if (fromRoot != targetRoot)
                    {
                        return normalizedTarget;
                    }

                    normalizedFrom = normalizedFrom.Substring(fromRoot.Length);
                    normalizedTarget = normalizedTarget.Substring(targetRoot.Length);
                }

                // from と target が共に相対パスの場合
                {
                    var fromParts = normalizedFrom.Split('\\');
                    var targetParts = normalizedTarget.Split('\\');

                    var minNumParts = Math.Min(fromParts.Length, targetParts.Length);
                    int numMatch = 0;

                    for (numMatch = 0; numMatch < minNumParts; ++numMatch)
                    {
                        if (fromParts[numMatch] != targetParts[numMatch])
                        {
                            break;
                        }
                    }

                    var up = MakeRepeatString("..\\", fromParts.Length - numMatch);
                    var down = string.Join("\\", targetParts.Skip(numMatch));

                    var join = up + down;

                    return (join.Length == 0) ? "." : join;
                }
            }

            // それ以外
            //   → エラー

            throw new ArgumentException("相対パスが作成できません。",
                string.Format("基点={0}\n対象={1}\n", baseDirectory, path));
        }

        private static Regex s_DoubleBackSlash = new Regex(@"\\+", RegexOptions.Compiled);
        private static Regex s_DriveLetter = new Regex(@"[a-z]:", RegexOptions.Compiled);
        private static Regex s_Dotdot = new Regex(@"([^\\]+)\\\.\.\\", RegexOptions.Compiled);

        private static string NormalizePath(string path)
        {
            // スラッシュをバックスラッシュに
            var p1 = path.Replace('/', '\\');

            // バックスラッシュの連続を一つのバックスラッシュに
            var p2 = s_DoubleBackSlash.Replace(p1, @"\");

            // ドライブ文字を大文字に
            var p3 = s_DriveLetter.Replace(p2, x => x.Value.ToUpper());

            // . を消す
            // TODO: UNC の場合にローカルホストを消してしまう
            var p4 = p3.Replace(@"\.\", @"\");

            // .. を畳む
            var p5 = p4;
            {
                int start = 0;

                for (;;)
                {
                    bool isMatch = false;
                    p5 = s_Dotdot.Replace(p5, (m) =>
                    {
                        isMatch = true;
                        if (m.Groups[1].Value == "..")
                        {
                            start = m.Index + 2;
                            return m.Value;
                        }
                        else
                        {
                            return string.Empty;
                        }
                    }, 1, start);

                    if (!isMatch)
                    {
                        break;
                    }
                }
            }
            if (Path.IsPathRooted(path) && !Path.IsPathRooted(p5))
            {
                throw new ArgumentException(
                    "パスに含まれる .. が多すぎます。", string.Format("パス={0}\n", path));
            }

            return p5;
        }

        private static string MakeRepeatString(string one, int num)
        {
            var sb = new StringBuilder(one.Length * num);
            for (int i = 0; i < num; ++i)
            {
                sb.Append(one);
            }
            return sb.ToString();
        }

        #endregion
    }
}
