﻿using System;
using System.Collections.Generic;
using System.ComponentModel.Composition.Hosting;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.Build.Construction;
using Nintendo.MakePropertySheets.Constructor;

namespace Nintendo.MakePropertySheets
{
    internal class Program
    {
        public static CompositionContainer CompositionContainer = CreateCompositionContainer();

        private static void Main(string[] args)
        {
            var option = ParseCommandLineOptionOrDie(args);

            var project = CreateProject(option);

            SaveProject(project, option);
        }

        private static CommandLineOption ParseCommandLineOptionOrDie(string[] args)
        {
            var option = default(CommandLineOption);
            try
            {
                option = CommandLineOption.Parse(args);

                if (option == null)
                {
                    // ヘルプメッセージを表示したので終了
                    Environment.Exit(0);
                }
            }
            catch (ArgumentException)
            {
                // コマンドラインオプションのパースに失敗したので終了
                Environment.Exit(1);
            }
            return option;
        }

        private static ProjectRootElement CreateProject(CommandLineOption option)
        {
            var elementComposer = new ProjectElementConstructorComposer(CompositionContainer);

            var releaseIntent = new ReleaseIntent(option.BuildEnvironment, option.Spec, option.Deliverable, option.PublishSystemProgramSettings);

            return ProjectConstructor.Construct(Path.GetFileNameWithoutExtension(option.OutputPath), elementComposer.ElementConstructors, releaseIntent);
        }

        private static CompositionContainer CreateCompositionContainer()
        {
            var thisAssembly = System.Reflection.Assembly.GetExecutingAssembly();

            var catalog = new AggregateCatalog();
            catalog.Catalogs.Add(new AssemblyCatalog(thisAssembly));

            return new CompositionContainer(catalog);
        }

        private static void SaveProject(ProjectRootElement project, CommandLineOption option)
        {
            var oldTempFile = option.OutputPath + "-old";
            var newTempFile = option.OutputPath + "-new";

            using (var mem = new MemoryStream())
            {
                using (var writer = new StreamWriter(mem, Encoding.UTF8, 65536, true))
                {
                    project.Save(writer);
                }
                mem.Seek(0, SeekOrigin.Begin);

                var xml = new System.Xml.XmlDocument();
                xml.Load(mem);
                xml.DocumentElement.PrependChild(xml.CreateComment(" This is an auto-generated file. Do not modify this file directly. "));
                xml.DocumentElement.PrependChild(xml.CreateComment(" これは自動生成されたファイルです。このファイルを直接変更しないでください。"));
                xml.Save(newTempFile);
            }

            if (File.Exists(option.OutputPath))
            {
                File.Delete(oldTempFile);
                File.Move(option.OutputPath, oldTempFile);
            }

            File.Move(newTempFile, option.OutputPath);

            File.Delete(oldTempFile);
        }
    }
}
