﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"
#include "CustomFilter.h"

#include <string>
#include <iostream>
#include <fstream>
#include <git2/sys/filter.h>

// LNK4248 警告抑制用のダミー定義
struct git_filter_source
{
};

namespace Libgit2 {
    namespace {
        // ハッシュ化されたファイルの中身に書き込まれている接頭辞
        static const char* HashPrefix = "GitExternalStorage:";

        // フィルタの属性。sdk/.gitattributes の filter=... の値に合わせる。
        static const char* ExpandFilterAttribute = "GitExternalStorage";
        // libgit2 で管理する際に使うフィルタの識別名
        static const char* ExpandFilterName = "GitExternalStorageExpandSmudgeFilter";
        // フィルタの適用順のための優先度
        static const int ExpandFilterPriority = 50;
        // キャッシュファイルが準備できるまでポーリングする際の待ち時間（ミリ秒）
        static const int ExpandFilterPollingIntervalMillisec = 100;


        class ExpandSmudgeFilterImpl
            : public git_filter
        {
        public:
            static ExpandSmudgeFilterImpl& GetInstance()
            {
                static ExpandSmudgeFilterImpl s_Instance;
                return s_Instance;
            }


            // git_filter_apply_fn
            static int FilterApply(
                git_filter* selfBase,
                void** payload, /* may be read and/or set */
                git_buf* to,
                const git_buf* from,
                const git_filter_source* src)
            {
                ExpandSmudgeFilterImpl* self = static_cast<ExpandSmudgeFilterImpl*>(selfBase);

                // prefix が正しいか検査。
                // ついでに pSrc がハッシュの先頭を指すようにする。
                const char* pSrc = from->ptr;
                const char* pCorrect = HashPrefix;
                while(*pCorrect != '\0')
                {
                    if(*pCorrect != *pSrc)
                    {
                        // prefix が違っていたら入力をそのまま出力にする。
                        git_buf_set(to, from->ptr, from->size);
                        return 0;
                    }

                    pCorrect++;
                    pSrc++;
                }

                // ファイル名を作成
                std::string filepath;
                filepath.reserve(1024);
                filepath += self->GetCacheDirectoryPath();
                filepath += "\\";
                filepath += pSrc;

                // キャッシュファイルの中身を読み込む
                std::ifstream ifs;
                for(;;)
                {
                    // （キャッシュが用意されて）ファイルが開けるようになるまで待つ
                    ifs.open(filepath.c_str(), std::ios::binary);
                    if(ifs.is_open())
                    {
                        break;
                    }
                    // ダウンロード用スレッドが走っていなければ待っても無駄なので抜ける
                    if(ExpandSmudgeFilter::GetNumberOfRunningCacheDownloadThreads() == 0)
                    {
                        break;
                    }
                    System::Threading::Thread::Sleep(ExpandFilterPollingIntervalMillisec);
                }
                if(!ifs.is_open())
                {
                    // 開けなかったら入力をそのまま出力にする。
                    git_buf_set(to, from->ptr, from->size);
                    return 0;
                }

                size_t cacheDataSize = 0;
                {
                    ifs.seekg(0, std::ios::end);
                    std::ifstream::pos_type endPos = ifs.tellg();
                    ifs.seekg(0, std::ios::beg);
                    cacheDataSize = static_cast<size_t>(endPos - ifs.tellg());
                }

                // キャッシュの中身で置き換え
                git_buf_grow(to, cacheDataSize);
                ifs.read(to->ptr, cacheDataSize);
                to->size = cacheDataSize;

                return 0; // success
            }

            void SetCacheDirectoryPath(const char* value)
            {
                this->m_CacheDirectoryPath = value;
            }
            const char* GetCacheDirectoryPath() const
            {
                return this->m_CacheDirectoryPath;
            }

        private:
            ExpandSmudgeFilterImpl()
            {
            }
            ExpandSmudgeFilterImpl(const ExpandSmudgeFilterImpl&);
            ExpandSmudgeFilterImpl& operator= (const ExpandSmudgeFilterImpl&);

        private:
            const char* m_CacheDirectoryPath;
        };
    }

    void ExpandSmudgeFilter::Init(const char* cacheDirectoryPath)
    {
        ExpandSmudgeFilterImpl& impl = ExpandSmudgeFilterImpl::GetInstance();

        // キャッシュファイルのパスを設定
        impl.SetCacheDirectoryPath(cacheDirectoryPath);

        // git_filter の設定
        impl.version = GIT_FILTER_VERSION;
        impl.attributes = ExpandFilterAttribute;
        impl.initialize = NULL;
        impl.shutdown = NULL;
        impl.check = NULL;
        impl.apply = ExpandSmudgeFilterImpl::FilterApply;
        impl.cleanup = NULL;
    }

    void ExpandSmudgeFilter::Shutdown()
    {
        Unregister();
    }

    void ExpandSmudgeFilter::Register()
    {
        ExpandSmudgeFilterImpl& impl = ExpandSmudgeFilterImpl::GetInstance();
        // 二重に登録しようとしてもエラーコードが返ってくるだけなので無視する
        int result = git_filter_register(
            ExpandFilterName,
            &impl,
            ExpandFilterPriority);
    }

    void ExpandSmudgeFilter::Unregister()
    {
        // 二重に削除しようとしてもエラーコードが返ってくるだけなので無視する
        int result = git_filter_unregister(ExpandFilterName);
    }


    void ExpandSmudgeFilter::ResetNumberOfRunningCacheDownloadThreads(int n)
    {
        if(g_MutexForNumberOfRunningCacheDownloadThreads == nullptr)
        {
            g_MutexForNumberOfRunningCacheDownloadThreads = gcnew System::Threading::Mutex();
        }
        g_MutexForNumberOfRunningCacheDownloadThreads->WaitOne();
        g_NumberOfRunningCacheDownloadThreads = n;
        g_MutexForNumberOfRunningCacheDownloadThreads->ReleaseMutex();
    }
    int ExpandSmudgeFilter::GetNumberOfRunningCacheDownloadThreads()
    {
        int result;
        g_MutexForNumberOfRunningCacheDownloadThreads->WaitOne();
        result = g_NumberOfRunningCacheDownloadThreads;
        g_MutexForNumberOfRunningCacheDownloadThreads->ReleaseMutex();
        return result;
    }
    void ExpandSmudgeFilter::DecrementNumberOfRunningCacheDownloadThreads()
    {
        g_MutexForNumberOfRunningCacheDownloadThreads->WaitOne();
        g_NumberOfRunningCacheDownloadThreads--;
        g_MutexForNumberOfRunningCacheDownloadThreads->ReleaseMutex();
    }


}
