﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace GitExternalStorage.Transports
{
    /// <summary>
    /// 外部ストレージとして PC 上のストレージを利用する転送クラスです。
    /// </summary>
    public class LocalTransport : TransportBase
    {
        public string LocalPath { get; private set; }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="cacheDirectoryPath">キャッシュディレクトリへのパス</param>
        /// <param name="localpath">外部ストレージとする PC 上のディレクトリへのパス</param>
        public LocalTransport(string cacheDirectoryPath, string localpath)
            : base(cacheDirectoryPath)
        {
            this.LocalPath = localpath;
        }

        protected override void GetFile(string sha, string destFilePath)
        {
            var sourceFilePath = Path.Combine(this.LocalPath, sha);

            try
            {
                File.Copy(sourceFilePath, destFilePath);
            }
            catch (FileNotFoundException e)
            {
                throw new TransportFileNotFoundException(string.Format("File '{0}' not found.", sourceFilePath), e);
            }
            catch (DirectoryNotFoundException e)
            {
                throw new TransportPathNotFoundException(string.Format("Path '{0}' or '{1}' not found.", sourceFilePath, destFilePath), e);
            }
        }

        protected override void PutFile(string sha, string sourceFilePath)
        {
            var destFilePath = Path.Combine(this.LocalPath, sha);

            try
            {
                File.Copy(sourceFilePath, destFilePath);
            }
            catch (DirectoryNotFoundException e)
            {
                throw new TransportPathNotFoundException(string.Format("Path '{0}' or '{1}' not found.", sourceFilePath, destFilePath), e);
            }
        }

        public override IEnumerable<string> GetUnpushed(IEnumerable<string> files)
        {
            var externalFiles = new DirectoryInfo(this.LocalPath).GetFiles().Select(x => x.Name);
            return files.Except(externalFiles);
        }

        public override bool Exists(string sha)
        {
            var filePath = Path.Combine(this.LocalPath, sha);

            return File.Exists(filePath);
        }

        public override string GetPath()
        {
            return this.LocalPath;
        }
    }
}
