﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using GitExternalStorage.Git;

namespace GitExternalStorage.Core
{
    public static class Reference
    {
        /// <summary>
        /// ハッシュ化した各ファイルの状態を返します。
        /// </summary>
        /// <returns>ファイルの状態</returns>
        public static ReferenceInfo FindReferences(GitRepositoryBase gitRepository)
        {
            var entries = gitRepository.GetHeadTree();

            // ファイルサイズと内容から、ハッシュ化したファイルだけを抽出する。
            var targetEntries = entries.Where(x =>
            {
                if (x.FileSize == HashedContent.GetContentSize())
                {
                    var content = gitRepository.ShowHead(x.FilePath);
                    return HashedContent.IsMatch(content);
                }
                else
                {
                    return false;
                }
            });

            var all = new List<ReferenceEntry>();
            var toExpand = new List<ReferenceEntry>();
            var expanded = new List<ReferenceEntry>();
            var deleted = new List<ReferenceEntry>();

            foreach (var entry in targetEntries)
            {
                var referenceEntry = new ReferenceEntry()
                {
                    FilePath = entry.FilePath,
                    FileSize = entry.FileSize
                };

                all.Add(referenceEntry);

                if (File.Exists(entry.FilePath))
                {
                    using (var reader = new StreamReader(entry.FilePath))
                    {
                        var content = reader.ReadLine();
                        if (content != null && HashedContent.IsMatch(content))
                        {
                            referenceEntry.Sha = HashedContent.Parse(content).Sha;
                            toExpand.Add(referenceEntry);
                        }
                        else
                        {
                            expanded.Add(referenceEntry);
                        }
                    }
                }
                else
                {
                    deleted.Add(referenceEntry);
                }
            }

            return new ReferenceInfo() { All = all, ToExpand = toExpand, Expanded = expanded, Deleted = deleted };
        }
    }
}
