﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using GitExternalStorage.Consoles;
using GitExternalStorage.Core;

namespace GitExternalStorage.Commands
{
    /// <summary>
    /// ハッシュ化したファイルの状態を表示するコマンドです。
    /// </summary>
    public class StatusCommand : CommandBase
    {
        public StatusCommand()
            : base()
        {
        }

        public StatusCommand(CommandConsoleBase console)
            : base(console)
        {
        }

        public override void Run()
        {
            this.PrintReferences();
            this.PrintLocalCaches();
        }

        private void PrintReferences()
        {
            var referencesInfo = Reference.FindReferences(Environments.GitRepository);

            if (referencesInfo.ToExpand.Count() > 0)
            {
                CommandConsole.WriteLineToOut("== Unexpanded File ==");
                foreach (var e in referencesInfo.ToExpand)
                {
                    CommandConsole.WriteLineToOut("{0} {1}", e.Sha.Substring(0, 8), e.FilePath);
                }
            }
            if (referencesInfo.Expanded.Count() > 0)
            {
                CommandConsole.WriteLineToOut("== Expanded File ==");
                foreach (var e in referencesInfo.Expanded)
                {
                    var size = new FileInfo(e.FilePath).Length;
                    CommandConsole.WriteLineToOut("{0, 8} {1}", Utils.ValueToStringWithSizeUnit(size), e.FilePath);
                }
            }
            if (referencesInfo.Deleted.Count() > 0)
            {
                CommandConsole.WriteLineToOut("== Deleted File ==");
                foreach (var e in referencesInfo.Deleted)
                {
                    CommandConsole.WriteLineToOut("         {0}", e.FilePath);
                }
            }
        }

        private void PrintLocalCaches()
        {
            var cacheInfo = LocalCache.FindLocalCaches(Environments.GetCacheDirectoryPath(), Environments.Transport);
            var cacheDirectoryPath = Environments.GetCacheDirectoryPath();

            if (cacheInfo.Unpushed.Count() > 0)
            {
                CommandConsole.WriteLineToOut("== Unpushed File ==");
                foreach (var e in cacheInfo.Unpushed)
                {
                    var size = new FileInfo(Path.Combine(cacheDirectoryPath, e.Sha)).Length;
                    CommandConsole.WriteLineToOut("{0, 8} {1}", Utils.ValueToStringWithSizeUnit(size), e.Sha.Substring(0, 8));
                }
            }
            if (cacheInfo.Pushed.Count() > 0)
            {
                CommandConsole.WriteLineToOut("== Pushed File ==");
                foreach (var e in cacheInfo.Pushed)
                {
                    var size = new FileInfo(Path.Combine(cacheDirectoryPath, e.Sha)).Length;
                    CommandConsole.WriteLineToOut("{0, 8} {1}", Utils.ValueToStringWithSizeUnit(size), e.Sha.Substring(0, 8));
                }
            }
        }
    }
}
