﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace GitExternalRepository.Repository
{
    public abstract class RepositoryBase
    {
        /// <summary>
        /// 自身のクラスに定義された RepositoryDefinitionAttribute 属性を取得します。
        /// </summary>
        /// <returns>リポジトリの属性</returns>
        public RepositoryDefinitionAttribute GetRepositoryDefinitionAttribute()
        {
            return (RepositoryDefinitionAttribute)Attribute.GetCustomAttribute(this.GetType(), typeof(RepositoryDefinitionAttribute));
        }

        /// <summary>
        /// 自身のリポジトリ種別を返します。
        /// </summary>
        /// <returns></returns>
        public RepositoryType GetRepositoryType()
        {
            return GetRepositoryDefinitionAttribute().RepositoryType;
        }

        #region 抽象メソッド

        /// <summary>
        /// リポジトリを初期化します。
        /// </summary>
        /// <param name="directory">初期化するディレクトリ</param>
        public abstract void Init();

        /// <summary>
        /// 管理ディレクトリを分離できるかどうかを返します。
        /// </summary>
        /// <returns>分離可能なときは True</returns>
        public abstract bool CanSeparateMetaDirectory();

        /// <summary>
        /// リポジトリをクローンします。
        /// </summary>
        /// <param name="url">クローンするリポジトリの URL</param>
        /// <param name="revision">クローンするリビジョン</param>
        public abstract void Clone(string url, string revision);

        /// <summary>
        /// リポジトリをクローンします。管理ディレクトリを指定の場所に配置します。
        /// </summary>
        /// <param name="url">クローンするリポジトリの URL</param>
        /// <param name="revision">クローンするリビジョン</param>
        /// <param name="gitDirectory">管理ディレクトリの配置場所</param>
        public abstract void Clone(string url, string revision, string gitDirectory);

        /// <summary>
        /// リポジトリをクローンします。
        /// </summary>
        /// <param name="url">クローンするリポジトリの URL</param>
        /// <param name="cloneCheckoutOption">クローン方法</param>
        public abstract void Clone(string url, CloneCheckoutOption cloneCheckoutOption);

        /// <summary>
        /// リポジトリをクローンします。管理ディレクトリを指定の場所に配置します。
        /// </summary>
        /// <param name="url">クローンするリポジトリの URL</param>
        /// <param name="cloneCheckoutOption">クローン方法</param>
        /// <param name="gitDirectory">管理ディレクトリ</param>
        /// <param name="metaPathFormat">管理ファイルに記述するパスの形式</param>
        public abstract void Clone(string url, CloneCheckoutOption cloneCheckoutOption, string gitDirectory);

        /// <summary>
        /// リポジトリを開きます。
        /// </summary>
        public abstract void Open();

        /// <summary>
        /// 管理ディレクトリのパスを返します。
        /// </summary>
        /// <returns>管理ディレクトリパス</returns>
        public abstract string GetMetaDirectory();

        /// <summary>
        /// ルートディレクトリのパスを返します。
        /// </summary>
        /// <returns>ルートディレクトリパス</returns>
        public abstract string GetRepositoryRoot();

        /// <summary>
        /// リポジトリの URL を返します。
        /// </summary>
        /// <returns>リポジトリ URL</returns>
        public abstract string GetRepositoryRootUrl();

        /// <summary>
        /// ワーキングディレクトリがリポジトリであるかどうかを返します。
        /// </summary>
        /// <returns>git リポジトリ内であれば True, そうでなければ False</returns>
        public abstract bool IsInsideRepository();

        /// <summary>
        /// ワーキングディレクトリがリポジトリのルートであるかどうかを返します。
        /// </summary>
        /// <returns>git リポジトリルートであれば True, そうでなければ False</returns>
        public abstract bool IsRepositoryRoot();

        /// <summary>
        /// コンフィグ値を取得します。
        /// </summary>
        /// <param name="name">コンフィグ名</param>
        /// <returns>コンフィグ値</returns>
        public abstract string GetConfig(string name);

        /// <summary>
        /// コンフィグ値を設定します。
        /// </summary>
        /// <param name="name">コンフィグ名</param>
        public abstract void SetConfig(string name, string value);

        /// <summary>
        /// リビジョンを取得します。
        /// </summary>
        /// <param name="refs"></param>
        /// <returns></returns>
        public abstract string GetRevision(string reference);

        /// <summary>
        /// 現在のリビジョンを取得します。
        /// </summary>
        /// <returns></returns>
        public abstract string GetHeadRevision();

        /// <summary>
        /// 指定のファイルをステージします。
        /// </summary>
        /// <param name="filepath"></param>
        public abstract void Add(string filepath);

        /// <summary>
        /// コミットします。
        /// </summary>
        /// <param name="message"></param>
        public abstract void Commit(string message);

        /// <summary>
        /// 指定のリビジョンをチェックアウトします。
        /// </summary>
        /// <param name="revision">リビジョン</param>
        /// <param name="force">リポジトリに変更があってもチェックアウトを行うかどうか</param>
        ///
        public abstract void Checkout(string revision, bool force);

        /// <summary>
        /// クリーンします。
        /// </summary>
        public abstract void Clean();

        /// <summary>
        /// HEAD のファイルツリーを取得します。
        /// </summary>
        /// <returns>ツリー上のファイル一覧</returns>
        public abstract IEnumerable<string> GetHeadTree();

        /// <summary>
        /// メタディレクトリであるかどうかを確認します。
        /// </summary>
        /// <param name="metaDirectory"></param>
        /// <returns></returns>
        public abstract bool IsMetaDirectory();

        #endregion
    }
}
