﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace GitExternalRepository.Repository.Git
{
    /// <summary>
    /// 外部 git コマンドを利用して機能を提供するクラスです。
    /// </summary>
    [RepositoryDefinition(RepositoryType.Git)]
    public class GitRepositoryCommand : GitRepositoryBase
    {
        /*
         * git ls-tree 各行の出力フォーマット。
         *
         *   <mode> SP <type> SP <object> SP <object size> TAB <file>
         */
        private static readonly Regex LsTreeLineRegEx = new Regex(@"^(\S*) +(\S*) +(\S*) +(\S*)\t(.*)");

        private string workingDirectory;

        public GitRepositoryCommand(string workingDirectory)
        {
            this.workingDirectory = workingDirectory;
        }

        #region GitRepositoryBase の抽象メソッド実装

        public override void Init()
        {
            GitCommandUtils.RunGit(".", string.Format("init \"{0}\"", this.workingDirectory));
        }

        public override bool CanSeparateMetaDirectory()
        {
            return true;
        }

        public override void Clone(string url, string revision)
        {
            this.Clone(url, CloneCheckoutOption.None);
            this.Checkout(revision, false);
        }

        public override void Clone(string url, string revision, string gitDirectory)
        {
            this.Clone(url, CloneCheckoutOption.None, gitDirectory);
            this.Checkout(revision, false);
        }

        public override void Clone(string url, CloneCheckoutOption cloneCheckoutOption)
        {
            // clone 後に HEAD をチェックアウトするかどうかを指定するオプションを生成します。
            var cloneCheckoutOptionString = string.Empty;
            if (cloneCheckoutOption == CloneCheckoutOption.None)
            {
                cloneCheckoutOptionString = "--no-checkout";
            }

            GitCommandUtils.RunGit(".", string.Format("clone {0} {1} \"{2}\"", url, cloneCheckoutOptionString, this.workingDirectory));
        }

        public override void Clone(string url, CloneCheckoutOption cloneCheckoutOption, string gitDirectory)
        {
            // .git ディレクトリの配置場所は、親ディレクトリまでが存在する状態にしなければならない
            var gitParentDirectory = Path.GetDirectoryName(gitDirectory);
            if (gitParentDirectory != string.Empty && !Directory.Exists(gitParentDirectory))
            {
                Directory.CreateDirectory(gitParentDirectory);
            }

            // clone 後に HEAD をチェックアウトするかどうかを指定するオプションを生成します。
            var cloneCheckoutOptionString = string.Empty;
            if (cloneCheckoutOption == CloneCheckoutOption.None)
            {
                cloneCheckoutOptionString = "--no-checkout";
            }

            var output = string.Empty;
            if (Directory.Exists(gitDirectory))
            {
                // .git ディレクトリが存在するなら、これを再利用します。

                // 作業ディレクトリを作成
                Directory.CreateDirectory(this.workingDirectory);

                // .git ディレクトリの index ファイルを削除することで、チェックアウトの実行を可能にします。
                var indexFile = Path.Combine(gitDirectory, "index");
                File.Delete(indexFile);
            }
            else
            {
                // .git ディレクトリがないのでクローンします。
                GitCommandUtils.RunGit(".", string.Format("clone {0} --separate-git-dir \"{1}\" {2} \"{3}\"", url, gitDirectory, cloneCheckoutOptionString, this.workingDirectory));
            }

            // 以上の処理により、.git ディレクトリおよび作業ディレクトリ（中身は空、あるいは、.git ファイルだけがある状態）となります。

            // gitdir を相対パスに書き換えます。
            // 書き込み制限がかかっている場合があるため、一旦 .git ファイルを削除してから書き込みます。
            var dotGitFile = PathUtility.Combine(this.workingDirectory, ".git");
            var gitDirectoryRelativePath = PathUtility.GetRelativePath(this.workingDirectory + @"\", gitDirectory);
            File.Delete(dotGitFile);
            File.WriteAllText(dotGitFile, string.Format("gitdir: {0}", gitDirectoryRelativePath));

            // core.worktree を設定します。
            // 設定しなくても良いのですが、core.worktree に絶対パスが書き込まれてしまう環境があるらしいので、相対パスを設定しておきます。
            var gitConfigFile = Path.Combine(gitDirectory, "config");
            var workDirectoryRelativePath = PathUtility.GetRelativePath(gitDirectory + @"\", this.workingDirectory);
            this.SetConfig("core.worktree", workDirectoryRelativePath);
        }

        /// <summary>
        /// リポジトリを開きます。
        /// </summary>
        public override void Open()
        {
            // do nothing
        }

        public override string GetMetaDirectory()
        {
            string GitDirectory = GitCommandUtils.GetGitOutput(this.workingDirectory, "rev-parse --git-dir");
            try
            {
                using (var reader = File.OpenText(GitDirectory + "/commondir"))
                {
                    GitDirectory += "/" + reader.ReadLine();
                }
            }
            catch (FileNotFoundException)
            {
            }
            return GitDirectory;
        }

        public override string GetRepositoryRoot()
        {
            return GitCommandUtils.GetGitOutput(this.workingDirectory, "rev-parse --show-toplevel");
        }

        public override string GetRepositoryRootUrl()
        {
            return this.GetConfig("remote.origin.url");
        }

        public override bool IsInsideRepository()
        {
            try
            {
                return GitCommandUtils.GetGitOutput(this.workingDirectory, "rev-parse --is-inside-work-tree") == "true";
            }
            catch (GitRepositoryCommandFailedException)
            {
                // リポジトリ内にないとき、コマンドは "false" を出力するのではなく、
                // 終了ステータスとして 0 以外を吐いて失敗します。
                // そのため、ここで例外をハンドルし、"false" を返すようにします。
                return false;
            }
        }

        public override bool IsRepositoryRoot()
        {
            var metaPath = Path.Combine(this.workingDirectory, ".git");

            // 直下に .git ファイル/ディレクトリを持つかどうか判断したのち、正式なチェックを行います。
            if(File.Exists(metaPath) || Directory.Exists(metaPath))
            {
                return this.IsInsideRepository() && PathUtility.AreSame(this.GetRepositoryRoot(), this.workingDirectory);
            }

            return false;
        }

        public override string GetConfig(string name)
        {
            return GitCommandUtils.GetGitOutput(this.workingDirectory, string.Format("config --get {0}", name));
        }

        public override void SetConfig(string name, string value)
        {
            GitCommandUtils.RunGit(this.workingDirectory, string.Format("config {0} \"{1}\"", name, value));
        }

        public override string GetRevision(string reference)
        {
            return GitCommandUtils.GetGitOutput(this.workingDirectory, string.Format("rev-parse {0}", reference));
        }

        public override string GetHeadRevision()
        {
            return this.GetRevision("HEAD");
        }

        public override void Add(string filepath)
        {
            GitCommandUtils.RunGit(this.workingDirectory, string.Format("add \"{0}\"", filepath));
        }

        public override void Commit(string message)
        {
            GitCommandUtils.RunGit(this.workingDirectory, string.Format("commit -m \"{0}\"", message));
        }

        public override void Checkout(string revision, bool force)
        {
            if (!this.IsRevisionContained(revision))
            {
                this.Fetch();
            }

            var forceOptionString = string.Empty;
            if (force)
            {
                forceOptionString = "--force";
            }

            GitCommandUtils.RunGit(this.workingDirectory, string.Format("checkout {1} {2}", this.workingDirectory, revision, forceOptionString));
        }

        public override void Clean()
        {
            GitCommandUtils.RunGit(this.workingDirectory, "clean -xdff");
        }

        public override IEnumerable<string> GetHeadTree()
        {
            /*
             * ls-tree は、指定のリビジョンに含まれるファイルを列挙する
             *
             * -l, --long:
             *   ファイルサイズを表示する
             *
             * -z:
             *   各行をヌル文字で終端する
             *
             * -r:
             *   ツリーを再帰する
             *
             */
            var output = GitCommandUtils.GetGitOutput(this.workingDirectory, "ls-tree -lz -r HEAD");

            // 行ごとにパースする。ls-tree の各行はヌル文字で終端されている。
            var outputLines = output.Split(new[] { '\0' }, StringSplitOptions.RemoveEmptyEntries);

            var entries = outputLines.Select(x =>
            {
                var m = LsTreeLineRegEx.Match(x);
                if (m.Success)
                {
                    return m.Groups[5].Value;
                }
                else
                {
                    return null;
                }
            }).Where(x => x != null);

            return entries;
        }

        public override bool IsMetaDirectory()
        {
            try
            {
                // コマンドが成功するとき、対象はメタディレクトリである
                return GitCommandUtils.GetGitOutput(this.workingDirectory, string.Format("rev-parse --resolve-git-dir .")).Length > 0;
            }
            catch (GitRepositoryCommandFailedException)
            {
                // 対象がメタディレクトリでないとき、コマンドは失敗します。
                return false;
            }
        }

        #endregion

        /// <summary>
        /// 指定のリビジョンをリポジトリが含むかどうかを返します。
        /// </summary>
        /// <param name="revision">リビジョン</param>
        /// <returns>リビジョンを含むかどうか</returns>
        private bool IsRevisionContained(string revision)
        {
            try
            {
                // コマンドの成否のみを見るため、戻り値は評価しません。
                GitCommandUtils.GetGitOutput(this.workingDirectory, string.Format("cat-file -e {0}", revision));

                return true;
            }
            catch (GitRepositoryCommandFailedException)
            {
                // リポジトリ内にないとき、コマンドは "false" を出力するのではなく、
                // 終了ステータスとして 0 以外を吐いて失敗します。
                // そのため、ここで例外をハンドルし、"false" を返すようにします。
                return false;
            }
        }

        /// <summary>
        /// フェッチします。
        /// </summary>
        private void Fetch()
        {
            GitCommandUtils.RunGit(this.workingDirectory, "fetch origin");
        }
    }
}
