﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using GitExternalRepository.Configs;
using GitExternalRepository.Consoles;
using GitExternalRepository.Extensions;
using Nintendo.Foundation.IO;

namespace GitExternalRepository.Commands
{
    /// <summary>
    /// 外部リポジトリの非アクティブ化を行うコマンドです。
    /// </summary>
    [CommandDefinition(typeof(Parameters))]
    public class DeinitCommand : CommandBase
    {
        public class Parameters
        {
            [CommandLineOption('a', "all", Description = "Deinitialize all external repositories.", IsRequired = false)]
            public bool All { get; set; }

            [CommandLineValue(0, ValueName = "path", Description = "Specify an external repository directory.", IsRequired = false)]
            public string Path { get; set; }
        }

        public DeinitCommand()
        {
        }

        public DeinitCommand(ConsoleBase console)
            : base(console)
        {
        }

        public override bool Execute(object parameters)
        {
            var commandParameters = (Parameters)parameters;

            if (!this.IsValidParameters(commandParameters))
            {
                this.CommandConsole.WriteLineToOut("Please see the command's help with '-h/--help' options.");

                return true;
            }

            var activeRepositoryList = Environments.ActiveRepositoryList;

            var entries = activeRepositoryList.Read();

            if (commandParameters.All)
            {
                // アクティブリポジトリリスト上の全エントリを非アクティブ化します。
                foreach (var e in entries)
                {
                    this.DeinitializeEntry(e);
                }
            }
            else if(commandParameters.Path != null)
            {
                // 指定のディレクトリだけ非アクティブ化します。

                // 与えられた引数
                var path = commandParameters.Path.TrimSlash();
                // 親リポジトリのルートからの相対パス
                var relatedPathFromRoot = PathUtility.GetRelativePath(Environments.GitRepository.GetRepositoryRoot() + @"\", path);

                var entry = entries.FirstOrDefault(x => x.TargetDirectory == relatedPathFromRoot);
                if (entry != null)
                {
                    this.DeinitializeEntry(entry);
                }
                else
                {
                    throw new GitExternalRepositoryException(string.Format("'{0}' does not exist in the active repository list '{1}'.", relatedPathFromRoot, activeRepositoryList.FilePath));
                }
            }

            return true;
        }

        /// <summary>
        /// 引数の組み合わせが正しいかを検査します。
        /// </summary>
        /// <param name="parameters">引数</param>
        /// <returns>引数が正しければ True, そうでなければ False を返します。</returns>
        private bool IsValidParameters(Parameters parameters)
        {
            if (parameters.Path != null && !parameters.All)
            {
                return true;
            }

            if (parameters.Path == null && parameters.All)
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// 指定のエントリの外部リポジトリを非アクティブ化します。
        /// </summary>
        /// <param name="entry">リポジトリリストエントリ</param>
        private void DeinitializeEntry(ActiveRepositoryListEntry entry)
        {
            this.CommandConsole.WriteLineToOut("Deinitialize '{0}'.", entry.TargetDirectory);

            var activeRepositoryList = Environments.ActiveRepositoryList;

            this.CommandConsole.WriteLineToOut("Unregistering from the acitve repository list '{0}'.", activeRepositoryList.FilePath);

            // アクティブリポジトリリストからエントリを削除します。
            activeRepositoryList.Remove(new ActiveRepositoryListEntry()
            {
                TargetDirectory = entry.TargetDirectory,
                Url = entry.Url,
            });

            this.CommandConsole.WriteLineToOut("Done.");
        }
    }
}
