﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using GitExternalRepository.Configs;
using GitExternalRepository.Consoles;
using GitExternalRepository.Extensions;
using GitExternalRepository.Repository;
using Nintendo.Foundation.IO;

namespace GitExternalRepository.Commands
{
    /// <summary>
    /// 外部リポジトリの強制クリーンを行うコマンドです。
    /// </summary>
    [CommandDefinition(typeof(Parameters))]
    public class CleanCommand : CommandBase
    {
        public class Parameters
        {
            [CommandLineValue(0, ValueName = "path", Description = "Specify an external repository directory.", IsRequired = false)]
            public string Path { get; set; }
        }

        public CleanCommand()
        {
        }

        public CleanCommand(ConsoleBase console)
            : base(console)
        {
        }

        public override bool Execute(object parameters)
        {
            var commandParameters = (Parameters)parameters;

            var activeEntries = Environments.ActiveRepositoryList.Read();

            var repositoryList = Environments.RepositoryList;
            var entries = repositoryList.Exists() ? repositoryList.Read() : new List<RepositoryListEntry>();

            if (commandParameters.Path == null)
            {
                // アクティブリポジトリリスト上の全エントリをクリアします。
                foreach (var e in activeEntries)
                {
                    var entry = entries.FirstOrDefault(x => x.TargetDirectory == e.TargetDirectory);
                    if (entry != null)
                    {
                        this.Clean(entry);
                    }
                    else
                    {
                        // リビジョンの切り替えにより、アクティブリポジトリリストには存在するが、リポジトリリストには存在しない状態が起こりえる。
                        // このため、リポジトリリストにエントリが見つからない場合は、これを警告するメッセージを表示しつつ、処理を続行する。
                        this.CommandConsole.WriteLineToOut(string.Format("Info: '{0}' does not currently exist in the repository list '{1}'.", e.TargetDirectory, repositoryList.FilePath));
                    }
                }
            }
            else
            {
                // 指定のディレクトリだけクリーンします。

                // 与えられた引数
                var path = commandParameters.Path.TrimSlash();
                // 親リポジトリのルートからの相対パス
                var relatedPathFromRoot = PathUtility.GetRelativePath(Environments.GitRepository.GetRepositoryRoot() + @"\", path);

                var entry = entries.FirstOrDefault(x => x.TargetDirectory == relatedPathFromRoot);
                if (entry != null)
                {
                    if (activeEntries.Any(x => x.TargetDirectory == relatedPathFromRoot))
                    {
                        this.Clean(entry);
                    }
                    else
                    {
                        throw new GitExternalRepositoryException(string.Format("'{0}' is not initialized.", relatedPathFromRoot));
                    }
                }
                else
                {
                    throw new GitExternalRepositoryException(string.Format("'{0}' does not exist in the repository list '{1}'.", relatedPathFromRoot, repositoryList.FilePath));
                }
            }

            return true;
        }

        /// <summary>
        /// 指定のエントリの外部リポジトリをクリーンします。
        /// </summary>
        /// <param name="entry">リポジトリリストエントリ</param>
        private void Clean(RepositoryListEntry entry)
        {
            // ディレクトリパス
            var path = PathUtility.Combine(Environments.GitRepository.GetRepositoryRoot(), entry.TargetDirectory);
            // 親リポジトリのルートからの相対パス = エントリに記述されたパス
            var relatedPathFromRoot = entry.TargetDirectory;

            // 外部リポジトリの GitRepository インスタンス
            var exrepo = Environments.RepositoryFactory.Create(entry.RepositoryType, path);

            this.CommandConsole.WriteLineToOut("Clean '{0}'.", relatedPathFromRoot);

            // ディレクトリが存在しない、または、そのディレクトリがリポジトリルートでない場合は、
            // exrepo でチェックアウトしたリポジトリが存在していないとみなし、何もしません。
            if (!Directory.Exists(path) || !exrepo.IsRepositoryRoot())
            {
                this.CommandConsole.WriteLineToOut(string.Format("Info: '{0}' does not repository.", relatedPathFromRoot));
            }
            else
            {
                try
                {
                    exrepo.Clean();
                }
                catch (RepositoryOperationFailedException e)
                {
                    throw new GitExternalRepositoryException(
                        string.Format("Failed to clean the repository '{0}'.", relatedPathFromRoot), e);
                }

                this.CommandConsole.WriteLineToOut("Done.");
            }
        }
    }
}
