﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Text;

namespace Bin2Cpp
{
    /// <summary>
    /// C++ ソースコードを生成するためのクラスです。
    /// </summary>
    internal class CodeGenerator
    {
        private readonly byte[] m_BinaryData;
        private readonly string m_SymbolBegin;
        private readonly string m_SymbolEnd;
        private readonly string[] m_NameSpace;
        private readonly bool m_IsConst;
        private readonly int m_Align;

        private const string SigloHeader = @"/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * このファイルは直接編集しないで下さい。
 * このファイルは自動生成されます。
 * Don't edit this file directly.
 * This file is generated automatically.
 */";

        public CodeGenerator(byte[] binaryData, string symbolBegin, string symbolEnd, string nameSpace, bool isConst, int align)
        {
            if (binaryData == null || symbolBegin == null || symbolEnd == null)
            {
                throw new ArgumentNullException();
            }

            m_BinaryData = binaryData;
            m_SymbolBegin = symbolBegin;
            m_SymbolEnd = symbolEnd;
            m_IsConst = isConst;
            m_Align = align;

            if (nameSpace != null)
            {
                m_NameSpace = nameSpace.Split(new[] { ':' }, StringSplitOptions.RemoveEmptyEntries);
            }
        }

        public string GetHeader()
        {
            string nameSpaceBegin = string.Empty;
            string nameSpaceEnd = string.Empty;

            if (m_NameSpace != null)
            {
                nameSpaceBegin = $"namespace {m_NameSpace} {{";
                nameSpaceEnd = "}}";
            }

            return $@"{SigloHeader}

#pragma once

{nameSpaceBegin}

    extern void* {m_SymbolBegin};
    extern void* {m_SymbolEnd};

{nameSpaceEnd}
";
        }

        public string GetCpp()
        {
            var nameSpaceBegin = new StringBuilder();
            var nameSpaceEnd = new StringBuilder();

            if (m_NameSpace != null)
            {
                foreach (var name in m_NameSpace)
                {
                    nameSpaceBegin.AppendLine($"namespace {name} {{ ");
                    nameSpaceEnd.AppendLine("}");
                }
            }

            var body = new StringBuilder();
            foreach (var data in m_BinaryData)
            {
                body.AppendLine($"        0x{data:x2},");
            }

            var align = m_Align != 0 ? $@"NN_ALIGNAS({m_Align})" : "";
            var qualifier = m_IsConst ? "const" : "";

            return $@"{SigloHeader}

#include <nn/nn_Common.h>

{nameSpaceBegin.ToString()}

namespace {{
    {align} {qualifier} unsigned char _data[] =
    {{
{body.ToString()}
    }};
}}

    {qualifier} void* {m_SymbolBegin} = _data;
    {qualifier} void* {m_SymbolEnd} = _data + sizeof(_data);

{nameSpaceEnd.ToString()}
";
        }

    }
}
