﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/utilTool/utilTool_XmlParser.h>

NN_PRAGMA_PUSH_WARNINGS
#pragma warning( disable:4365 )
#pragma warning( disable:4625 )
#pragma warning( disable:4626 )

#include <rapidxml.hpp>

using namespace rapidxml;

NN_PRAGMA_POP_WARNINGS

namespace nn { namespace utilTool {

namespace detail {

typedef xml_node<> NodeImpl;

inline
const NodeImpl* ToImpl( const XmlNode* pNode )
{
    return reinterpret_cast< const NodeImpl* >( pNode );
}

inline
const XmlNode* ToNode( const NodeImpl* pImpl )
{
    return reinterpret_cast< const XmlNode* >( pImpl );
}

} // namespace detail

//--------------------------------------------------------------------------------------------------

class XmlParser::Impl : public xml_document<>
{
    NN_DISALLOW_COPY( Impl );
public:
    Impl() : xml_document(), errWhat(), errWhere() {}

    nn::util::string_view errWhat;
    nn::util::string_view errWhere;
};

XmlParser::XmlParser()
{
    m_pImpl = new Impl();
}

XmlParser::~XmlParser()
{
    delete m_pImpl;
}

bool XmlParser::Parse( char* pDoc )
{
    m_pImpl->errWhat.clear();
    m_pImpl->errWhere.clear();
    try
    {
        m_pImpl->parse< 0 >( pDoc );
        return true;
    }
    catch( parse_error& err )
    {
        m_pImpl->errWhat = err.what();
        m_pImpl->errWhere = err.where< char >();
    }
    catch( ... )
    {
        m_pImpl->errWhat = "unknown error";
        m_pImpl->errWhere = "";
    }
    return false;
}

void XmlParser::Clear()
{
    m_pImpl->clear();
}

const XmlNode* XmlParser::GetRoot() const
{
    return detail::ToNode( m_pImpl->first_node() );
}

nn::util::string_view XmlParser::GetErrorString() const
{
    return m_pImpl->errWhat;
}

nn::util::string_view XmlParser::GetErrorPosition() const
{
    return m_pImpl->errWhere;
}

//--------------------------------------------------------------------------------------------------

nn::util::string_view XmlNode::GetName() const
{
    const detail::NodeImpl* pImpl = detail::ToImpl( this );
    return nn::util::string_view( pImpl->name(), static_cast< int >( pImpl->name_size() ) );
}

const XmlNode* XmlNode::GetParent() const
{
    const xml_node<>* pParent = detail::ToImpl( this )->parent();
    if ( pParent->type() == node_element )
    {
        return detail::ToNode( pParent );
    }
    else
    {
        return NULL;
    }
}

const XmlNode* XmlNode::GetChild() const
{
    const xml_node<>* pChild = detail::ToImpl( this )->first_node();
    while ( pChild && pChild->type() != node_element )
    {
        pChild = pChild->next_sibling();
    }
    return detail::ToNode( pChild );
}

const XmlNode* XmlNode::GetNext() const
{
    const xml_node<>* pNext = detail::ToImpl( this )->next_sibling();
    while ( pNext && pNext->type() != node_element )
    {
        pNext = pNext->next_sibling();
    }
    return detail::ToNode( pNext );
}

const XmlNode* XmlNode::GetPrev() const
{
    const xml_node<>* pPrev = detail::ToImpl( this )->previous_sibling();
    while ( pPrev && pPrev->type() != node_element )
    {
        pPrev = pPrev->previous_sibling();
    }
    return detail::ToNode( pPrev );
}

const XmlNode* XmlNode::FindChild( const nn::util::string_view& name ) const
{
    return detail::ToNode( detail::ToImpl( this )->first_node(
        name.data(), static_cast< size_t >( name.size() ) ) );
}

const XmlNode* XmlNode::FindNext() const
{
    const detail::NodeImpl* pImpl = detail::ToImpl( this );
    return detail::ToNode( pImpl->next_sibling( pImpl->name(), pImpl->name_size() ) );
}

const XmlNode* XmlNode::FindPrev() const
{
    const detail::NodeImpl* pImpl = detail::ToImpl( this );
    return detail::ToNode( pImpl->previous_sibling( pImpl->name(), pImpl->name_size() ) );
}

nn::util::string_view XmlNode::FindAttribute( const nn::util::string_view& name ) const
{
    const xml_attribute<>* pAttr = detail::ToImpl( this )->first_attribute(
        name.data(), static_cast< size_t >( name.size() ) );
    if ( pAttr )
    {
        return nn::util::string_view( pAttr->value(), static_cast< int >( pAttr->value_size() ) );
    }
    else
    {
        return nn::util::string_view();
    }
}

nn::util::string_view XmlNode::GetText() const
{
    const detail::NodeImpl* pImpl = detail::ToImpl( this );
    return nn::util::string_view( pImpl->value(), static_cast< int >( pImpl->value_size() ) );
}

}} // namespace nn::utilTool
