﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <nn/nn_Assert.h>
#include <nn/utilTool/utilTool_StringUtility.h>

namespace nn {
    namespace utilTool {

        namespace {

            template<typename CharType> int CharToInteger(CharType c);

            template<>
            int CharToInteger(wchar_t c)
            {
                if (L'0' <= c && c <= L'9')
                {
                    return static_cast<int>(c - L'0');
                }
                if (L'A' <= c && c <= L'Z')
                {
                    return static_cast<int>(c - L'A') + 10;
                }
                if (L'a' <= c && c <= L'z')
                {
                    return static_cast<int>(c - L'a') + 10;
                }
                return -1;
            }

            template<>
            int CharToInteger(char c)
            {
                if ('0' <= c && c <= '9')
                {
                    return static_cast<int>(c - '0');
                }
                if ('A' <= c && c <= 'Z')
                {
                    return static_cast<int>(c - 'A') + 10;
                }
                if ('a' <= c && c <= 'z')
                {
                    return static_cast<int>(c - 'a') + 10;
                }
                return -1;
            }

            template<typename CharType> bool IsSpace(CharType c);

            template<>
            bool IsSpace(wchar_t c)
            {
                return c == L' ' || c == L'\t' || c == L'\r' || c == L'\n';
            }

            template<>
            bool IsSpace(char c)
            {
                return c == ' ' || c == '\t' || c == '\r' || c == '\n';
            }

            template<typename Char> struct Character;

            template<>
            struct Character<wchar_t>
            {
                static const wchar_t Plus = L'+';
                static const wchar_t Minus = L'-';
                static const wchar_t C0 = L'0';
                static const wchar_t Cx = L'x';
                static const wchar_t CX = L'X';
            };

            template<>
            struct Character<char>
            {
                static const char Plus = '+';
                static const char Minus = '-';
                static const char C0 = '0';
                static const char Cx = 'x';
                static const char CX = 'X';
            };

            template <typename IntegerType> struct UnsignedType;

            template <> struct UnsignedType<int8_t> { typedef uint8_t Type; };
            template <> struct UnsignedType<uint8_t> { typedef uint8_t Type; };
            template <> struct UnsignedType<int16_t> { typedef uint16_t Type; };
            template <> struct UnsignedType<uint16_t> { typedef uint16_t Type; };
            template <> struct UnsignedType<int32_t> { typedef uint32_t Type; };
            template <> struct UnsignedType<uint32_t> { typedef uint32_t Type; };
            template <> struct UnsignedType<int64_t> { typedef uint64_t Type; };
            template <> struct UnsignedType<uint64_t> { typedef uint64_t Type; };

            template <typename IntegerType> struct SignedType;

            template <> struct SignedType<int8_t> { typedef int8_t Type; };
            template <> struct SignedType<uint8_t> { typedef int8_t Type; };
            template <> struct SignedType<int16_t> { typedef int16_t Type; };
            template <> struct SignedType<uint16_t> { typedef int16_t Type; };
            template <> struct SignedType<int32_t> { typedef int32_t Type; };
            template <> struct SignedType<uint32_t> { typedef int32_t Type; };
            template <> struct SignedType<int64_t> { typedef int64_t Type; };
            template <> struct SignedType<uint64_t> { typedef int64_t Type; };

            template <typename Char, typename Integer>
            Integer TStrToInteger(const Char *nptr, const Char **endptr, int32_t base);

            template <typename Char, typename Integer>
            Integer TStrToInteger(const Char *nptr, const Char **endptr, int32_t base)
            {
                const Char *s = nptr;
                Char c;
                typename UnsignedType<Integer>::Type value = 0;
                bool negative = false;
                bool errorOccured = false;

                c = *s++;
                while (IsSpace(c)) c = *s++;

                if (c == Character<Char>::Minus)
                {
                    negative = true;
                    c = *s++;
                }
                else
                {
                    negative = false;
                    if (c == Character<Char>::Plus)
                    {
                        c = *s++;
                    }
                }

                if ((base == 0 || base == 16) &&
                    c == Character<Char>::C0 &&
                    (*s == Character<Char>::Cx ||
                    *s == Character<Char>::CX))
                {
                    c = s[1];
                    s += 2;
                    base = 16;
                }
                if (base == 0)
                {
                    if (c == Character<Char>::C0)
                    {
                        base = 8;
                    }
                    else
                    {
                        base = 10;
                    }
                }
                if (base < 2 || 36 < base)
                {
                    return 0;
                }

                typename UnsignedType<Integer>::Type limit = static_cast<typename UnsignedType<Integer>::Type>(std::numeric_limits<Integer>::max());

                if (negative && std::numeric_limits<Integer>::is_signed)
                {
                    limit += 1;
                }

                typename UnsignedType<Integer>::Type maxValue = static_cast<typename UnsignedType<Integer>::Type>(limit / base);
                typename UnsignedType<Integer>::Type maxDigit = limit % base;

                for (;; c = *s++)
                {
                    int ci = CharToInteger(c);
                    if (!(0 <= ci && ci < base))
                    {
                        break;
                    }
                    typename UnsignedType<Integer>::Type cvalue = static_cast<typename UnsignedType<Integer>::Type>(ci);

                    //NN_LOG_FORCE("ret: %lld, cval: %lld", (int64_t)value, (int64_t)cvalue );

                    if (!errorOccured && (value < maxValue || (maxValue == value && cvalue <= maxDigit)))
                    {
                        value *= static_cast<typename UnsignedType<Integer>::Type>(base);
                        value += cvalue;
                    }
                    else
                    {
                        errorOccured = true;
                        value = limit;
                    }
                }

                //NN_LOG_FORCE("ret: %lld", (int64_t)value );

                if (negative)
                {
                    value = static_cast<typename UnsignedType<Integer>::Type>(
                        -static_cast<typename SignedType<Integer>::Type>(
                        value));
                }

                if (endptr != NULL)
                {
                    *endptr = s - 1;
                }

                return static_cast<Integer>(value);
            }

            template  int8_t TStrToInteger<char, int8_t>(const char *nptr, const char **endptr, int32_t base);
            template int16_t TStrToInteger<char, int16_t>(const char *nptr, const char **endptr, int32_t base);
            template int32_t TStrToInteger<char, int32_t>(const char *nptr, const char **endptr, int32_t base);
            template int64_t TStrToInteger<char, int64_t>(const char *nptr, const char **endptr, int32_t base);
            template  uint8_t TStrToInteger<char, uint8_t>(const char *nptr, const char **endptr, int32_t base);
            template uint16_t TStrToInteger<char, uint16_t>(const char *nptr, const char **endptr, int32_t base);
            template uint32_t TStrToInteger<char, uint32_t>(const char *nptr, const char **endptr, int32_t base);
            template uint64_t TStrToInteger<char, uint64_t>(const char *nptr, const char **endptr, int32_t base);
            template  int8_t TStrToInteger<wchar_t, int8_t>(const wchar_t *nptr, const wchar_t **endptr, int32_t base);
            template int16_t TStrToInteger<wchar_t, int16_t>(const wchar_t *nptr, const wchar_t **endptr, int32_t base);
            template int32_t TStrToInteger<wchar_t, int32_t>(const wchar_t *nptr, const wchar_t **endptr, int32_t base);
            template int64_t TStrToInteger<wchar_t, int64_t>(const wchar_t *nptr, const wchar_t **endptr, int32_t base);
            template  uint8_t TStrToInteger<wchar_t, uint8_t>(const wchar_t *nptr, const wchar_t **endptr, int32_t base);
            template uint16_t TStrToInteger<wchar_t, uint16_t>(const wchar_t *nptr, const wchar_t **endptr, int32_t base);
            template uint32_t TStrToInteger<wchar_t, uint32_t>(const wchar_t *nptr, const wchar_t **endptr, int32_t base);
            template uint64_t TStrToInteger<wchar_t, uint64_t>(const wchar_t *nptr, const wchar_t **endptr, int32_t base);

        }

        int8_t StrToInt8(const char *nptr, const char **endptr, int32_t base)
        {
            return TStrToInteger<char, int8_t>(nptr, endptr, base);
        }
        int16_t StrToInt16(const char *nptr, const char **endptr, int32_t base)
        {
            return TStrToInteger<char, int16_t>(nptr, endptr, base);
        }
        int32_t StrToInt32(const char *nptr, const char **endptr, int32_t base)
        {
            return TStrToInteger<char, int32_t>(nptr, endptr, base);
        }
        int64_t StrToInt64(const char *nptr, const char **endptr, int32_t base)
        {
            return TStrToInteger<char, int64_t>(nptr, endptr, base);
        }
        int8_t WStrToInt8(const wchar_t *nptr, const wchar_t **endptr, int32_t base)
        {
            return TStrToInteger<wchar_t, int8_t>(nptr, endptr, base);
        }
        int16_t WStrToInt16(const wchar_t *nptr, const wchar_t **endptr, int32_t base)
        {
            return TStrToInteger<wchar_t, int16_t>(nptr, endptr, base);
        }
        int32_t WStrToInt32(const wchar_t *nptr, const wchar_t **endptr, int32_t base)
        {
            return TStrToInteger<wchar_t, int32_t>(nptr, endptr, base);
        }
        int64_t WStrToInt64(const wchar_t *nptr, const wchar_t **endptr, int32_t base)
        {
            return TStrToInteger<wchar_t, int64_t>(nptr, endptr, base);
        }
        uint8_t StrToUInt8(const char *nptr, const char **endptr, int32_t base)
        {
            return TStrToInteger<char, uint8_t>(nptr, endptr, base);
        }
        uint16_t StrToUInt16(const char *nptr, const char **endptr, int32_t base)
        {
            return TStrToInteger<char, uint16_t>(nptr, endptr, base);
        }
        uint32_t StrToUInt32(const char *nptr, const char **endptr, int32_t base)
        {
            return TStrToInteger<char, uint32_t>(nptr, endptr, base);
        }
        uint64_t StrToUInt64(const char *nptr, const char **endptr, int32_t base)
        {
            return TStrToInteger<char, uint64_t>(nptr, endptr, base);
        }
        uint8_t WStrToUInt8(const wchar_t *nptr, const wchar_t **endptr, int32_t base)
        {
            return TStrToInteger<wchar_t, uint8_t>(nptr, endptr, base);
        }
        uint16_t WStrToUInt16(const wchar_t *nptr, const wchar_t **endptr, int32_t base)
        {
            return TStrToInteger<wchar_t, uint16_t>(nptr, endptr, base);
        }
        uint32_t WStrToUInt32(const wchar_t *nptr, const wchar_t **endptr, int32_t base)
        {
            return TStrToInteger<wchar_t, uint32_t>(nptr, endptr, base);
        }
        uint64_t WStrToUInt64(const wchar_t *nptr, const wchar_t **endptr, int32_t base)
        {
            return TStrToInteger<wchar_t, uint64_t>(nptr, endptr, base);
        }
    }
}
