﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_Utf8StringUtil.h>
#include <nn/nn_SdkAssert.h>

#define USING_CODE_POINT_BYTE_LENGTH_TABLE

namespace nn { namespace util {

namespace
{
    // RFC 3629 の 4. Syntax of UTF-8 Byte Sequences を参照。

#if defined (USING_CODE_POINT_BYTE_LENGTH_TABLE)

    const uint8_t CodePointByteLengthTable[256] =
    {
        // 0x00 - 0x7F
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        // 0x80 - 0xBF は 2 バイト目以降専用
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        // 0xC0 - 0xC1 は冗長判定により不正, 0xC2 - 0xDF
        0, 0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        // 0xE0 - 0xEF, 0xF0 - 0xF4, 0xF5 - 0xFF は範囲外判定により不正
        3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
    };

#endif

    inline size_t GetCodePointByteLength(Bit8 c) NN_NOEXCEPT
    {
#if defined (USING_CODE_POINT_BYTE_LENGTH_TABLE)

        return CodePointByteLengthTable[c];

#else

        // 00000000-0000007F | 0xxxxxxx
        if ((c & 0x80) == 0)
        {
            return 1;
        }
        // 00000080-000007FF | 110xxxxx 10xxxxxx
        if ((c & 0xE0) == 0xC0)
        {
            return 2;
        }
        // 00000800-0000FFFF | 1110xxxx 10xxxxxx 10xxxxxx
        if ((c & 0xF0) == 0xE0)
        {
            return 3;
        }
        // 00010000-0010FFFF | 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
        if ((c & 0xF8) == 0xF0)
        {
            return 4;
        }

        // 不正なシーケンス、もしくは、5 バイト以上である。
        return 0;

#endif
    }

    inline bool IsValidTail(Bit8 c) NN_NOEXCEPT
    {
        // 2 バイト目以降は (10xxxxxx) である。
        return ((c & 0xC0) == 0x80);
    }

    inline bool VerifyCode(const Bit8* code, size_t byteLength) NN_NOEXCEPT
    {
        switch (byteLength)
        {
        case 1:
            break;
        case 2:
            {
#if !defined (USING_CODE_POINT_BYTE_LENGTH_TABLE)
                // UTF8-2 = %xC2-DF UTF8-tail → 冗長判定
                if ((code[0] & 0x1E) == 0)
                {
                    // 2 バイト文字 (11 ビット長) なのに 7 ビットに収まるため、冗長なコードと判定する。
                    return false;
                }
#endif
                if (!IsValidTail(code[1]))
                {
                    return false;
                }
            }
            break;
        case 3:
            {
                // UTF8-3 = %xE0 %xA0-BF UTF8-tail → 冗長判定
                if (code[0] == 0xE0 && (code[1] & 0x20) == 0)
                {
                    // 3 バイト文字 (16 ビット長) なのに 11 ビットに収まるため、冗長なコードと判定する。
                    return false;
                }
                // UTF8-3 = %xED %x80-9F UTF8-tail → サロゲート領域 (U+D800 ～ U+DFFF) 判定
                if (code[0] == 0xED && (code[1] & 0x20) != 0)
                {
                    // サロゲート領域の使用禁止。
                    return false;
                }
                if (!IsValidTail(code[1]) || !IsValidTail(code[2]))
                {
                    return false;
                }
            }
            break;
        case 4:
            {
                // UTF8-4 = %xF0 %x90-BF 2( UTF8-tail ) → 冗長判定
                if (code[0] == 0xF0 && (code[1] & 0x30) == 0)
                {
                    // 4 バイト文字 (21 ビット長) なのに 16 ビットに収まるため、冗長なコードと判定する。
                    return false;
                }
                // UTF8-4 = %xF4 %x80-8F 2( UTF8-tail ) → 範囲外判定
                if (code[0] == 0xF4 && (code[1] & 0x30) != 0)
                {
                    // 0x10FFFF より大きい場合、利用可能範囲外。
                    return false;
                }
#if !defined (USING_CODE_POINT_BYTE_LENGTH_TABLE)
                if (code[0] > 0xF4)
                {
                    // 0x10FFFF より大きい場合、利用可能範囲外。
                    return false;
                }
#endif
                if (!IsValidTail(code[1]) || !IsValidTail(code[2]) || !IsValidTail(code[3]))
                {
                    return false;
                }
            }
            break;
        default:
            return false;
        }

        return true;
    }
}

bool VerifyUtf8String(const char* string, size_t byteLength) NN_NOEXCEPT
{
    return (GetCodePointCountOfUtf8String(string, byteLength) != -1);
}

int GetCodePointCountOfUtf8String(const char* string, size_t byteLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(string);
    NN_SDK_REQUIRES_GREATER(byteLength, 0u);

    int codePointCount = 0;

    while (byteLength > 0)
    {
        const Bit8* code = reinterpret_cast<const Bit8*>(string);

        size_t codeByteLength = GetCodePointByteLength(code[0]);

        if (codeByteLength > byteLength)
        {
            return -1;
        }
        if (!VerifyCode(code, codeByteLength))
        {
            return -1;
        }

        string += codeByteLength;
        byteLength -= codeByteLength;

        codePointCount++;
    }

    return codePointCount;
}

size_t CopyUtf8String(char* output, size_t outputSize, const char* input, size_t inputByteLength, int maxCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(output);
    NN_SDK_REQUIRES_GREATER(outputSize, 0u);
    NN_SDK_REQUIRES_NOT_NULL(input);
    NN_SDK_REQUIRES_GREATER(inputByteLength, 0u);
    NN_SDK_REQUIRES_GREATER(maxCount, 0);

    const char* p = input;

    while (inputByteLength > 0)
    {
        const Bit8* code = reinterpret_cast<const Bit8*>(p);

        size_t codeByteLength = GetCodePointByteLength(code[0]);

        if (codeByteLength > inputByteLength)
        {
            break;
        }
        if (!VerifyCode(code, codeByteLength))
        {
            break;
        }

        size_t length = static_cast<size_t>(p + codeByteLength - input);

        if (length + 1 > outputSize)
        {
            break;
        }

        p += codeByteLength;
        inputByteLength -= codeByteLength;

        if (--maxCount == 0)
        {
            break;
        }
    }

    size_t byteLength = static_cast<size_t>(p - input);

    NN_SDK_ASSERT(byteLength + 1 <= outputSize);

    if (byteLength > 0)
    {
        std::memcpy(output, input, byteLength);
    }
    output[byteLength] = '\0';

    return byteLength;
}

}}
