﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <climits>

#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Abort.h>

#include <nn/util/util_StreamingDecompression.h>

#include "zutil.h"
#include "zlib.h"
#include "inftrees.h"
#include "inflate.h"

#include "detail/util_CompressionUtillity.h"

namespace {

// Streamig 伸長のエラーを表示する関数
void DisplayErr(int errCode)  NN_NOEXCEPT
{
    switch(errCode)
    {
    case Z_STREAM_ERROR:
        // Z_STREAM_ERROR はユーザーがバッファに NULL を指定した時に返る(事前条件違反)ので、
        // ASSERT で止める。
        NN_SDK_ASSERT(false, "ERROR (%d) : Source Buffer or Destination Buffer is inconsistent.\n", errCode);
        break;

    case Z_VERSION_ERROR:
        // サポートされていないバージョンのデータが入ったときに返るエラー
        NN_SDK_LOG("ERROR (%d) : This library can not support this zlib version file.\n", errCode);
        break;

    case Z_MEM_ERROR:
        // ワークバッファ不足で返る。
        // NN_SDK_REQUIRES で事前にチェックしているはずなので、DisplayErr が返るタイミングで
        // これが返る = ライブラリのバグなので想定外のメッセージ。
        NN_SDK_ASSERT(false, "ERROR (%d) : Unexpected Error\n", errCode);
        break;

    case Z_BUF_ERROR:
        // 最終出力時に入力が足りていないもしくは出力バッファが足りていないときに返るエラー
        // 本ライブラリのストリーミング伸長では Z_STREAM_END の直後に Z_FINISH を指定して
        // 終了しているため、このパターンは起きえない。
        // また、inflate 時に SrcSize または DstSize に 0 を入れた場合、このエラーが返ってくる。
        // これは入力データ(各バッファ)に問題があるので、ログを出す。
        NN_SDK_LOG("ERROR (%d) : Src Buffer Size or Dst Buffer Size is 0.\n", errCode);
        break;

    case Z_DATA_ERROR:
        // 入力データが壊れている場合のエラー
        NN_SDK_LOG("ERROR (%d) : the input data was corrupted.\n", errCode);
        break;

    case Z_NEED_DICT:
        // 今回は DICT の指定をしていないため、このエラーは返らないはず
        NN_SDK_ASSERT(false, "ERROR (%d) : DICT is inconsistent.\n", errCode);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
}

// 内部で使用する Context の実態
class StreamingDecompressContextImpl
{
public:

    // Context の初期化処理
    void Initialize()
    {
        m_IsStreamEnd      = false;

        // zlib に渡す stream の設定を初期化
        m_Stream.next_in   = NULL;
        m_Stream.avail_in  = 0;
        m_Stream.next_out  = NULL;
        m_Stream.avail_out = 0;

        m_Stream.zalloc    = nn::util::detail::ZlibAllocator::Allocate;
        m_Stream.zfree     = nn::util::detail::ZlibAllocator::Free;

        // placement new を使って ZlibAllocator として初期化した結果をセットする。
        m_Stream.opaque    = new (m_AllocatorBuffer) nn::util::detail::ZlibAllocator(&m_WorkBuffer, sizeof(m_WorkBuffer));
    }

    // ストリームに Src と Dst を設定する関数
    // zlib 用の stream 構造体に渡すため、64bit 環境であっても一度に 4GB 以上のサイズは扱えない
    void SetStream(void* pDst, size_t dstSize, const void* pSrc, size_t srcSize)
    {
        NN_SDK_ASSERT(srcSize <= UINT_MAX);
        NN_SDK_ASSERT(dstSize <= UINT_MAX);
        m_Stream.next_in   = reinterpret_cast<nnutilZlib_Bytef*>(const_cast<void*>(pSrc));
        m_Stream.avail_in  = static_cast<nnutilZlib_uInt>(srcSize);
        m_Stream.next_out  = reinterpret_cast<nnutilZlib_Bytef*>(pDst);
        m_Stream.avail_out = static_cast<nnutilZlib_uInt>(dstSize);
    }


    z_stream* GetStream()
    {
        return &m_Stream;
    }

    void SetStreamEnd(bool isStreamEnd)
    {
        m_IsStreamEnd = isStreamEnd;
    }

    bool IsStreamEnd()
    {
        return m_IsStreamEnd;
    }

private:

    static const size_t WorkSize = sizeof(inflate_state)  + (1 << MAX_WBITS);

    nn::Bit8                         m_AllocatorBuffer[sizeof(nn::util::detail::ZlibAllocator)];
    z_stream                         m_Stream;
    bool                             m_IsStreamEnd;
    NN_ALIGNAS(4) nn::Bit8           m_WorkBuffer[WorkSize];
};

// static_assert で context のサイズチェック
NN_STATIC_ASSERT( sizeof(nn::util::StreamingDecompressZlibContext) >= sizeof(StreamingDecompressContextImpl));


// Context 初期化の共通処理
void InitializeStreamingDecompressContextCommon(nn::util::StreamingDecompressZlibContext* pContext, int wbits) NN_NOEXCEPT
{
    // placement new を使って、コンテキストを内部で使用する形に割り当てる
    StreamingDecompressContextImpl* pContextImpl = new (pContext->_context) StreamingDecompressContextImpl;

    // Context の初期化処理
    pContextImpl->Initialize();

    // zlib の初期化処理
    int err = inflateInit2(pContextImpl->GetStream(), wbits);
    if(err != Z_OK)
    {
        // ここで inflateIniti2 から返りえる値は、
        // Z_OK, Z_STREAM_ERROR, Z_VERSION_ERROR, Z_MEM_ERROR の四種類
        // 実装ミス以外ではどれも返らないはず。
        NN_ABORT("ERROR (%d) : Unexpected Error\n", err);
    }
}


// StreamingDecompress の共通処理
bool StreamingDecompressCommon(size_t* pOutDstSize, size_t* pOutConsumedSrcSize,
                               void* pDst,          size_t dstSize,
                               const void* pSrc,    size_t srcSize,
                               nn::util::StreamingDecompressZlibContext* pContext) NN_NOEXCEPT
{

    // 内部で使用するコンテキストへ変換
    StreamingDecompressContextImpl* pContextImpl = reinterpret_cast<StreamingDecompressContextImpl*>(pContext->_context);

    // ストリーミング処理が終了していた場合は入出力を両方 0 にして true を返すことで、
    // 処理の完了を伝える。
    if(pContextImpl->IsStreamEnd())
    {
        *pOutDstSize = 0;
        *pOutConsumedSrcSize = 0;
        return true;
    }

    // ストリームに Src と Dst をセットする
    pContextImpl->SetStream(pDst, dstSize, pSrc, srcSize);

    // 伸長前の入出力バッファの残りのサイズを保持しておく
    const uint32_t availInBefore  = static_cast<uint32_t>(srcSize);
    const uint32_t availOutBefore = static_cast<uint32_t>(dstSize);

    // inflate(伸長) を実行
    int err = nnutilZlib_inflate(pContextImpl->GetStream(), Z_NO_FLUSH);
    if ((err != Z_OK) && (err != Z_STREAM_END))
    {
        // エラー処理
        // ここで inflate から返りえる値は、
        // Z_STREAM_END, Z_NEED_DICT, Z_DATA_ERROR, Z_STREAM_ERROR, Z_MEM_ERROR, Z_BUF_ERRORの六種類
        DisplayErr(err);
        return false;
    }
    else if( err == Z_STREAM_END )
    {
        // すべての伸長が終了したので、Z_FINISH を指定して伸長を終了させる
        pContextImpl->SetStreamEnd(true);
        err = nnutilZlib_inflate(pContextImpl->GetStream(), Z_FINISH);
        if(err != Z_STREAM_END)
        {
            // ここで inflate から返りえる値は、
            // Z_STREAM_END, Z_NEED_DICT, Z_DATA_ERROR, Z_STREAM_ERROR, Z_MEM_ERROR, Z_BUF_ERRORの六種類
            DisplayErr(err);
            return false;
        }

        // inflate の終了処理
        err = nnutilZlib_inflateEnd(pContextImpl->GetStream());
        if(err != Z_OK)
        {
            // ここで inflateEnd から返りえる値は、
            // Z_OK, Z_STREAM_ERROR の二種類
            DisplayErr(err);
            return false;
        }
    }

    // 出力サイズ = 処理前の出力先バッファのサイズ - 今の出力先バッファのサイズ
    *pOutDstSize = availOutBefore - pContextImpl->GetStream()->avail_out;

    // 消費した入力 = 処理前の入力バッファのサイズ - 今の入力バッファのサイズ
    *pOutConsumedSrcSize = availInBefore - pContextImpl->GetStream()->avail_in;

    return true;
}


}



namespace nn {
namespace util {


void InitializeStreamingDecompressZlibContext(StreamingDecompressZlibContext* pContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pContext);
    NN_SDK_REQUIRES(sizeof(pContext->_context) == sizeof(StreamingDecompressZlibContext));
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(pContext) % NN_ALIGNOF(StreamingDecompressContextImpl) == 0);

    InitializeStreamingDecompressContextCommon(pContext, detail::ZlibFormatWbits);
}

void InitializeStreamingDecompressGzipContext(StreamingDecompressGzipContext* pContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pContext);
    NN_SDK_REQUIRES(sizeof(pContext->_context) == sizeof(StreamingDecompressGzipContext));
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(pContext) % NN_ALIGNOF(StreamingDecompressContextImpl) == 0);

    InitializeStreamingDecompressContextCommon(pContext, detail::GzipFormatWbits);
}

void InitializeStreamingDecompressDeflateContext(StreamingDecompressDeflateContext* pContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pContext);
    NN_SDK_REQUIRES(sizeof(pContext->_context) == sizeof(StreamingDecompressZlibContext));
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(pContext) % NN_ALIGNOF(StreamingDecompressContextImpl) == 0);

    InitializeStreamingDecompressContextCommon(pContext, detail::RawDeflateFormatWbits);
}

bool StreamingDecompressZlib(size_t* pOutDstSize, size_t* pOutConsumedSrcSize,
                             void* pDst,          size_t dstSize,
                             const void* pSrc,    size_t srcSize,
                             StreamingDecompressZlibContext* pContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_REQUIRES(sizeof(pContext->_context) == sizeof(StreamingDecompressZlibContext));

    return StreamingDecompressCommon(pOutDstSize,  pOutConsumedSrcSize, pDst, dstSize, pSrc, srcSize, pContext);
}

bool StreamingDecompressGzip(size_t* pOutDstSize, size_t* pOutConsumedSrcSize,
                             void* pDst,          size_t dstSize,
                             const void* pSrc,    size_t srcSize,
                             StreamingDecompressZlibContext* pContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_REQUIRES(sizeof(pContext->_context) == sizeof(StreamingDecompressZlibContext));

    return StreamingDecompressCommon(pOutDstSize,  pOutConsumedSrcSize, pDst, dstSize, pSrc, srcSize, pContext);
}

bool StreamingDecompressDeflate(size_t* pOutDstSize, size_t* pOutConsumedSrcSize,
                             void* pDst,          size_t dstSize,
                             const void* pSrc,    size_t srcSize,
                             StreamingDecompressZlibContext* pContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_REQUIRES(sizeof(pContext->_context) == sizeof(StreamingDecompressZlibContext));

    return StreamingDecompressCommon(pOutDstSize,  pOutConsumedSrcSize, pDst, dstSize, pSrc, srcSize, pContext);
}

} //util
} //nn
