﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_Country.h>

namespace nn { namespace util {

namespace {

struct CountryDefinition
{
    char    code[2];
    Country country;
};

// http://www.iso.org/iso/home/standards/country_codes.htm
CountryDefinition g_Table[] = {
    {{'A', 'D'}, Country_Ad}, // AD, 20, Andorra
    {{'A', 'E'}, Country_Ae}, // AE, 784, United Arab Emirates (the)
    {{'A', 'F'}, Country_Af}, // AF, 4, Afghanistan
    {{'A', 'G'}, Country_Ag}, // AG, 28, Antigua and Barbuda
    {{'A', 'I'}, Country_Ai}, // AI, 660, Anguilla
    {{'A', 'L'}, Country_Al}, // AL, 8, Albania
    {{'A', 'M'}, Country_Am}, // AM, 51, Armenia
    {{'A', 'O'}, Country_Ao}, // AO, 24, Angola
    {{'A', 'Q'}, Country_Aq}, // AQ, 10, Antarctica
    {{'A', 'R'}, Country_Ar}, // AR, 32, Argentina
    {{'A', 'S'}, Country_As}, // AS, 16, American Samoa
    {{'A', 'T'}, Country_At}, // AT, 40, Austria
    {{'A', 'U'}, Country_Au}, // AU, 36, Australia
    {{'A', 'W'}, Country_Aw}, // AW, 533, Aruba
    {{'A', 'X'}, Country_Ax}, // AX, 248, Åland Islands
    {{'A', 'Z'}, Country_Az}, // AZ, 31, Azerbaijan
    {{'B', 'A'}, Country_Ba}, // BA, 70, Bosnia and Herzegovina
    {{'B', 'B'}, Country_Bb}, // BB, 52, Barbados
    {{'B', 'D'}, Country_Bd}, // BD, 50, Bangladesh
    {{'B', 'E'}, Country_Be}, // BE, 56, Belgium
    {{'B', 'F'}, Country_Bf}, // BF, 854, Burkina Faso
    {{'B', 'G'}, Country_Bg}, // BG, 100, Bulgaria
    {{'B', 'H'}, Country_Bh}, // BH, 48, Bahrain
    {{'B', 'I'}, Country_Bi}, // BI, 108, Burundi
    {{'B', 'J'}, Country_Bj}, // BJ, 204, Benin
    {{'B', 'L'}, Country_Bl}, // BL, 652, Saint Barthélemy
    {{'B', 'M'}, Country_Bm}, // BM, 60, Bermuda
    {{'B', 'N'}, Country_Bn}, // BN, 96, Brunei Darussalam
    {{'B', 'O'}, Country_Bo}, // BO, 68, Bolivia (Plurinational State of)
    {{'B', 'Q'}, Country_Bq}, // BQ, 535, Bonaire, Sint Eustatius and Saba
    {{'B', 'R'}, Country_Br}, // BR, 76, Brazil
    {{'B', 'S'}, Country_Bs}, // BS, 44, Bahamas (the)
    {{'B', 'T'}, Country_Bt}, // BT, 64, Bhutan
    {{'B', 'V'}, Country_Bv}, // BV, 74, Bouvet Island
    {{'B', 'W'}, Country_Bw}, // BW, 72, Botswana
    {{'B', 'Y'}, Country_By}, // BY, 112, Belarus
    {{'B', 'Z'}, Country_Bz}, // BZ, 84, Belize
    {{'C', 'A'}, Country_Ca}, // CA, 124, Canada
    {{'C', 'C'}, Country_Cc}, // CC, 166, Cocos (Keeling) Islands (the)
    {{'C', 'D'}, Country_Cd}, // CD, 180, Congo (the Democratic Republic of the)
    {{'C', 'F'}, Country_Cf}, // CF, 140, Central African Republic (the)
    {{'C', 'G'}, Country_Cg}, // CG, 178, Congo (the)
    {{'C', 'H'}, Country_Ch}, // CH, 756, Switzerland
    {{'C', 'I'}, Country_Ci}, // CI, 384, Côte d'Ivoire
    {{'C', 'K'}, Country_Ck}, // CK, 184, Cook Islands (the)
    {{'C', 'L'}, Country_Cl}, // CL, 152, Chile
    {{'C', 'M'}, Country_Cm}, // CM, 120, Cameroon
    {{'C', 'N'}, Country_Cn}, // CN, 156, China
    {{'C', 'O'}, Country_Co}, // CO, 170, Colombia
    {{'C', 'R'}, Country_Cr}, // CR, 188, Costa Rica
    {{'C', 'U'}, Country_Cu}, // CU, 192, Cuba
    {{'C', 'V'}, Country_Cv}, // CV, 132, Cabo Verde
    {{'C', 'W'}, Country_Cw}, // CW, 531, Curaçao
    {{'C', 'X'}, Country_Cx}, // CX, 162, Christmas Island
    {{'C', 'Y'}, Country_Cy}, // CY, 196, Cyprus
    {{'C', 'Z'}, Country_Cz}, // CZ, 203, Czechia
    {{'D', 'E'}, Country_De}, // DE, 276, Germany
    {{'D', 'J'}, Country_Dj}, // DJ, 262, Djibouti
    {{'D', 'K'}, Country_Dk}, // DK, 208, Denmark
    {{'D', 'M'}, Country_Dm}, // DM, 212, Dominica
    {{'D', 'O'}, Country_Do}, // DO, 214, Dominican Republic (the)
    {{'D', 'Z'}, Country_Dz}, // DZ, 12, Algeria
    {{'E', 'C'}, Country_Ec}, // EC, 218, Ecuador
    {{'E', 'E'}, Country_Ee}, // EE, 233, Estonia
    {{'E', 'G'}, Country_Eg}, // EG, 818, Egypt
    {{'E', 'H'}, Country_Eh}, // EH, 732, Western Sahara*
    {{'E', 'R'}, Country_Er}, // ER, 232, Eritrea
    {{'E', 'S'}, Country_Es}, // ES, 724, Spain
    {{'E', 'T'}, Country_Et}, // ET, 231, Ethiopia
    {{'F', 'I'}, Country_Fi}, // FI, 246, Finland
    {{'F', 'J'}, Country_Fj}, // FJ, 242, Fiji
    {{'F', 'K'}, Country_Fk}, // FK, 238, Falkland Islands (the) [Malvinas]
    {{'F', 'M'}, Country_Fm}, // FM, 583, Micronesia (Federated States of)
    {{'F', 'O'}, Country_Fo}, // FO, 234, Faroe Islands (the)
    {{'F', 'R'}, Country_Fr}, // FR, 250, France
    {{'G', 'A'}, Country_Ga}, // GA, 266, Gabon
    {{'G', 'B'}, Country_Gb}, // GB, 826, United Kingdom of Great Britain and Northern Ireland (the)
    {{'G', 'D'}, Country_Gd}, // GD, 308, Grenada
    {{'G', 'E'}, Country_Ge}, // GE, 268, Georgia
    {{'G', 'F'}, Country_Gf}, // GF, 254, French Guiana
    {{'G', 'G'}, Country_Gg}, // GG, 831, Guernsey
    {{'G', 'H'}, Country_Gh}, // GH, 288, Ghana
    {{'G', 'I'}, Country_Gi}, // GI, 292, Gibraltar
    {{'G', 'L'}, Country_Gl}, // GL, 304, Greenland
    {{'G', 'M'}, Country_Gm}, // GM, 270, Gambia (the)
    {{'G', 'N'}, Country_Gn}, // GN, 324, Guinea
    {{'G', 'P'}, Country_Gp}, // GP, 312, Guadeloupe
    {{'G', 'Q'}, Country_Gq}, // GQ, 226, Equatorial Guinea
    {{'G', 'R'}, Country_Gr}, // GR, 300, Greece
    {{'G', 'S'}, Country_Gs}, // GS, 239, South Georgia and the South Sandwich Islands
    {{'G', 'T'}, Country_Gt}, // GT, 320, Guatemala
    {{'G', 'U'}, Country_Gu}, // GU, 316, Guam
    {{'G', 'W'}, Country_Gw}, // GW, 624, Guinea-Bissau
    {{'G', 'Y'}, Country_Gy}, // GY, 328, Guyana
    {{'H', 'K'}, Country_Hk}, // HK, 344, Hong Kong
    {{'H', 'M'}, Country_Hm}, // HM, 334, Heard Island and McDonald Islands
    {{'H', 'N'}, Country_Hn}, // HN, 340, Honduras
    {{'H', 'R'}, Country_Hr}, // HR, 191, Croatia
    {{'H', 'T'}, Country_Ht}, // HT, 332, Haiti
    {{'H', 'U'}, Country_Hu}, // HU, 348, Hungary
    {{'I', 'D'}, Country_Id}, // ID, 360, Indonesia
    {{'I', 'E'}, Country_Ie}, // IE, 372, Ireland
    {{'I', 'L'}, Country_Il}, // IL, 376, Israel
    {{'I', 'M'}, Country_Im}, // IM, 833, Isle of Man
    {{'I', 'N'}, Country_In}, // IN, 356, India
    {{'I', 'O'}, Country_Io}, // IO, 86, British Indian Ocean Territory (the)
    {{'I', 'Q'}, Country_Iq}, // IQ, 368, Iraq
    {{'I', 'R'}, Country_Ir}, // IR, 364, Iran (Islamic Republic of)
    {{'I', 'S'}, Country_Is}, // IS, 352, Iceland
    {{'I', 'T'}, Country_It}, // IT, 380, Italy
    {{'J', 'E'}, Country_Je}, // JE, 832, Jersey
    {{'J', 'M'}, Country_Jm}, // JM, 388, Jamaica
    {{'J', 'O'}, Country_Jo}, // JO, 400, Jordan
    {{'J', 'P'}, Country_Jp}, // JP, 392, Japan
    {{'K', 'E'}, Country_Ke}, // KE, 404, Kenya
    {{'K', 'G'}, Country_Kg}, // KG, 417, Kyrgyzstan
    {{'K', 'H'}, Country_Kh}, // KH, 116, Cambodia
    {{'K', 'I'}, Country_Ki}, // KI, 296, Kiribati
    {{'K', 'M'}, Country_Km}, // KM, 174, Comoros (the)
    {{'K', 'N'}, Country_Kn}, // KN, 659, Saint Kitts and Nevis
    {{'K', 'P'}, Country_Kp}, // KP, 408, Korea (the Democratic People's Republic of)
    {{'K', 'R'}, Country_Kr}, // KR, 410, Korea (the Republic of)
    {{'K', 'W'}, Country_Kw}, // KW, 414, Kuwait
    {{'K', 'Y'}, Country_Ky}, // KY, 136, Cayman Islands (the)
    {{'K', 'Z'}, Country_Kz}, // KZ, 398, Kazakhstan
    {{'L', 'A'}, Country_La}, // LA, 418, Lao People's Democratic Republic (the)
    {{'L', 'B'}, Country_Lb}, // LB, 422, Lebanon
    {{'L', 'C'}, Country_Lc}, // LC, 662, Saint Lucia
    {{'L', 'I'}, Country_Li}, // LI, 438, Liechtenstein
    {{'L', 'K'}, Country_Lk}, // LK, 144, Sri Lanka
    {{'L', 'R'}, Country_Lr}, // LR, 430, Liberia
    {{'L', 'S'}, Country_Ls}, // LS, 426, Lesotho
    {{'L', 'T'}, Country_Lt}, // LT, 440, Lithuania
    {{'L', 'U'}, Country_Lu}, // LU, 442, Luxembourg
    {{'L', 'V'}, Country_Lv}, // LV, 428, Latvia
    {{'L', 'Y'}, Country_Ly}, // LY, 434, Libya
    {{'M', 'A'}, Country_Ma}, // MA, 504, Morocco
    {{'M', 'C'}, Country_Mc}, // MC, 492, Monaco
    {{'M', 'D'}, Country_Md}, // MD, 498, Moldova (the Republic of)
    {{'M', 'E'}, Country_Me}, // ME, 499, Montenegro
    {{'M', 'F'}, Country_Mf}, // MF, 663, Saint Martin (French part)
    {{'M', 'G'}, Country_Mg}, // MG, 450, Madagascar
    {{'M', 'H'}, Country_Mh}, // MH, 584, Marshall Islands (the)
    {{'M', 'K'}, Country_Mk}, // MK, 807, Macedonia (the former Yugoslav Republic of)
    {{'M', 'L'}, Country_Ml}, // ML, 466, Mali
    {{'M', 'M'}, Country_Mm}, // MM, 104, Myanmar
    {{'M', 'N'}, Country_Mn}, // MN, 496, Mongolia
    {{'M', 'O'}, Country_Mo}, // MO, 446, Macao
    {{'M', 'P'}, Country_Mp}, // MP, 580, Northern Mariana Islands (the)
    {{'M', 'Q'}, Country_Mq}, // MQ, 474, Martinique
    {{'M', 'R'}, Country_Mr}, // MR, 478, Mauritania
    {{'M', 'S'}, Country_Ms}, // MS, 500, Montserrat
    {{'M', 'T'}, Country_Mt}, // MT, 470, Malta
    {{'M', 'U'}, Country_Mu}, // MU, 480, Mauritius
    {{'M', 'V'}, Country_Mv}, // MV, 462, Maldives
    {{'M', 'W'}, Country_Mw}, // MW, 454, Malawi
    {{'M', 'X'}, Country_Mx}, // MX, 484, Mexico
    {{'M', 'Y'}, Country_My}, // MY, 458, Malaysia
    {{'M', 'Z'}, Country_Mz}, // MZ, 508, Mozambique
    {{'N', 'A'}, Country_Na}, // NA, 516, Namibia
    {{'N', 'C'}, Country_Nc}, // NC, 540, New Caledonia
    {{'N', 'E'}, Country_Ne}, // NE, 562, Niger (the)
    {{'N', 'F'}, Country_Nf}, // NF, 574, Norfolk Island
    {{'N', 'G'}, Country_Ng}, // NG, 566, Nigeria
    {{'N', 'I'}, Country_Ni}, // NI, 558, Nicaragua
    {{'N', 'L'}, Country_Nl}, // NL, 528, Netherlands (the)
    {{'N', 'O'}, Country_No}, // NO, 578, Norway
    {{'N', 'P'}, Country_Np}, // NP, 524, Nepal
    {{'N', 'R'}, Country_Nr}, // NR, 520, Nauru
    {{'N', 'U'}, Country_Nu}, // NU, 570, Niue
    {{'N', 'Z'}, Country_Nz}, // NZ, 554, New Zealand
    {{'O', 'M'}, Country_Om}, // OM, 512, Oman
    {{'P', 'A'}, Country_Pa}, // PA, 591, Panama
    {{'P', 'E'}, Country_Pe}, // PE, 604, Peru
    {{'P', 'F'}, Country_Pf}, // PF, 258, French Polynesia
    {{'P', 'G'}, Country_Pg}, // PG, 598, Papua New Guinea
    {{'P', 'H'}, Country_Ph}, // PH, 608, Philippines (the)
    {{'P', 'K'}, Country_Pk}, // PK, 586, Pakistan
    {{'P', 'L'}, Country_Pl}, // PL, 616, Poland
    {{'P', 'M'}, Country_Pm}, // PM, 666, Saint Pierre and Miquelon
    {{'P', 'N'}, Country_Pn}, // PN, 612, Pitcairn
    {{'P', 'R'}, Country_Pr}, // PR, 630, Puerto Rico
    {{'P', 'S'}, Country_Ps}, // PS, 275, Palestine, State of
    {{'P', 'T'}, Country_Pt}, // PT, 620, Portugal
    {{'P', 'W'}, Country_Pw}, // PW, 585, Palau
    {{'P', 'Y'}, Country_Py}, // PY, 600, Paraguay
    {{'Q', 'A'}, Country_Qa}, // QA, 634, Qatar
    {{'R', 'E'}, Country_Re}, // RE, 638, Réunion
    {{'R', 'O'}, Country_Ro}, // RO, 642, Romania
    {{'R', 'S'}, Country_Rs}, // RS, 688, Serbia
    {{'R', 'U'}, Country_Ru}, // RU, 643, Russian Federation (the)
    {{'R', 'W'}, Country_Rw}, // RW, 646, Rwanda
    {{'S', 'A'}, Country_Sa}, // SA, 682, Saudi Arabia
    {{'S', 'B'}, Country_Sb}, // SB, 90, Solomon Islands
    {{'S', 'C'}, Country_Sc}, // SC, 690, Seychelles
    {{'S', 'D'}, Country_Sd}, // SD, 729, Sudan (the)
    {{'S', 'E'}, Country_Se}, // SE, 752, Sweden
    {{'S', 'G'}, Country_Sg}, // SG, 702, Singapore
    {{'S', 'H'}, Country_Sh}, // SH, 654, Saint Helena, Ascension and Tristan da Cunha
    {{'S', 'I'}, Country_Si}, // SI, 705, Slovenia
    {{'S', 'J'}, Country_Sj}, // SJ, 744, Svalbard and Jan Mayen
    {{'S', 'K'}, Country_Sk}, // SK, 703, Slovakia
    {{'S', 'L'}, Country_Sl}, // SL, 694, Sierra Leone
    {{'S', 'M'}, Country_Sm}, // SM, 674, San Marino
    {{'S', 'N'}, Country_Sn}, // SN, 686, Senegal
    {{'S', 'O'}, Country_So}, // SO, 706, Somalia
    {{'S', 'R'}, Country_Sr}, // SR, 740, Suriname
    {{'S', 'S'}, Country_Ss}, // SS, 728, South Sudan
    {{'S', 'T'}, Country_St}, // ST, 678, Sao Tome and Principe
    {{'S', 'V'}, Country_Sv}, // SV, 222, El Salvador
    {{'S', 'X'}, Country_Sx}, // SX, 534, Sint Maarten (Dutch part)
    {{'S', 'Y'}, Country_Sy}, // SY, 760, Syrian Arab Republic
    {{'S', 'Z'}, Country_Sz}, // SZ, 748, Swaziland
    {{'T', 'C'}, Country_Tc}, // TC, 796, Turks and Caicos Islands (the)
    {{'T', 'D'}, Country_Td}, // TD, 148, Chad
    {{'T', 'F'}, Country_Tf}, // TF, 260, French Southern Territories (the)
    {{'T', 'G'}, Country_Tg}, // TG, 768, Togo
    {{'T', 'H'}, Country_Th}, // TH, 764, Thailand
    {{'T', 'J'}, Country_Tj}, // TJ, 762, Tajikistan
    {{'T', 'K'}, Country_Tk}, // TK, 772, Tokelau
    {{'T', 'L'}, Country_Tl}, // TL, 626, Timor-Leste
    {{'T', 'M'}, Country_Tm}, // TM, 795, Turkmenistan
    {{'T', 'N'}, Country_Tn}, // TN, 788, Tunisia
    {{'T', 'O'}, Country_To}, // TO, 776, Tonga
    {{'T', 'R'}, Country_Tr}, // TR, 792, Turkey
    {{'T', 'T'}, Country_Tt}, // TT, 780, Trinidad and Tobago
    {{'T', 'V'}, Country_Tv}, // TV, 798, Tuvalu
    {{'T', 'W'}, Country_Tw}, // TW, 158, Taiwan (Province of China)
    {{'T', 'Z'}, Country_Tz}, // TZ, 834, Tanzania, United Republic of
    {{'U', 'A'}, Country_Ua}, // UA, 804, Ukraine
    {{'U', 'G'}, Country_Ug}, // UG, 800, Uganda
    {{'U', 'M'}, Country_Um}, // UM, 581, United States Minor Outlying Islands (the)
    {{'U', 'S'}, Country_Us}, // US, 840, United States of America (the)
    {{'U', 'Y'}, Country_Uy}, // UY, 858, Uruguay
    {{'U', 'Z'}, Country_Uz}, // UZ, 860, Uzbekistan
    {{'V', 'A'}, Country_Va}, // VA, 336, Holy See (the)
    {{'V', 'C'}, Country_Vc}, // VC, 670, Saint Vincent and the Grenadines
    {{'V', 'E'}, Country_Ve}, // VE, 862, Venezuela (Bolivarian Republic of)
    {{'V', 'G'}, Country_Vg}, // VG, 92, Virgin Islands (British)
    {{'V', 'I'}, Country_Vi}, // VI, 850, Virgin Islands (U.S.)
    {{'V', 'N'}, Country_Vn}, // VN, 704, Viet Nam
    {{'V', 'U'}, Country_Vu}, // VU, 548, Vanuatu
    {{'W', 'F'}, Country_Wf}, // WF, 876, Wallis and Futuna
    {{'W', 'S'}, Country_Ws}, // WS, 882, Samoa
    {{'Y', 'E'}, Country_Ye}, // YE, 887, Yemen
    {{'Y', 'T'}, Country_Yt}, // YT, 175, Mayotte
    {{'Z', 'A'}, Country_Za}, // ZA, 710, South Africa
    {{'Z', 'M'}, Country_Zm}, // ZM, 894, Zambia
    {{'Z', 'W'}, Country_Zw}, // ZW, 716, Zimbabwe
};

} // ~namespace nn::util::anonymous

#define FAILURE_UNLESS(expression) \
    do { auto _r = (expression); if (!_r) return false; } while (NN_STATIC_CONDITION(false))

bool GetCountryFromIso31661Alpha2(Country* pOutCountry, const char* code, size_t bytes) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCountry);
    NN_SDK_REQUIRES_NOT_NULL(code);
    NN_SDK_REQUIRES_LESS(strnlen(code, bytes), bytes);

    // 形式のチェック
    FAILURE_UNLESS(bytes >= 3);
    FAILURE_UNLESS(code[0] >= 'A' && code[0] <= 'Z');
    FAILURE_UNLESS(code[1] >= 'A' && code[1] <= 'Z');
    FAILURE_UNLESS(code[2] == '\0');

    // 定義のチェック
    // TODO: 遅かったら二分探索等して効率化する
    for (const auto& d : g_Table)
    {
        if (d.code[0] == code[0] && d.code[1] == code[1])
        {
            *pOutCountry = d.country;
            return true;
        }
    }
    return false;
}

#undef FAILURE_UNLESS

}} // ~namespace nn::util
