﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "zlib.h"

namespace nn {
namespace util {
namespace detail {

// Zlib 形式として出力/入力する際の Wbits の値
const int ZlibFormatWbits = MAX_WBITS;

// Gzip 形式として出力/入力する際の Wbits の値
// zlib.h で定義されているように zlib 形式に 16 をたす
const int GzipFormatWbits = MAX_WBITS + 16;

// Raw Deflate 形式として出力/入力する際の Wbits の値
// zlib.h で定義されているように zlib 形式に - を付ける
const int RawDeflateFormatWbits = - MAX_WBITS;


/**
* @brief zlib 用のアロケーター
*
* @details  zlib に渡すためのアロケーターです。上位から渡されたバッファの中から単純に先頭からサイズ分アロケートします。
*           また、Free は再度 Allocate されることはないという前提で何もしません。
*/
class ZlibAllocator
{
public:

    ZlibAllocator(void* p, size_t size) NN_NOEXCEPT :
        m_pHead(reinterpret_cast<Bit8*>(p)),
        m_pEnd (reinterpret_cast<Bit8*>(p) + size)
        {}

    static voidpf Allocate(voidpf pObj, uInt numItems, uInt size) NN_NOEXCEPT
    {
        return static_cast<ZlibAllocator*>(pObj)->Allocate(numItems * size);
    }

    static void Free(voidpf pObj, voidpf pMemory) NN_NOEXCEPT
    {
        // zlib の実装に依存し、
        // Free されたのち Allocate されることはないことを前提として
        // 何もしない
        NN_UNUSED(pObj);
        NN_UNUSED(pMemory);
    }

private:

    void* Allocate(size_t size) NN_NOEXCEPT
    {
        if( m_pHead + size <= m_pEnd )
        {
            void* pRet = m_pHead;
            m_pHead += size;

            return pRet;
        }
        else
        {
            return NULL;
        }
    }

private:
    Bit8*   m_pHead;
    Bit8*   m_pEnd;
};


} // detail
} // util
} // nn
