﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "image_JpegConfig.h"

/**
    @file Exif 情報の解析/構築のための諸々を定義するファイル
 */

namespace nn { namespace image { namespace detail {

/**
    @brief Exif の仕様上の数値
 */
enum ExifProperty : uint16_t
{
    ExifProperty_SizeMax = 0xFFFF - 0x0008,
    ExifProperty_TiffHeaderSize = 8u,
    ExifProperty_IfdSizeMin = 6u,
    ExifProperty_IfdTagSize = 12u,
};

/**
    @brief IfdTag 構造体の offset や count の特異値。
 */
enum IfdTagState: uint16_t
{
    IfdTagState_NotSpecified = 0x0000u, //< offset や count が指定されていないことを示す。
};

//! @brief 2バイト読み込む関数型
typedef uint16_t (*FunctionToRead2Bytes)(const uint8_t *data);
//! @brief 4バイト読み込む関数型
typedef uint32_t (*FunctionToRead4Bytes)(const uint8_t *data);
//! @brief 2バイト書き込む関数型
typedef void (*FunctionToWrite2Bytes)(uint8_t *data, uint16_t v);
//! @brief 4バイト書き込む関数型
typedef void (*FunctionToWrite4Bytes)(uint8_t *data, uint32_t v);

/**
    @brief      バイナリデータの読み書き用関数を保持する構造体
    @details    エンディアンによって分ける必要があるため関数ポインタ。
 */
struct BinaryIo
{
    FunctionToRead2Bytes read2;
    FunctionToRead4Bytes read4;
    FunctionToWrite2Bytes write2;
    FunctionToWrite4Bytes write4;
};

/**
    @brief      Exif情報バイナリ関連の要素をまとめて保持する構造体
 */
struct ExifBinary
{
    const uint8_t *data;
    uint16_t size;
    BinaryIo io;
};

/**
    @brief      エンディアン種別
 */
enum Endian : uint16_t
{
    Endian_Little = 0x4949, //< リトルエンディアン
    Endian_Big = 0x4D4D     //< ビッグエンディアン
};

/**
    @brief      Tiff ヘッダ
 */
struct TiffHeader
{
    Endian endian;          //< エンディアン
    BinaryIo io;            //< バイナリ読み書き用関数群
    uint16_t offset0thIfd;  //< 0th TIFF IFD へのオフセット
};

/**
    @brief      IFD ヘッダ
 */
struct IfdHeader
{
    uint16_t offset;        //< IFD へのオフセット
    uint16_t tagCount;      //< IFD 中のタグ数
    uint16_t nextOffset;    //< 次の IFD があれば、そこへのオフセット
};

/**
    @brief      IFD タグの種類
 */
enum IfdTagType
{
    /* Must have same value as specification. */
    IfdTagType_Byte = 1,               //< 任意の 8bit データ
    IfdTagType_Ascii = 2,              //< 0x00～0x7F までの ASCII
    IfdTagType_Short = 3,              //< 2byte 符号なし整数
    IfdTagType_Long = 4,               //< 4byte 符号なし整数
    IfdTagType_Rational = 5,           //< 符号なし分数 (Long/Long)
    IfdTagType_Undefined = 7,          //< 任意のバイナリ列
#ifdef NN_DETAIL_IMAGE_JPEG_CONFIG_EXIF_SIGNED_VALUE_SUPPORTED
    IfdTagType_SignedLong = 9,         //< 4byte 符号あり整数
    IfdTagType_SignedRational = 10,    //< 符号あり分数 (SignedLong/SignedLong)
#endif
    IfdTagType_ShortOrLong = 128,      //< ShortかLongのどちらでもよい場合 (ライブラリ実装の都合上追加したタイプ)
};

/**
    @brief IFD タグを表す
 */
struct IfdTag
{
    uint16_t id;            //< タグ ID
    IfdTagType type;        //< タグの種類
    uint16_t valueCount;    //< 値の数
    uint16_t valueOffset;   //< 値の格納位置までのオフセット
};

/**
    @brief 符号なし分数
 */
struct Rational
{
    uint32_t numerator;     //< 分子
    uint32_t denominator;   //< 分母
};

/**
    @brief 符号あり分数
 */
struct SignedRational
{
    int32_t numerator;      //< 分子
    int32_t denominator;    //< 分母
};

/**
    @brief Exif 情報バイナリ構築用の値付き IFD タグ
 */
struct ValuedIfdTag
{
    IfdTag tag; //< 対象となるタグ
    union
    {
        const uint8_t *vBytes;
        const char *vAscii;
        const uint16_t *vShort;
        const uint32_t *vLong;
        const Rational *vRational;
        const uint8_t *vUndefined;
#ifdef NN_DETAIL_IMAGE_JPEG_CONFIG_EXIF_SIGNED_VALUE_SUPPORTED
        const int32_t *vSignedLong;
        const SignedRational *vSignedRational;
#endif
    } value; //< タグに書き込む値
};

//! @name 既定の TIFF タグ
//! @{
const IfdTag IfdTagCompression =  {0x0103, IfdTagType_Short,    1,  IfdTagState_NotSpecified}; // 1st IFD only
const IfdTag IfdTagMake =         {0x010F, IfdTagType_Ascii, IfdTagState_NotSpecified, IfdTagState_NotSpecified}; // 0th IFD only
const IfdTag IfdTagModel =        {0x0110, IfdTagType_Ascii, IfdTagState_NotSpecified, IfdTagState_NotSpecified}; // 0th IFD only
const IfdTag IfdTagOrientaion =   {0x0112, IfdTagType_Short,    1,  IfdTagState_NotSpecified}; // 0th IFD only
const IfdTag IfdTagXResolution =  {0x011A, IfdTagType_Rational, 1,  IfdTagState_NotSpecified}; // both IFD
const IfdTag IfdTagYResolution =  {0x011B, IfdTagType_Rational, 1,  IfdTagState_NotSpecified}; // both IFD
const IfdTag IfdTagResolutionUnit ={0x0128, IfdTagType_Short,   1,  IfdTagState_NotSpecified}; // both IFD
const IfdTag IfdTagSoftware =     {0x0131, IfdTagType_Ascii, IfdTagState_NotSpecified, IfdTagState_NotSpecified}; // 0th IFD only
const IfdTag IfdTagDateTime =     {0x0132, IfdTagType_Ascii,    20, IfdTagState_NotSpecified}; // 0th IFD only
const IfdTag IfdTagJpegFormat =   {0x0201, IfdTagType_Long,     1,  IfdTagState_NotSpecified}; // 1st IFD
const IfdTag IfdTagJpegFormatLen ={0x0202, IfdTagType_Long,     1,  IfdTagState_NotSpecified}; // 1st IFD
const IfdTag IfdTagYccPosition =  {0x0213, IfdTagType_Short,    1,  IfdTagState_NotSpecified}; // 0th IFD only
const IfdTag IfdTagExifIfd =      {0x8769, IfdTagType_Long,     1,  IfdTagState_NotSpecified}; // 0th IFD only
const IfdTag IfdTagGpsIfd =       {0x8825, IfdTagType_Long,     1,  IfdTagState_NotSpecified}; // 0th IFD only
//! @}

//! @name 既定の Exif タグ
//! @{
const IfdTag IfdTagExifVersion =  {0x9000, IfdTagType_Undefined,    4,  IfdTagState_NotSpecified};
const IfdTag IfdTagCompConfig =   {0x9101, IfdTagType_Undefined,    4,  IfdTagState_NotSpecified};
const IfdTag IfdTagMakerNote =    {0x927C, IfdTagType_Undefined, IfdTagState_NotSpecified, IfdTagState_NotSpecified};
const IfdTag IfdTagFlashpixVer =  {0xA000, IfdTagType_Undefined,    4,  IfdTagState_NotSpecified};
const IfdTag IfdTagColorSpace =   {0xA001, IfdTagType_Short,        1,  IfdTagState_NotSpecified};
const IfdTag IfdTagPxXDimension = {0xA002, IfdTagType_ShortOrLong,  1,  IfdTagState_NotSpecified};
const IfdTag IfdTagPxYDimension = {0xA003, IfdTagType_ShortOrLong,  1,  IfdTagState_NotSpecified};
const IfdTag IfdTagUniqueId =     {0xA420, IfdTagType_Ascii,        33, IfdTagState_NotSpecified};
//! @}

//! @name Exif 情報取得用関数
//! @{
/**
    @brief      TIFF ヘッダを取得します。
    @details    Exif 情報全体に係る情報を取得します。
                バイナリ入出力の関数群の初期化も行います。

                Exif 情報を取得する際には、まず最初に呼ぶ必要があります。
 */
bool ReadTiffHeader(
    BinaryIo *pBinaryIo,
    uint16_t *pOffset0thIfd,
    const uint8_t *exifData,
    const uint16_t kExifSize) NN_NOEXCEPT;

/**
    @brief      IFD ヘッダを取得します。
    @details    IFD の概要を取得します。
                概要には内包するタグの数や次の IFD へのオフセットが含まれます。
 */
bool ReadIfdHeader(
    IfdHeader *pIfdHeader,
    const uint16_t kIfdOffset,
    const ExifBinary &exif) NN_NOEXCEPT;

/**
    @brief      IFD からタグを取得します。
    @detail     解析済み IFD ヘッダと検索対象のタグの情報を与えると、
                与えられた IFD からタグを検索し、値の数と位置を取得します。

                もしタグの種類に IfdTagType_ShortOrLong を指定すると、
                該当するタグが Short か Long として見つかった場合、実際の種類で上書きされます。
                それ以外の場合は、実際の種類との一致を検査します。

                もしタグの値の個数に IfdTagState_NotSpecified を指定した場合、
                実際の個数で上書きされます。
                それ以外の場合は、個数の一致を検査します。

                見つからなかったタグは個数が0で上書きされます。
 */
bool SearchIfdTags(
    IfdTag tags[], // Must be sorted by tag ID (asc)
    const uint16_t kNumTags,
    const IfdHeader &ifd,
    const ExifBinary &exif) NN_NOEXCEPT;
//! @}

//! @name Exif 情報バイナリ書き込み用関数
//! @{
/**
    @brief      指定されたエンディアンで Tiff ヘッダを初期化します。
 */
void InitializeTiffHeader(
    TiffHeader *pTiffHeader,
    const Endian kEndian) NN_NOEXCEPT;

/**
    @brief      指定されたアドレスに Tiff ヘッダを出力します。
 */
void WriteTiffHeader(
    uint8_t *exifData,
    const uint16_t kExifSize,
    const TiffHeader &tiffHeader) NN_NOEXCEPT;

/**
    @brief      IFD ヘッダを初期化します。
    @details    タグの種類と値の個数、IFD のオフセットは最終値を指定してください。
                IFD ヘッダの「次の IFD へのオフセット」には、この IFD の末尾+1byteのアドレスが保存されます。
 */
bool InitializeIfdHeader(
    IfdHeader *pIfdHeader,
    const ValuedIfdTag tags[],
    const uint16_t kNumTags,
    const uint16_t kIfdOffset) NN_NOEXCEPT;

/**
    @brief      指定されたアドレスに IFD を出力します。
    @details    IFD ヘッダと、値をすべて書き込みます。
                値はヘッダの直後に書き込まれます。
 */
void WriteIfd(
    uint8_t *exifData,
    const uint16_t kExifSize,
    const ValuedIfdTag tags[],
    const IfdHeader &ifdHeader,
    const BinaryIo &io) NN_NOEXCEPT;
//! @}

}}}
