﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/edid/edid_Edid.h>
#include <nn/edid/edid_DisplayModeInfo.h>
#include <nn/edid/edid_DisplayTimingInfo.h>
#include <nn/edid/edid_ImageSizeInfo.h>
#include "../detail/edid_Edid.h"
#include "../detail/edid_DisplayModeInfo.h"
#include "edid_Cea861.h"

const std::uint8_t  ExtensionBlockId            = 0x02;
const std::uint16_t DataBlockTagMask            = 0xE000;
const std::uint16_t DataBlockTagExtendedMask    = DataBlockTagMask | 0xFF;

const std::uint8_t  EdidBlockCountLocation      = 0x7E;
const std::uint8_t  RevisionLocation            = 0x01;
const std::uint8_t  DetailedTimingOffset        = 0x02;
const std::uint8_t  DataBlockCollectionOffset   = 0x04;

const std::uint8_t  DataBlockSizeMask           = 0x1F;

void nn::edid::cea861::VisitBlocks(const Edid* pEdid, BlockVisitor visitor, void* pUserData) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pEdid);
    NN_SDK_ASSERT_NOT_NULL(visitor);

    if( !detail::IsValid(pEdid) )
    {
        return;
    }

    const std::uint8_t* pData = static_cast<const std::uint8_t*>(pEdid->_data);
    std::uint8_t blockCount = pData[EdidBlockCountLocation];

    for( const std::uint8_t* pBlock = pData + detail::BlockSize;
         pBlock < pData + pEdid->_size && pBlock != pData + ((blockCount + 1) * detail::BlockSize);
         pBlock += detail::BlockSize )
    {
        // first byte of extension block is ID
        if( *pBlock == ExtensionBlockId )
        {
            bool isContinuing = visitor(pBlock, detail::BlockSize, pUserData);

            if( !isContinuing )
            {
                break;
            }
        }
    }

    // TODO: could use block maps to find more directly
}

nn::edid::cea861::BlockTag nn::edid::cea861::GetBlockTag(std::uint16_t value) NN_NOEXCEPT
{
    #define BLOCK_TAG_CASE(tag) case static_cast<std::uint16_t>(BlockTag::tag): return BlockTag::tag

    switch( value & DataBlockTagMask )
    {
        BLOCK_TAG_CASE(Audio);
        BLOCK_TAG_CASE(Video);
        BLOCK_TAG_CASE(VendorSpecific);
        BLOCK_TAG_CASE(SpeakerAllocation);
        BLOCK_TAG_CASE(VesaDtc);

        case static_cast<std::uint16_t>(nn::edid::cea861::BlockTag::VideoCapability):
        {
            switch( value & DataBlockTagExtendedMask )
            {
                BLOCK_TAG_CASE(VideoCapability);
                BLOCK_TAG_CASE(VendorSpecificVideo);
                BLOCK_TAG_CASE(VesaDisplayInfo);
                BLOCK_TAG_CASE(VesaVideo);
                BLOCK_TAG_CASE(HdmiVideo);
                BLOCK_TAG_CASE(Colorimetry);
                BLOCK_TAG_CASE(AudioMisc);
                BLOCK_TAG_CASE(VendorSpecificAudio);
                BLOCK_TAG_CASE(HdmiAudio);

                default:
                    return nn::edid::cea861::BlockTag::Undefined;
            }
        }
        break;

        default:
            return nn::edid::cea861::BlockTag::Undefined;

    }

    #undef BLOCK_TAG_CASE
}

void nn::edid::cea861::VisitDataBlocks(const std::uint8_t* pBlock, size_t size, DataBlockVisitor visitor, void* pUserData) NN_NOEXCEPT
{
    // data blocks exist only in revision 3 and up
    if( pBlock[RevisionLocation] >= 3 )
    {
        const std::uint8_t* pDataBlock = pBlock + DataBlockCollectionOffset;

        while( pDataBlock < pBlock + size && pDataBlock < pBlock + pBlock[DetailedTimingOffset] )
        {
            // shifting to match BlockTag definitions
            std::uint16_t rawTag = *pDataBlock << 8;

            // make sure there's another byte available in the block
            if( (*pDataBlock & DataBlockSizeMask) > 0 )
            {
                // second byte may be extended tag information
                rawTag |= pDataBlock[1];
            }

            BlockTag tag = GetBlockTag(rawTag);

            // size is number of following bytes, hence the +1
            size_t dataBlockSize = (*pDataBlock & DataBlockSizeMask) + 1;

            if( tag != BlockTag::Undefined )
            {
                bool isContinuing = visitor(pDataBlock, dataBlockSize, tag, pUserData);

                if( !isContinuing )
                {
                    break;
                }
            }

            pDataBlock += dataBlockSize;
        }
    }
}

void nn::edid::cea861::VisitDetailedTimings(const std::uint8_t* pBlock, size_t size, DetailedTimingVisitor visitor, void* pUserData) NN_NOEXCEPT
{
    int offset = pBlock[DetailedTimingOffset];

    // TODO: bounds check
    if( offset != 0 )
    {
        for( const std::uint8_t* pDetailedTiming = pBlock + offset;
             pDetailedTiming + detail::DetailedTimingStride < pBlock + size;
             pDetailedTiming += detail::DetailedTimingStride )
        {
            DisplayModeInfo mode;
            DisplayTimingInfo timing;
            ImageSizeInfo image;

            if( detail::GetDetailedTiming(&mode,
                                          &timing,
                                          &image,
                                          pDetailedTiming,
                                          detail::DetailedTimingStride) )
            {
                bool isContinuing = visitor(&mode, &timing, &image, pUserData);

                if( !isContinuing )
                {
                    break;
                }
            }
            else
            {
                // DTD in CEA-861 must be contiguous
                break;
            }
        }
    }
}

static bool WriteVic(nn::edid::DisplayModeInfo* pOutMode, int width, int height, float refreshRate, float imageAspectRatio, float pixelAspectRatio, bool isInterlaced)
{
    pOutMode->width = width;
    pOutMode->height = height;
    pOutMode->refreshRate = refreshRate;
    pOutMode->imageAspectRatio = imageAspectRatio;
    pOutMode->pixelAspectRatio = pixelAspectRatio;
    pOutMode->stereoMode = nn::edid::StereoMode_None;
    pOutMode->isInterlaced = isInterlaced;

    // odd, but saves typing in ConvertVic...
    return true;
}

// Check Table 4 in CEA-861-E spec
bool nn::edid::cea861::ConvertVic(DisplayModeInfo* pOutMode, std::uint8_t vic) NN_NOEXCEPT
{
    switch( vic )
    {
    case 1:
        return WriteVic(pOutMode, 640, 480, 60.00f, 4.f / 3.f, 1.f, false);
    case 2:
        return WriteVic(pOutMode, 720, 480, 60.00f, 4.f / 3.f, 8.f / 9.f, false);
    case 3:
        return WriteVic(pOutMode, 720, 480, 60.00f, 16.f / 9.f, 32.f / 27.f, false);
    case 4:
        return WriteVic(pOutMode, 1280, 720, 60.00f, 16.f / 9.f, 1.f, false);
    case 5:
        return WriteVic(pOutMode, 1920, 1080, 60.00f, 16.f / 9.f, 1.f, true);
    case 6:
        return WriteVic(pOutMode, 720, 480, 60.00f, 4.f / 3.f, 8.f / 9.f, true);
    case 7:
        return WriteVic(pOutMode, 720, 480, 60.00f, 16.f / 9.f, 32.f / 27.f, true);
    case 8:
        return WriteVic(pOutMode, 720, 240, 60.00f, 4.f / 3.f, 4.f / 9.f, false);
    case 9:
        return WriteVic(pOutMode, 720, 240, 60.00f, 16.f / 9.f, 16.f / 27.f, false);
    case 10:
        return WriteVic(pOutMode, 2880, 480, 60.00f, 4.f / 3.f, 2.f / 9.f, true);
    case 11:
        return WriteVic(pOutMode, 2880, 480, 60.00f, 16.f / 9.f, 8.f / 27.f, true);
    case 12:
        return WriteVic(pOutMode, 2880, 240, 60.00f, 4.f / 3.f, 1.f / 9.f, false);
    case 13:
        return WriteVic(pOutMode, 2880, 240, 60.00f, 16.f / 9.f, 4.f / 27.f, false);
    case 14:
        return WriteVic(pOutMode, 1440, 480, 60.00f, 4.f / 3.f, 4.f / 9.f, false);
    case 15:
        return WriteVic(pOutMode, 1440, 480, 60.00f, 16.f / 9.f, 16.f / 27.f, false);
    case 16:
        return WriteVic(pOutMode, 1920, 1080, 60.00f, 16.f / 9.f, 1.f, false);
    case 17:
        return WriteVic(pOutMode, 720, 576, 50.00f, 4.f / 3.f, 16.f / 15.f, false);
    case 18:
        return WriteVic(pOutMode, 720, 576, 50.00f, 16.f / 9.f, 64.f / 45.f, false);
    case 19:
        return WriteVic(pOutMode, 1280, 720, 50.00f, 16.f / 9.f, 1.f, false);
    case 20:
        return WriteVic(pOutMode, 1920, 1080, 50.00f, 16.f / 9.f, 1.f, true);
    case 21:
        return WriteVic(pOutMode, 720, 576, 50.00f, 4.f / 3.f, 16.f / 15.f, true);
    case 22:
        return WriteVic(pOutMode, 720, 576, 50.00f, 16.f / 9.f, 64.f / 45.f, true);
    case 23:
        return WriteVic(pOutMode, 720, 288, 50.00f, 4.f / 3.f, 8.f / 15.f, false);
    case 24:
        return WriteVic(pOutMode, 720, 288, 50.00f, 16.f / 9.f, 32.f / 45.f, false);
    case 25:
        return WriteVic(pOutMode, 2880, 576, 50.00f, 4.f / 3.f, 2.f / 15.f, true);
    case 26:
        return WriteVic(pOutMode, 2880, 576, 50.00f, 16.f / 9.f, 16.f / 45.f, true);
    case 27:
        return WriteVic(pOutMode, 2880, 288, 50.00f, 4.f / 3.f, 1.f / 15.f, false);
    case 28:
        return WriteVic(pOutMode, 2880, 288, 50.00f, 16.f / 9.f, 8.f / 45.f, false);
    case 29:
        return WriteVic(pOutMode, 1440, 576, 50.00f, 4.f / 3.f, 8.f / 15.f, false);
    case 30:
        return WriteVic(pOutMode, 1440, 576, 50.00f, 16.f / 9.f, 32.f / 45.f, false);
    case 31:
        return WriteVic(pOutMode, 1920, 1080, 50.00f, 16.f / 9.f, 1.f, false);
    case 32:
        return WriteVic(pOutMode, 1920, 1080, 24.00f, 16.f / 9.f, 1.f, false);
    case 33:
        return WriteVic(pOutMode, 1920, 1080, 25.00f, 16.f / 9.f, 1.f, false);
    case 34:
        return WriteVic(pOutMode, 1920, 1080, 30.00f, 16.f / 9.f, 1.f, false);
    case 35:
        return WriteVic(pOutMode, 2880, 480, 60.00f, 4.f / 3.f, 2.f / 9.f, false);
    case 36:
        return WriteVic(pOutMode, 2880, 480, 60.00f, 16.f / 9.f, 8.f / 27.f, false);
    case 37:
        return WriteVic(pOutMode, 2880, 576, 50.00f, 4.f / 3.f, 4.f / 15.f, false);
    case 38:
        return WriteVic(pOutMode, 2880, 576, 50.00f, 16.f / 9.f, 16.f / 45.f, false);
    case 39:
        return WriteVic(pOutMode, 1920, 1080, 50.00f, 16.f / 9.f, 1.f, true);
    case 40:
        return WriteVic(pOutMode, 1920, 1080, 100.00f, 16.f / 9.f, 1.f, true);
    case 41:
        return WriteVic(pOutMode, 1280, 720, 100.00f, 16.f / 9.f, 1.f, false);
    case 42:
        return WriteVic(pOutMode, 720, 576, 100.00f, 4.f / 3.f, 16.f / 15.f, false);
    case 43:
        return WriteVic(pOutMode, 720, 576, 100.00f, 16.f / 9.f, 64.f / 45.f, false);
    case 44:
        return WriteVic(pOutMode, 720, 576, 100.00f, 4.f / 3.f, 16.f / 15.f, true);
    case 45:
        return WriteVic(pOutMode, 720, 576, 100.00f, 16.f / 9.f, 64.f / 45.f, true);
    case 46:
        return WriteVic(pOutMode, 1920, 1080, 120.00f, 16.f / 9.f, 1.f, true);
    case 47:
        return WriteVic(pOutMode, 1280, 720, 120.00f, 16.f / 9.f, 1.f, false);
    case 48:
        return WriteVic(pOutMode, 720, 480, 120.00f, 4.f / 3.f, 8.f / 9.f, false);
    case 49:
        return WriteVic(pOutMode, 720, 480, 120.00f, 16.f / 9.f, 32.f / 27.f, false);
    case 50:
        return WriteVic(pOutMode, 720, 480, 120.00f, 4.f / 3.f, 8.f / 9.f, true);
    case 51:
        return WriteVic(pOutMode, 720, 480, 120.00f, 16.f / 9.f, 32.f / 27.f, true);
    case 52:
        return WriteVic(pOutMode, 720, 576, 200.00f, 4.f / 3.f, 16.f / 15.f, false);
    case 53:
        return WriteVic(pOutMode, 720, 576, 200.00f, 16.f / 9.f, 64.f / 45.f, false);
    case 54:
        return WriteVic(pOutMode, 720, 576, 200.00f, 4.f / 3.f, 16.f / 15.f, true);
    case 55:
        return WriteVic(pOutMode, 720, 576, 200.00f, 16.f / 9.f, 64.f / 45.f, true);
    case 56:
        return WriteVic(pOutMode, 720, 480, 240.00f, 4.f / 3.f, 8.f / 9.f, false);
    case 57:
        return WriteVic(pOutMode, 720, 480, 240.00f, 16.f / 9.f, 32.f / 27.f, false);
    case 58:
        return WriteVic(pOutMode, 720, 480, 240.00f, 4.f / 3.f, 8.f / 9.f, true);
    case 59:
        return WriteVic(pOutMode, 720, 480, 240.00f, 16.f / 9.f, 32.f / 27.f, true);
    case 60:
        return WriteVic(pOutMode, 1280, 720, 24.00f, 16.f / 9.f, 1.f, false);
    case 61:
        return WriteVic(pOutMode, 1280, 720, 25.00f, 16.f / 9.f, 1.f, false);
    case 62:
        return WriteVic(pOutMode, 1280, 720, 30.00f, 16.f / 9.f, 1.f, false);
    case 63:
        return WriteVic(pOutMode, 1920, 1080, 120.00f, 16.f / 9.f, 1.f, false);
    case 64:
        return WriteVic(pOutMode, 1920, 1080, 100.00f, 16.f / 9.f, 1.f, false);
    case 65:
        return WriteVic(pOutMode, 1280, 720, 24.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 66:
        return WriteVic(pOutMode, 1280, 720, 25.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 67:
        return WriteVic(pOutMode, 1280, 720, 30.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 68:
        return WriteVic(pOutMode, 1280, 720, 50.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 69:
        return WriteVic(pOutMode, 1280, 720, 60.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 70:
        return WriteVic(pOutMode, 1280, 720, 100.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 71:
        return WriteVic(pOutMode, 1280, 720, 120.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 72:
        return WriteVic(pOutMode, 1920, 1080, 24.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 73:
        return WriteVic(pOutMode, 1920, 1080, 25.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 74:
        return WriteVic(pOutMode, 1920, 1080, 30.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 75:
        return WriteVic(pOutMode, 1920, 1080, 50.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 76:
        return WriteVic(pOutMode, 1920, 1080, 60.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 77:
        return WriteVic(pOutMode, 1920, 1080, 100.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 78:
        return WriteVic(pOutMode, 1920, 1080, 120.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 79:
        return WriteVic(pOutMode, 1680, 720, 24.00f, 64.f / 27.f, 64.f / 63.f, false);
    case 80:
        return WriteVic(pOutMode, 1680, 720, 25.00f, 64.f / 27.f, 64.f / 63.f, false);
    case 81:
        return WriteVic(pOutMode, 1680, 720, 30.00f, 64.f / 27.f, 64.f / 63.f, false);
    case 82:
        return WriteVic(pOutMode, 1680, 720, 50.00f, 64.f / 27.f, 64.f / 63.f, false);
    case 83:
        return WriteVic(pOutMode, 1680, 720, 60.00f, 64.f / 27.f, 64.f / 63.f, false);
    case 84:
        return WriteVic(pOutMode, 1680, 720, 100.00f, 64.f / 27.f, 64.f / 63.f, false);
    case 85:
        return WriteVic(pOutMode, 1680, 720, 120.00f, 64.f / 27.f, 64.f / 63.f, false);
    case 86:
        return WriteVic(pOutMode, 2560, 1080, 24.00f, 64.f / 27.f, 1.f, false);
    case 87:
        return WriteVic(pOutMode, 2560, 1080, 25.00f, 64.f / 27.f, 1.f, false);
    case 88:
        return WriteVic(pOutMode, 2560, 1080, 30.00f, 64.f / 27.f, 1.f, false);
    case 89:
        return WriteVic(pOutMode, 2560, 1080, 50.00f, 64.f / 27.f, 1.f, false);
    case 90:
        return WriteVic(pOutMode, 2560, 1080, 60.00f, 64.f / 27.f, 1.f, false);
    case 91:
        return WriteVic(pOutMode, 2560, 1080, 100.00f, 64.f / 27.f, 1.f, false);
    case 92:
        return WriteVic(pOutMode, 2560, 1080, 120.00f, 64.f / 27.f, 1.f, false);
    case 93:
        return WriteVic(pOutMode, 3840, 2160, 24.00f, 16.f / 9.f, 1.f, false);
    case 94:
        return WriteVic(pOutMode, 3840, 2160, 25.00f, 16.f / 9.f, 1.f, false);
    case 95:
        return WriteVic(pOutMode, 3840, 2160, 30.00f, 16.f / 9.f, 1.f, false);
    case 96:
        return WriteVic(pOutMode, 3840, 2160, 50.00f, 16.f / 9.f, 1.f, false);
    case 97:
        return WriteVic(pOutMode, 3840, 2160, 60.00f, 16.f / 9.f, 1.f, false);
    case 98:
        return WriteVic(pOutMode, 4096, 2160, 24.00f, 256.f / 135.f, 1.f, false);
    case 99:
        return WriteVic(pOutMode, 4096, 2160, 25.00f, 256.f / 135.f, 1.f, false);
    case 100:
        return WriteVic(pOutMode, 4096, 2160, 30.00f, 256.f / 135.f, 1.f, false);
    case 101:
        return WriteVic(pOutMode, 4096, 2160, 50.00f, 256.f / 135.f, 1.f, false);
    case 102:
        return WriteVic(pOutMode, 4096, 2160, 60.00f, 256.f / 135.f, 1.f, false);
    case 103:
        return WriteVic(pOutMode, 3840, 2160, 24.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 104:
        return WriteVic(pOutMode, 3840, 2160, 25.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 105:
        return WriteVic(pOutMode, 3840, 2160, 30.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 106:
        return WriteVic(pOutMode, 3840, 2160, 50.00f, 64.f / 27.f, 4.f / 3.f, false);
    case 107:
        return WriteVic(pOutMode, 3840, 2160, 60.00f, 64.f / 27.f, 4.f / 3.f, false);
    default:
        return false;
    }
} // NOLINT(impl/function_size)
