﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/crypto/detail/crypto_XtsModeImpl.h>
#include "crypto_UpdateImpl.h"

namespace nn { namespace crypto { namespace detail {

size_t XtsModeImpl::UpdateGeneric(void* pDst, size_t dstSize, const void* pSrc, size_t srcSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_State == State_Initialized || m_State == State_Processing, "Invalid state. Please restart from Initialize().");

    return UpdateImpl<void>(this, pDst, dstSize, pSrc, srcSize);
}

size_t XtsModeImpl::ProcessBlocksGeneric(uint8_t* pDst8, const uint8_t* pSrc8, int numBlocks) NN_NOEXCEPT
{
    size_t processed = (numBlocks - 1) * BlockSize;

    // 1ブロック以上処理済みの場合はまずバッファされているブロックを処理する
    if (m_State == State_Processing)
    {
        ProcessBlock(pDst8, m_LastBlock);
        pDst8 += BlockSize;
        processed += BlockSize;
    }

    // 最後のブロック以外を処理して dst に書き出す
    while (--numBlocks > 0)
    {
        ProcessBlock(pDst8, pSrc8);
        pDst8 += BlockSize;
        pSrc8 += BlockSize;
    }

    // 最後のブロックは Finalize で使われる可能性があるので平文で保存しておく
    std::memcpy(m_LastBlock, pSrc8, BlockSize);

    m_State = State_Processing;

    return processed;
}

template <>
size_t XtsModeImpl::Update<AesEncryptor128>(void* pDst, size_t dstSize, const void* pSrc, size_t srcSize) NN_NOEXCEPT
{
    return UpdateGeneric(pDst, dstSize, pSrc, srcSize);
}

template <>
size_t XtsModeImpl::Update<AesDecryptor128>(void* pDst, size_t dstSize, const void* pSrc, size_t srcSize) NN_NOEXCEPT
{
    return UpdateGeneric(pDst, dstSize, pSrc, srcSize);
}

}}} // namespace nn::crypto::detail
