﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#ifndef __cplusplus

/**
* @file
* @brief   SDK 用の、ログを出力する C 言語用 API の宣言
*/

#include <nnc/nn_Common.h>
#include <nnc/diag/detail/diag_DetailLog.h>

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
#define NN_DETAIL_ENABLE_SDK_LOG
#elif defined(NN_SDK_BUILD_RELEASE)
#else
#error "ビルドタイプマクロが定義されていません。"
#endif

#if defined(NN_DETAIL_ENABLE_SDK_LOG)

/**
* @brief 引数に指定された文字列を、ログとして出力します。
*
* @param[in] ...    printf 形式で指定する、ログとして出力する文字列の書式指定文字列と、それに与える引数
*/
#define NNC_SDK_LOG(...)                                                    \
    do                                                                      \
    {                                                                       \
        nndiagLogMetaData logMetaData;                                      \
        logMetaData.sourceInfo.lineNumber = __LINE__;                       \
        logMetaData.sourceInfo.fileName = __FILE__;                         \
        logMetaData.sourceInfo.functionName = NNC_CURRENT_FUNCTION_NAME;    \
        logMetaData.moduleName = "$";                                       \
        logMetaData.severity = nndiagLogSeverity_Info;                      \
        logMetaData.verbosity = 0;                                          \
        logMetaData.useDefaultLocaleCharset = false;                        \
        logMetaData.pAdditionalData = NULL;                                 \
        logMetaData.additionalDataBytes = 0;                                \
        nndetailDiagLogImplForC(&logMetaData, __VA_ARGS__);                 \
    } while (NNC_STATIC_CONDITION(0))

/**
* @brief 引数に指定された文字列を、ログとして出力します。引数を可変長引数リストで与えることができます。
*
* @param[in] format printf 形式で指定する書式指定文字列
* @param[in] args   書式指定文字列に与える引数
*/
#define NNC_SDK_VLOG(format, args)                                          \
    do                                                                      \
    {                                                                       \
        nndiagLogMetaData logMetaData;                                      \
        logMetaData.sourceInfo.lineNumber = __LINE__;                       \
        logMetaData.sourceInfo.fileName = __FILE__;                         \
        logMetaData.sourceInfo.functionName = NNC_CURRENT_FUNCTION_NAME;    \
        logMetaData.moduleName = "$";                                       \
        logMetaData.severity = nndiagLogSeverity_Info;                      \
        logMetaData.verbosity = 0;                                          \
        logMetaData.useDefaultLocaleCharset = false;                        \
        logMetaData.pAdditionalData = NULL;                                 \
        logMetaData.additionalDataBytes = 0;                                \
        nndetailDiagVLogImplForC(&logMetaData, format, args);               \
    } while (NNC_STATIC_CONDITION(0))

/**
* @brief 引数に指定された文字列を、ログとして出力します。出力する文字列のバイトサイズを引数で指定します。
*
* @param[in] text   出力する文字列
* @param[in] length 出力する文字列のバイトサイズ
*/
#define NNC_SDK_PUT(text, length)                                           \
    do                                                                      \
    {                                                                       \
        nndiagLogMetaData logMetaData;                                      \
        logMetaData.sourceInfo.lineNumber = __LINE__;                       \
        logMetaData.sourceInfo.fileName = __FILE__;                         \
        logMetaData.sourceInfo.functionName = NNC_CURRENT_FUNCTION_NAME;    \
        logMetaData.moduleName = "$";                                       \
        logMetaData.severity = nndiagLogSeverity_Info;                      \
        logMetaData.verbosity = 0;                                          \
        logMetaData.useDefaultLocaleCharset = false;                        \
        logMetaData.pAdditionalData = NULL;                                 \
        logMetaData.additionalDataBytes = 0;                                \
        nndetailDiagPutImplForC(&logMetaData, text, length);                \
    } while (NNC_STATIC_CONDITION(0))

#else

#define NNC_SDK_LOG(...) (void)(0)
#define NNC_SDK_VLOG(...) (void)(0)
#define NNC_SDK_PUT(...) (void)(0)

#endif

#endif // __cplusplus
