﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
* @file
* @brief   条件を満たしていない時に停止するための API の C言語用の宣言
*/

#include <nnc/nn_Common.h>
#include <nnc/diag/detail/diag_DetailAssert.h>
#include <nnc/diag/detail/diag_DetailNamedAssert.h>

#ifndef __cplusplus

/**
* @brief 実行を停止します。このマクロは常に有効です。
*
* @param[in] ...          ログとして出力するメッセージの書式指定文字列と引数
*/
#define NNC_ABORT(...) \
    NNC_DETAIL_ABORT_UNLESS_IMPL(false, "NNC_ABORT", ##__VA_ARGS__)

/**
* @brief 条件を満たしていることをテストし、満たされていない場合に停止します。このマクロは常に有効です。
*
* @param[in] condition    テストする条件
* @param[in] ...          条件を満たしていない場合にログとして出力するメッセージの書式指定文字列と引数
*/
#define NNC_ABORT_UNLESS(condition, ...) \
    NNC_DETAIL_ABORT_UNLESS_IMPL((condition), NNC_MACRO_STRINGIZE(condition), ##__VA_ARGS__)

/**
* @brief ポインタがヌルポインタではないことをテストし、そうではない場合に停止します。
*        満たされるべき条件は pointer != nullptr です。
*        このマクロは常に有効です。
*/
#define NNC_ABORT_UNLESS_NOT_NULL(pointer) \
    NNC_DETAIL_NAMED_ASSERT_IMPL_1( \
        NNC_DETAIL_ABORT_UNLESS_IMPL, NNC_DETAIL_ASSERT_NOT_NULL_IMPL, \
        (pointer), NNC_MACRO_STRINGIZE(pointer))

/**
* @brief 値が等しいことをテストし、そうではない場合に停止します。
*        満たされるべき条件は lhs == rhs です。
*        このマクロは常に有効です。
*/
#define NNC_ABORT_UNLESS_EQUAL(lhs, rhs) \
    NNC_DETAIL_NAMED_ASSERT_IMPL_2( \
        NNC_DETAIL_ABORT_UNLESS_IMPL, NNC_DETAIL_ASSERT_EQUAL_IMPL, \
        (lhs), (rhs), NNC_MACRO_STRINGIZE(lhs), NNC_MACRO_STRINGIZE(rhs))

/**
* @brief 値が等しくないことをテストし、そうではない場合に停止します。
*        満たされるべき条件は lhs != rhs です。
*        このマクロは常に有効です。
*/
#define NNC_ABORT_UNLESS_NOT_EQUAL(lhs, rhs) \
    NNC_DETAIL_NAMED_ASSERT_IMPL_2( \
        NNC_DETAIL_ABORT_UNLESS_IMPL, NNC_DETAIL_ASSERT_NOT_EQUAL_IMPL, \
        (lhs), (rhs), NNC_MACRO_STRINGIZE(lhs), NNC_MACRO_STRINGIZE(rhs))

/**
* @brief 値が範囲に含まれていることをテストし、そうではない場合に停止します。
*        満たされるべき条件は (begin <= value) && (value < end) です。
*        このマクロは常に有効です。
*/
#define NNC_ABORT_UNLESS_RANGE(value, begin, end) \
    NNC_DETAIL_NAMED_ASSERT_IMPL_3( \
        NNC_DETAIL_ABORT_UNLESS_IMPL, NNC_DETAIL_ASSERT_RANGE_IMPL, \
        (value), (begin), (end), NNC_MACRO_STRINGIZE(value), NNC_MACRO_STRINGIZE(begin), NNC_MACRO_STRINGIZE(end))

#endif // __cplusplus

