﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/crypto/detail/crypto_EccKey.h>

namespace nn { namespace crypto { namespace detail {

class EccP256Key
{
    typedef EccKey<256> Model;

public:
    static const size_t BitsCount = Model::BitsCount;
    static const int PrivateKeySize = Model::PrivateKeySize;
    static const int PublicKeySize = Model::PublicKeySize;
    static const int SignatureSize = Model::SignatureSize;
    static const int SharedSecretSize = Model::SharedSecretSize;

    void Initialize( const void* publicKey, size_t publicKeySize,
                     const void* privateKey, size_t privateKeySize )  NN_NOEXCEPT
    {
        m_Model.Initialize( publicKey, publicKeySize, privateKey, privateKeySize );
    }

    /*
    Recommended in Suite B Implementer’s Guide to FIPS 186-3
    The minimum PrivateKeySize is tolerated, to pass NIST test vectors
    */
    static const int RandomSize = PrivateKeySize + 64 / 8;

    /*
    It is imperative to provide a random input for every signature:
    from 2 signatures that use the same "random" the private key would leak
    */
    bool SignHash(void* outSignature, size_t signatureSize,
                  const void* pHash, size_t hashSize,
                  const void* random, size_t randomSize) const NN_NOEXCEPT;
    bool VerifyHash(const void* pSignature, size_t signatureSize,
                    const void* pHash, size_t hashSize ) const NN_NOEXCEPT;

    bool GenerateSharedSecret( void* outSharedSecret, size_t sharedSecretSize,
                               const EccP256Key& recipientPublicKey ) const NN_NOEXCEPT;

private:
    Model m_Model;
    const struct p256_point* GetPublicKey() const;
};

}}} // namespace nn::crypto::detail
