﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Net;
using System.Runtime.Serialization.Json;
using System.Web.Script.Serialization;
using System.IO;

namespace ZarfServerAccessor
{
    class MyWebClient : WebClient
    {
        int _timeoutSec;

        public MyWebClient(int timeoutSec)
        {
            _timeoutSec = timeoutSec;
        }

        protected override WebRequest GetWebRequest(Uri address)
        {
            var w = base.GetWebRequest(address);
            w.Timeout = _timeoutSec * 1000;
            return w;
        }
    }

    public class Accessor
    {
        public const string UserAgent = "Mozilla/4.0 (compatible; MSIE 6.0; Windows XP)";
        string _serverUrl = null;
        string _authToken = null;

        private string _CreateSendString(IDictionary<string, object> args)
        {
            JavaScriptSerializer serializer = new JavaScriptSerializer();
            return serializer.Serialize(args);
        }

        private T _SendRequest<T>(string command, IDictionary<string, object> args)
        {
            using (MyWebClient wc = new MyWebClient(3 * 60))
            {
                var url = _serverUrl + command + ".php";
                wc.Encoding = Encoding.UTF8;
                wc.Headers.Add("Authorization", string.Format("Bearer {0}", _authToken));
                wc.Headers.Add("Content-Type", "application/json");
                wc.Headers.Add("User-Agent", UserAgent);
                var res = wc.UploadString(url, "POST", _CreateSendString(args));
                JavaScriptSerializer selializer = new JavaScriptSerializer();
                if (selializer.MaxJsonLength < res.Length)
                {
                    selializer.MaxJsonLength = res.Length;
                }
                return (T)selializer.Deserialize(res, typeof(T));
            }
        }

        private static byte[] GetHeaderBytes(IEnumerable<string> headers, Encoding enc)
        {
            var ms = new MemoryStream();
            using (var sw = new StreamWriter(ms, enc))
            {
                sw.NewLine = "\r\n";

                foreach (var line in headers)
                {
                    sw.WriteLine(line);
                }

                sw.Flush();

                return ms.ToArray();
            }
        }

        private string UploadFile(string url, string filePath)
        {
            var fileName = Path.GetFileName(filePath);

            var enc = new UTF8Encoding(false);  // UTF-8(BOMなし)

            var boundary = Environment.TickCount.ToString(); // 区切り文字列

            // HttpWebRequestの作成
            var req = (HttpWebRequest)WebRequest.Create(url);
            req.Method = "POST";
            req.ContentType = "multipart/form-data; boundary=" + boundary;
            req.UserAgent = UserAgent;
            req.Timeout = 20 * 60 * 1000;
            // 大きいファイルサイズをアップロードする際、
            // falseにしないと OutOfMemoryExceptionが発生する。
            req.AllowWriteStreamBuffering = false;
            req.Headers.Add("Authorization", string.Format("Bearer {0}", _authToken));

            // POST送信するデータを作成
            var startDataStr = new[]
            {
                "--" + boundary,
                "Content-Disposition: form-data; name=\"upfile\"; filename=\"" + fileName + "\"",
                "Content-Type: application/octet-stream",
                "Content-Transfer-Encoding: binary",
                ""
            };
            var startData = GetHeaderBytes(startDataStr, enc);

            var endDataStr = new[]
            {
                "",
                "--" + boundary + "--"
            };
            var endData = GetHeaderBytes(endDataStr, enc);

            using (var fs = File.OpenRead(filePath))
            {
                // POST送信するデータの長さを指定
                req.ContentLength = startData.Length + endData.Length + fs.Length;

                using (var reqStream = req.GetRequestStream())
                {
                    reqStream.Write(startData, 0, startData.Length);
                    fs.CopyTo(reqStream);
                    reqStream.Write(endData, 0, endData.Length);
                }
            }

            // HttpWebResponseを取得
            var res = (HttpWebResponse)req.GetResponse();
            using (var sr = new StreamReader(res.GetResponseStream(), enc))
            {
                return sr.ReadToEnd();
            }
        }

        private T _SendFile<T>(string command, string filePath)
        {
            var resText = UploadFile(_serverUrl + command + ".php", filePath);

            var selializer = new JavaScriptSerializer();
            return (T)selializer.Deserialize(resText, typeof(T));
        }

        public Accessor()
        {
        }

        private void _LoginCheck()
        {
            if (_authToken == null)
            {
                throw new InvalidOperationException("ログインする必要があります。");
            }
        }

        public bool IsLogin
        {
            get
            {
                return _authToken != null;
            }
        }

        public bool TryUploadZarf(string zarf, out string errorMessage)
        {
            _LoginCheck();

            var result = _SendFile<IDictionary<string, object>>("FileUpload", zarf);
            if (!result.ContainsKey("Success"))
            {
                errorMessage = (string)result["Error"];
                return false;
            }
            else
            {
                errorMessage = null;
                return true;
            }
        }

        public IEnumerable<object> ListFamilies()
        {
            _LoginCheck();

            var res = _SendRequest<IEnumerable<object>>("ListFamilies", new Dictionary<string, object>());
            return res;
        }

        public IEnumerable<object> GetFamilyTree()
        {
            _LoginCheck();

            var res = _SendRequest<IEnumerable<object>>("GetFamilyTree", new Dictionary<string, object>());
            return res;
        }

        public IEnumerable<object> GetExperienceTree()
        {
            _LoginCheck();

            var res = _SendRequest<IEnumerable<object>>("GetExperienceTree", new Dictionary<string, object>());
            return res;
        }


        public void Login(string url, string id, string password)
        {
            if (_authToken != null)
            {
                throw new InvalidOperationException();
            }

            _serverUrl = url;
            var param = new Dictionary<string, object>(){
                {"loginName", id},
                {"password", password}
            };

            var res = _SendRequest <IDictionary<string, string>>("AuthenticateAdmin", param);
            if (!res.ContainsKey("Success"))
            {
                throw new ArgumentException("ログインに失敗しました。id/pass を確認してください。");
            }

            _authToken = res["token"];
        }

        public void Logout()
        {
            _authToken = null;
        }

        public bool TryRemoveRelease(int hiddenID, out string errorMessage)
        {
            var param = new Dictionary<string, object>() {
                {"hiddenID", hiddenID},
            };
            var result = _SendRequest<IDictionary<string, string>>("RemoveRelease", param);
            if (!result.ContainsKey("Success"))
            {
                errorMessage = (string)result["Error"];
                return false;
            }
            else
            {
                errorMessage = null;
                return true;
            }
        }

        public Dictionary<string, object> GetExperienceInfo(string expName)
        {
            _LoginCheck();

            var exps = GetExperienceTree();
            foreach (Dictionary<string, object> exp in exps)
            {
                Dictionary<string, object> text = exp["text"] as Dictionary<string, object>;
                var name = text["ja_JP"] as string;
                if (name == expName)
                {
                    return exp;
                }
            }

            return null;
        }

        public void EditExperienceInfo(IDictionary<string, object> exp)
        {
            _LoginCheck();

            var res = _SendRequest<IDictionary<string, object>>("EditExperience", exp);
            if (!res.ContainsKey("Success"))
            {
                throw new ArgumentException(string.Format("Experience の編集に失敗しました。\n{0}", res["Error"]));
            }
        }

        public void SetCoreReleaseZarf(string expName, string zarf) {
            _LoginCheck();

            string filename = Path.GetFileName(zarf);

            int id = -1;
            var families = GetFamilyTree();
            foreach (IDictionary<string, object> family in families)
            {
                var children = family["children"] as object[];

                foreach (Dictionary<string, object> release in children)
                {
                    var info = ((object[])release["children"])[0] as Dictionary<string, object>;
                    if (info["filename"] as string == filename)
                    {
                        id = (int)release["hiddenID"];
                    }
                }
            }

            var targetExp = GetExperienceInfo(expName);
            targetExp["coreRel"] = id.ToString();

            EditExperienceInfo(targetExp);
        }
    }
}
