﻿using Newtonsoft.Json;
using Newtonsoft.Json.Serialization;
using Nintendo.Zarf;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using ZarfCreator.MetaData;
using ZarfCreator.Parser;
using ZarfCreator.ZarfDefinitionData;

namespace ZarfCreator
{
    public class ZarfBuilder
    {
        // CreateSymbolicLink のインポート
        [System.Runtime.InteropServices.DllImport(
                "kernel32.dll", EntryPoint = "CreateHardLink", SetLastError = true)]
        private static extern bool CreateHardLink(
                string lpFileName, string lpExistingFileName, IntPtr lpSecurityAttributes);

        // CreateHardLink 呼び出し時、既にハードリンクが存在する場合に返されるエラー値の定義
        // System Error Codes (http://msdn.microsoft.com/ja-jp/library/ms681382.aspx) より
        private const int ErrorAlreadyExists = 0xB7;

        // MUGEN が必要とするメタデータのファイル名・ディレクトリ名
        private static readonly string LogicDirName = "deploy-logic";
        private static readonly string BinDirName = "deploy-bin";
        private static readonly string ZarfSpecFileName = "zarf-spec.json";
        private static readonly string NdiInstructionsFileName = "ndi-instructions.json";
        private static readonly string NdiRollbackFileName = "ndi-rollback.json";
        private static readonly string NdiMetaDataFileName = "ndi-metadata.json";

        private ZarfDefinition ZarfInfo { get; set; }
        private string RootPath { get; set; }
        private string[] Archives { get; set; }

        /// <summary>
        /// ZarfBuilder クラスの新しいインスタンスを作成します。
        /// </summary>
        /// <param name="zarfInfo">作成する Zarf の情報</param>
        /// <param name="root">SDK のルートディレクトリ</param>
        /// <param name="archives">含めるファイルのパスのリスト</param>
        public ZarfBuilder(ZarfDefinition zarfInfo, string root, string[] archives)
        {
            this.ZarfInfo = zarfInfo;
            this.RootPath = root;
            this.Archives = archives;
        }

        /// <summary>
        /// 設定された情報に基づいて Zarf ファイルを作成します。
        /// </summary>
        /// <param name="outPath">出力先ディレクトリ</param>
        public void Create(string outPath)
        {
            var zarfRootDir = Path.Combine(outPath, this.ZarfInfo.Name);
            var zarfBinDir = Path.Combine(zarfRootDir, BinDirName);
            var zarfLogicDir = Path.Combine(zarfRootDir, LogicDirName);

            var outputFilePath = Path.Combine(outPath, this.ZarfInfo.Name + ".zarf");

            if (Directory.Exists(zarfRootDir))
            {
                Directory.Delete(zarfRootDir, true);
            }
            if (File.Exists(outputFilePath))
            {
                File.Delete(outputFilePath);
            }
            Directory.CreateDirectory(zarfRootDir);
            Directory.CreateDirectory(zarfBinDir);
            Directory.CreateDirectory(zarfLogicDir);

            this.CopyIncludingFiles(zarfBinDir, zarfLogicDir);

            this.CreateMetaFiles(zarfRootDir, zarfBinDir, zarfLogicDir, this.Archives);

            Series1Zarf.CreateNewZarfFromExistingDirectory(outputFilePath, zarfRootDir, false);

            this.CleanUp(zarfRootDir);
        }

        private void CopyIncludingFiles(string zarfBinDir, string zarfLogicDir)
        {
            // --archive-path を deploy-bin へコピー
            CopyFiles(this.Archives, zarfBinDir);

            // SetupFiles を deploy-logic へコピー
            CopyFiles(
                this.ZarfInfo.SetupFiles.Select(file => Path.Combine(this.RootPath, file)).ToArray(),
                zarfLogicDir);
        }


        private void CopyFiles(string[] files, string outputPath)
        {
            foreach (var file in files)
            {
                if (!File.Exists(file))
                {
                    throw new ArgumentException(string.Format("'{0}' has not found.", file));
                }

                var destination = Path.Combine(outputPath, Path.GetFileName(file));
                if (!CreateHardLink(destination, file, IntPtr.Zero))
                {
                    var lastError = System.Runtime.InteropServices.Marshal.GetLastWin32Error();

                    if (lastError == ErrorAlreadyExists)
                    {
                        // 別のプロセスも同時にハードリンクを作成しようとして競合したと考え、無視する
                    }
                    else
                    {
                        throw new System.ComponentModel.Win32Exception(
                                lastError,
                                string.Format(
                                    "Cannot create the hard link from {0} to {1}.",
                                    file, destination));
                    }
                }
            }
        }

        private void CreateMetaFiles(string rootDirPath, string binDirPath, string logicDirPath, IEnumerable<string> archiveFiles)
        {
            var serializerSettings = new JsonSerializerSettings();

            serializerSettings.ContractResolver = new CamelCasePropertyNamesContractResolver();
            serializerSettings.Formatting = Formatting.Indented;

            var zarfSpecFile = new ZarfSpec(serializerSettings);
            zarfSpecFile.Create(Path.Combine(rootDirPath, ZarfSpecFileName));

            var ndiInstructionFile = new NdiInstructions(this.ZarfInfo, serializerSettings);
            ndiInstructionFile.Create(Path.Combine(logicDirPath, NdiInstructionsFileName));

            var ndiRollbackFile = new NdiRollback(archiveFiles, this.ZarfInfo.Instruction.InstallDirectory, serializerSettings, this.ZarfInfo.Rollback);
            ndiRollbackFile.Create(Path.Combine(logicDirPath, NdiRollbackFileName));

            // metadata は rollback.json や instructions.json のハッシュを計算するので最後に作る
            var ndiMetaDataFile = new NdiMetaData(this.ZarfInfo, logicDirPath, binDirPath, serializerSettings);
            ndiMetaDataFile.Create(Path.Combine(rootDirPath, NdiMetaDataFileName));
        }

        private void CleanUp(string zarfDir)
        {
            Directory.Delete(zarfDir, true);
        }
    }
}
