﻿using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using ZarfCreator.VersionData;
using ZarfCreator.ZarfDefinitionData;

namespace ZarfCreator.Parser
{
    public class ZarfDefinitionParser
    {
        private string rootPath { get; set; }
        private string contentsLocale { get; set; }
        private string snapshotVersion { get; set; }
        private readonly RelatedZarfParser relatedZarfParser;
        private readonly InstructionParser instructionParser;
        private readonly RollbackParser rollbackParser;

        /// <summary>
        /// ZarfDefinition のインスタンスを作成します。
        /// </summary>
        /// <param name="root">SDK のルートディレクトリ</param>
        public ZarfDefinitionParser(string root, string contentsLocale = null, string snapshotVersion = "")
        {
            this.rootPath = root;
            this.contentsLocale = contentsLocale;
            this.snapshotVersion = snapshotVersion;
            this.relatedZarfParser = new RelatedZarfParser();
            this.instructionParser = new InstructionParser();
            this.rollbackParser = new RollbackParser();
        }

        /// <summary>
        /// Zarf 定義ファイルから情報を取得します。互換先、依存先の Zarf 定義のパースや、バージョンファイルの読み込みも行います。
        /// </summary>
        /// <param name="sourcePath">Zarf 定義ファイルのパス</param>
        /// <param name="archives">Zarf に含めるファイルのリスト</param>
        /// <returns>読み込んだ Zarf の情報</returns>
        public ZarfDefinition ReadRecursively(string sourcePath, string[] archives)
        {
            try
            {
                var yamlData = ZarfYamlReader.Read(sourcePath);

                var zarfInfo = this.Parse(yamlData["Zarf"] as Dictionary<string, object>);
                this.ReadRelatedZarfs(zarfInfo);

                var archiveName = from archive in archives select Path.GetFileName(archive);
                this.ReplaceVariables(zarfInfo, archiveName.ToArray());

                var versionReader = new VersionData.VersionDataReader();
                zarfInfo.Version = versionReader.Read(zarfInfo.VersionFile, this.rootPath, zarfInfo.EnableRelstep);

                zarfInfo.Rollback = this.ReadRollbackFiles(zarfInfo.RollbackFiles);

                if(this.snapshotVersion != string.Empty)
                {
                    zarfInfo.Version = SnapshotVersionDataFactory.CreateSnapshotVersion(
                        zarfInfo.Version,
                        this.snapshotVersion,
                        Properties.Settings.Default.IsUsePrereleaseSnapshot);
                }

                return zarfInfo;
            }
            catch (Exception)
            {
                Console.Error.WriteLine("ERROR: Failed to read the zarf definition.");
                throw;
            }
        }

        /// <summary>
        /// Zarf 定義のパースのみを行います。互換先、依存先の Zarf 定義のパースは行いません。
        /// </summary>
        /// <param name="sourcePath">パースする Zarf 定義ファイルのパス</param>
        /// <returns>読み込んだ Zarf の情報</returns>
        internal ZarfDefinition Parse(Dictionary<string, object> node)
        {
            var zarf = new ZarfDefinition(this.contentsLocale);

            foreach (var key in node.Keys)
            {
                try
                {
                    switch (key)
                    {
                        case ZarfDefinitionElement.Name:
                            zarf.Name = (string)node[key];
                            break;
                        case ZarfDefinitionElement.Type:
                            zarf.Type = (string)node[key];
                            break;
                        case ZarfDefinitionElement.Family:
                            zarf.Family = (string)node[key];
                            break;
                        case ZarfDefinitionElement.Department:
                            zarf.Department = (string)node[key];
                            break;
                        case ZarfDefinitionElement.Description:
                            zarf.Description = TextInfoParser.Parse((Dictionary<string, object>)node[key]);
                            break;
                        case ZarfDefinitionElement.VersionFile:
                            zarf.VersionFile = (string)node[key];
                            break;
                        case ZarfDefinitionElement.EnableRelstep:
                            zarf.EnableRelstep = ParseBool((string)node[key]);
                            break;
                        case ZarfDefinitionElement.AddonTo:
                            zarf.AddonTargetZarf = new RelatedZarfInfo()
                            {
                                Target = (string)node[key],
                            };
                            break;
                        case ZarfDefinitionElement.SetupFiles:
                            // キーのみの場合は node[key] は空文字になる
                            if (node[key] as string != string.Empty)
                            {
                                zarf.SetupFiles = ((List<object>)node[key]).ConvertToStringList();
                            }
                            break;
                        case ZarfDefinitionElement.Dependencies:
                            if (node[key] as string != string.Empty)
                            {
                                zarf.Dependencies = this.relatedZarfParser.Parse((List<object>)node[key]);
                            }
                            break;
                        case ZarfDefinitionElement.Compatibilities:
                            if (node[key] as string != string.Empty)
                            {
                                zarf.Compatibilities = this.relatedZarfParser.Parse((List<object>)node[key]);
                            }
                            break;
                        case ZarfDefinitionElement.RuntimeDependencies:
                            if (node[key] as string != string.Empty)
                            {
                                zarf.RuntimeDependencies = this.relatedZarfParser.Parse((List<object>)node[key]);
                            }
                            break;
                        case ZarfDefinitionElement.ContentsLocale:
                            if (node[key] as string != string.Empty)
                            {
                                var locale = ((string)node[key]).ToLowerInvariant();
                                zarf.ContentsLocale = locale == "all" ? null : locale;
                            }
                            break;
                        case ZarfDefinitionElement.Platforms:
                            if (node[key] as string != string.Empty)
                            {
                                var platforms = new StringCollection();
                                platforms.AddRange(((List<object>)node[key]).ConvertToStringArray());
                                zarf.Platforms = platforms;
                            }
                            break;
                        case ZarfDefinitionElement.Instructions:
                            try
                            {
                                zarf.Instruction = this.instructionParser.Parse((Dictionary<string, object>)node[key]);
                            }
                            catch (Exception)
                            {
                                Console.Error.WriteLine("ERROR: Failed to parse the 'Instructions' field.");
                                throw;
                            }
                            break;
                        case ZarfDefinitionElement.Rollbacks:
                            if (node[key] as string != string.Empty)
                            {
                                zarf.RollbackFiles = ((List<object>)node[key]).ConvertToStringList();
                            }
                            break;
                        case ZarfDefinitionElement.Eula:
                            zarf.Eula = TextInfoParser.Parse((Dictionary<string, object>)node[key]);
                            break;
                        default:
                            throw new FormatException("Unknown key has specified.");
                    }
                }
                catch (Exception)
                {
                    Console.Error.WriteLine("ERROR: Error has occured at '{0}'. ", key);
                    throw;
                }
            }

            string error;
            if (!zarf.Validate(out error))
            {
                Console.Error.WriteLine(error);

                throw new FormatException("Required items have not been completed.");
            }

            return zarf;
        }

        /// <summary>
        /// 文字列で表現された Zarf 定義をパースするための正規表現
        ///
        /// 例：Nintendo:NintendoSDK (<Department>:<Family>)
        /// </summary>
        private const string ZarfStringRegexPattern = @"(.+):(.+)";

        /// <summary>
        /// 文字列で表現された Zarf 定義のパースを行います。
        /// </summary>
        /// <param name="value">Zarf を表現する文字列</param>
        /// <returns>Zarf の情報</returns>
        internal ZarfDefinition Parse(string value)
        {
            var result = Regex.Match(value, ZarfStringRegexPattern);
            if(!result.Success)
            {
                throw new FormatException("Invalid format.");
            }

            var zarf = new ZarfDefinition(this.contentsLocale);
            zarf.Department = result.Groups[1].Value;
            zarf.Family = result.Groups[2].Value;

            return zarf;
        }

        // 依存と互換の読み込み
        private void ReadRelatedZarfs(ZarfDefinition zarf)
        {
            zarf.Dependencies = this.relatedZarfParser.Read(zarf.Dependencies, this.rootPath, zarf.ContentsLocale);
            zarf.Compatibilities = this.relatedZarfParser.Read(zarf.Compatibilities, this.rootPath, zarf.ContentsLocale);
            zarf.RuntimeDependencies = this.relatedZarfParser.Read(zarf.RuntimeDependencies, this.rootPath, zarf.ContentsLocale);

            if(zarf.AddonTargetZarf != null)
            {
                zarf.AddonTargetZarf = this.relatedZarfParser.Read(zarf.AddonTargetZarf, this.rootPath, zarf.ContentsLocale);
            }
        }

        // 変数の展開
        private void ReplaceVariables(ZarfDefinition zarfInfo, string[] archives)
        {
            try
            {
                zarfInfo.Instruction.ReplaceVariables(archives);
            }
            catch (Exception)
            {
                Console.Error.WriteLine("ERROR: Failed to expand variables.");
                throw;
            }
        }

        /// <summary>
        /// Rollback 定義ファイル の読み込みを行います。
        /// </summary>
        /// <param name="rollbackFiles">Rollback 定義ファイルパス(相対)のリスト</param>
        /// <returns>RollbackInfoオブジェクト。</returns>
        private RollbackInfo ReadRollbackFiles(IEnumerable<string> rollbackFiles)
        {
            var rollbacks = new List<RollbackInfo>();

            foreach (var rollbackFile in rollbackFiles)
            {
                try
                {
                    var yamlData = YamlDataReader.Read(Path.Combine(rootPath, rollbackFile));
                    var subRollback = this.rollbackParser.Parse(yamlData["Rollbacks"] as Dictionary<string, object>);
                    rollbacks.Add(subRollback);
                }
                catch (Exception)
                {
                    Console.Error.WriteLine("ERROR: Failed to read rollback file '{0}'.", rollbackFile);
                    throw;
                }
            }

            return
                new RollbackInfo
                {
                    TrackFilePatternList = rollbacks.SelectMany(subRollback => subRollback.TrackFilePatternList)
                };
        }

        /// <summary>
        /// Bool文字列をパースします。。
        /// </summary>
        /// <param name="str">Bool文字列</param>
        /// <returns>Bool値</returns>
        private static bool ParseBool(string str)
        {
            switch (str.ToUpperInvariant())
            {
            case "TRUE":
                return true;
            case "FALSE":
                return false;
            default:
                throw new FormatException("Invalid value has specified. Must be specified TRUE/FALSE.");
            }
        }

        internal static class ZarfDefinitionElement
        {
            public const string Root = "Zarf";
            public const string Name = "Name";
            public const string Type = "Type";
            public const string Description = "Description";
            public const string VersionFile = "VersionFile";
            public const string EnableRelstep = "EnableRelstep";
            public const string Family = "Family";
            public const string Department = "Department";
            public const string SetupFiles = "SetupFiles";
            public const string Dependencies = "Dependencies";
            public const string Compatibilities = "Compatibilities";
            public const string Instructions = "Instructions";
            public const string ContentsLocale = "ContentsLocale";
            public const string Platforms = "Platforms";
            public const string RuntimeDependencies = "RuntimeDependencies";
            public const string AddonTo = "AddonTo";
            public const string Rollbacks = "Rollbacks";
            public const string Eula = "Eula";
        }
    }
}
