﻿using Newtonsoft.Json;
using Nintendo.DevInterface.DirScanner;
using Nintendo.Zarf.v1;
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using ZarfCreator.Parser;
using ZarfCreator.ZarfDefinitionData;

namespace ZarfCreator.MetaData
{
    internal class NdiMetaData
    {
        private ZarfDefinition zarfInfo { get; set; }
        private string logicDirPath { get; set; }
        private string binDirPath { get; set; }
        private JsonSerializerSettings serializerSettings { get; set; }

        /// <summary>
        /// NdiMetaData クラスのインスタンスを作成します。
        /// </summary>
        /// <param name="zarfInfo">Zarf に設定する情報</param>
        /// <param name="logicDirPath">deploy-logic ディレクトリのパス</param>
        /// <param name="binDirPath">deploy-bin ディレクトリのパス</param>
        /// <param name="serializerSettings"></param>
        public NdiMetaData(ZarfDefinition zarfInfo, string logicDirPath, string binDirPath, JsonSerializerSettings serializerSettings)
        {
            this.zarfInfo = zarfInfo;
            this.logicDirPath = logicDirPath;
            this.binDirPath = binDirPath;
            this.serializerSettings = serializerSettings;
        }

        /// <summary>
        /// ndi-metadata.json を作成します。
        /// </summary>
        /// <param name="outputDirectory">出力先</param>
        internal void Create(string outputDirectory)
        {
            try
            {
                var content = this.CreateContent();
                File.WriteAllText(outputDirectory, content);
            }
            catch (Exception)
            {
                Console.Error.WriteLine("ERROR: Failed to create the meta file.");
                throw;
            }
        }

        private string CreateContent()
        {
            try
            {
                var metaData = new ZarfNdiMetadata();

                metaData.LogicPieces = CreateFolderContents(this.logicDirPath);
                metaData.BinPieces = CreateFolderContents(this.binDirPath);

                metaData.About = new About();

                var family = new FamilyInfo();
                family.Key = this.zarfInfo.Family;
                family.Department = this.zarfInfo.Department;
                metaData.About.Family = family;

                var release = new ReleaseInfo();
                release.Version = this.zarfInfo.Version;

                var releaseNameText =
                    this.zarfInfo.ContentsLocale != null ?
                        string.Format("{0} {1} {2}", this.zarfInfo.Name, this.zarfInfo.Version, this.zarfInfo.ContentsLocale):
                        string.Format("{0} {1}",     this.zarfInfo.Name, this.zarfInfo.Version);
                var releaseName = new DualText();
                releaseName.Add(DualText.Japanese, releaseNameText);
                releaseName.Add(DualText.English, releaseNameText);
                release.Name = releaseName;

                if (this.zarfInfo.ContentsLocale != null)
                {
                    var contentsLocale = new StringCollection();
                    contentsLocale.AddRange(new[] { this.zarfInfo.ContentsLocale });
                    release.ContentsLocale = contentsLocale;
                }

                release.ConsolePlatform = this.zarfInfo.Platforms;

                if (zarfInfo.AddonTargetZarf != null)
                {
                    release.AddOnTo = (AddOnToSpecifier)zarfInfo.AddonTargetZarf;
                }

                release.RuntimeDependencies = zarfInfo.RuntimeDependencies.Select(item => (DependenciesSpecifier)item).ToList();

                release.Description = this.zarfInfo.Description.GetDualText();

                release.InstallerDependencies = zarfInfo.Dependencies.Select(item => (DependenciesSpecifier)item).ToList();

                release.Compatibilities = zarfInfo.Compatibilities.Select(item => (CompatibilitiesSpecifier)item).ToList();

                if (this.zarfInfo.Eula != null)
                {
                    release.EULA = this.zarfInfo.Eula.GetDualText();
                }

                metaData.About.Release = release;

                return JsonConvert.SerializeObject(metaData, this.serializerSettings);
            }
            catch (Exception)
            {
                Console.Error.WriteLine("ERROR: Failed to create json contents.");
                throw;
            }
        }

        private FolderContents CreateFolderContents(string folderPath)
        {
            var contents = new FolderContents();

            IEnumerable<IFileRecord> record = from file in Directory.GetFiles(folderPath)
                                            select new FileRecord(file);

            var inventories = from file in record.AsParallel()
                              select new FileInventory()
                              {
                                  Filename = file.FileName,
                                  Signature = file.CalculatedSignature,
                              };
            Array.ConvertAll(inventories.ToArray(), e => contents.Add(e));

            return contents;
        }

    }
}
