﻿using Newtonsoft.Json;
using Nintendo.Zarf.v1;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using ZarfCreator.Parser;
using ZarfCreator.ZarfDefinitionData;

namespace ZarfCreator.MetaData
{
    internal class NdiInstructions
    {
        private ZarfDefinition zarfInfo { get; set; }
        private JsonSerializerSettings serializerSettings { get; set; }

        /// <summary>
        /// NdiInstructions クラスのインスタンスを作成します。
        /// </summary>
        /// <param name="zarfInfo">Zarf に設定する情報</param>
        /// <param name="serializerSettings"></param>
        public NdiInstructions(ZarfDefinition zarfInfo, JsonSerializerSettings serializerSettings)
        {
            this.zarfInfo = zarfInfo;
            this.serializerSettings = serializerSettings;
        }

        /// <summary>
        /// ndi-instructions.json を作成します。
        /// </summary>
        /// <param name="outputDirectory">出力先</param>
        internal void Create(string outputDirectory)
        {
            try
            {
                var content = this.CreateContent();
                File.WriteAllText(outputDirectory, content);
            }
            catch (Exception)
            {
                Console.Error.WriteLine("ERROR: Failed to create the instruction file.");
                throw;
            }
        }

        private string CreateContent()
        {
            try
            {
                var instructions = new ZarfNdiInstructions();
                instructions.Install = new Install();

                var installCommandList = from command in this.zarfInfo.Instruction.Install
                                         select GetCommand(command);
                instructions.Install.CommandList = installCommandList.ToList();
                instructions.Install.RollBack = this.zarfInfo.Instruction.RollBackType;


                instructions.Uninstall = new Uninstall();

                var uninstallCommandList = from command in this.zarfInfo.Instruction.Uninstall
                                           select GetCommand(command);
                instructions.Uninstall.CommandList = uninstallCommandList.ToList();


                if (this.zarfInfo.Instruction.Prerequisites.Count != 0)
                {
                    instructions.Prerequisites = new Prerequisites();

                    var prerequirementCommandList = from command in this.zarfInfo.Instruction.Prerequisites
                                                    select GetCommand(command);
                    instructions.Prerequisites.CommandList = prerequirementCommandList.ToList();
                }

                if (this.zarfInfo.Instruction.PostInstall.Count != 0)
                {
                    instructions.PostInstall = new PostInstall();

                    var postInstallCommandList = from command in this.zarfInfo.Instruction.PostInstall
                                                    select GetCommand(command);
                    instructions.PostInstall.CommandList = postInstallCommandList.ToList();
                }

                instructions.InstallDirectory = this.zarfInfo.Instruction.InstallDirectory;
                instructions.LogicRequiresContentsFromBinFolder = this.zarfInfo.Instruction.RequireBinDir;

                return JsonConvert.SerializeObject(instructions, this.serializerSettings);
            }
            catch (Exception)
            {
                Console.Error.WriteLine("ERROR: Failed to create json contents.");
                throw;
            }
        }

        static private Command GetCommand(InstructionCommandInfo command)
        {
            return new Command()
            {
                Handler = (Command.HandlerKind)(command.Handler),
                CmdSpecifier = command.CmdSpecifier,
                CmdArgs = command.CmdArgs,
                SuccessReturnCodes = command.SuccessReturnCodes,
                OnFailure = command.OnFailure,
                Predicate = command.Predicate,
                PredicateArgs = command.PredicateArgs,
                DashboardDetails = GetDashboardDetails(command.DashboardDetails),
            };
        }

        static private DashboardDetails GetDashboardDetails(DashboardDetailsInfo dashboardDetails)
        {
            if (dashboardDetails == null)
            {
                return null;
            }

            return new DashboardDetails
            {
                GroupName = dashboardDetails.GroupName,
                GroupDisplayName = dashboardDetails.GroupDisplayName.GetDualText(),
                Icon = dashboardDetails.Icon,
                DisplayText = dashboardDetails.DisplayText.GetDualText(),
            };
        }
    }
}
