﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System.Collections.Generic;
    using System.Linq;
    using Executer;

    /// <summary>
    /// テストノードのローカル環境の検証と保持を行います。
    /// </summary>
    internal sealed class TestNodeInfo
    {
        internal const int MinimumLocalRepeat = 1;

        internal const int DefaultTimeout = (15 * 60) + 30;

        /// <summary>
        /// TestNodeInfo クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="node">テストノードです。</param>
        internal TestNodeInfo(object node)
        {
            this.MappingNode = node as Dictionary<string, object>;

            this.Resources = new List<string>();

            this.VariableManager = new VariableManager();
        }

        /// <summary>
        /// ワーキングディレクトリの絶対パスを取得または設定します。
        /// </summary>
        internal string WorkingDirectory { get; set; }

        /// <summary>
        /// テスト結果ファイルの出力先となるディレクトリの絶対パスを取得または設定します。
        /// </summary>
        internal string ResultDirectory { get; set; }

        /// <summary>
        /// 並列実行に対応しているか否かを示す値を取得または設定します。
        /// </summary>
        internal bool Parallelizable { get; set; }

        /// <summary>
        /// ブレークレベルを取得または設定します。
        /// </summary>
        internal BreakLevel BreakLevel { get; set; }

        /// <summary>
        /// テスト ID を取得または設定します。
        /// </summary>
        internal uint TestId { get; set; }

        /// <summary>
        /// グローバルリピート ID を取得または設定します。
        /// </summary>
        internal uint GlobalRepeatId { get; set; }

        /// <summary>
        /// ローカルリピート ID を取得または設定します。
        /// </summary>
        internal uint LocalRepeatId { get; set; }

        /// <summary>
        /// ローカルリピート回数を取得または設定します。
        /// </summary>
        internal int LocalRepeat { get; set; }

        /// <summary>
        /// 実行するプログラムが参照するファイルの絶対パスを取得または設定します。
        internal IReadOnlyList<string> Resources { get; set; }

        /// <summary>
        /// 実行するプログラムの失敗条件とするログ出力のパターンを取得または設定します。
        /// </summary>
        internal IReadOnlyList<string> FailurePatterns { get; set; }

        /// <summary>
        /// タイムアウトまでの待ち時間（秒）を取得または設定します。
        /// </summary>
        internal int Timeout { get; set; }

        /// <summary>
        /// テスト対象のプラットフォームを取得または設定します。
        /// </summary>
        internal string Platform { get; set; }

        /// <summary>
        /// テスト対象のビルドタイプを取得または設定します。
        /// </summary>
        internal string BuildType { get; set; }

        /// <summary>
        /// テスト対象のアプリケーションプログラムフォーマットを取得または設定します。
        /// </summary>
        internal string ApplicationProgramFormat { get; set; }

        /// <summary>
        /// 開発機の名前の正規表現を取得または設定します。
        /// </summary>
        internal string TargetNamePattern { get; set; }

        /// <summary>
        /// 開発機のインターフェイス名の正規表現を取得または設定します。
        /// </summary>
        internal string TargetInterfacePattern { get; set; }

        /// <summary>
        /// 変数マネージャを取得または設定します。
        /// </summary>
        internal VariableManager VariableManager { get; set; }

        private IReadOnlyDictionary<string, object> MappingNode { get; set; }

        /// <summary>
        /// シャローコピーを取得します。
        /// </summary>
        /// <returns>シャローコピーです。</returns>
        public TestNodeInfo Clone()
        {
            return (TestNodeInfo)this.MemberwiseClone();
        }

        /// <summary>
        /// マッピングノードが有効な LocalRepeat ノードを含むかどうかを示す値を取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>マッピングノードが有効な LocalRepeat ノードを含むかどうかを示す値です。</returns>
        internal static bool HasValidLocalRepeatNode(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return TestListValidation.HasValidIntegerNode(
                mappingNode, Tag.LocalRepeat, MinimumLocalRepeat);
        }

        /// <summary>
        /// LocalRepeat ノードの値をマッピングノードから取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>LocalRepeat ノードの値です。</returns>
        internal static int GetLocalRepeatNodeValue(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return int.Parse((string)mappingNode[Tag.LocalRepeat]);
        }

        /// <summary>
        /// マッピングノードが有効な Timeout ノードを含むかどうかを示す値を取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>マッピングノードが有効な Timeout ノードを含むかどうかを示す値です。</returns>
        internal static bool HasValidTimeoutNode(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return TestListValidation.HasValidIntegerNode(
                mappingNode,
                Tag.Timeout,
                TestExecuter.MinimumTimeout, TestExecuter.MaximumTimeout);
        }

        /// <summary>
        /// Timeout ノードの値をマッピングノードから取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>Timeout ノードの値です。</returns>
        internal static int GetTimeoutNodeValue(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return int.Parse((string)mappingNode[Tag.Timeout]);
        }

        /// <summary>
        /// マッピングノードが WRONG_YAML に該当する共通ノードを含むかどうかを示す値を取得します。
        /// </summary>
        /// <param name="str">エラーメッセージを取得します。</param>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>マッピングノードが WRONG_YAML に該当する共通ノードを含むかどうかを示す値です。</returns>
        internal static bool HasWrongCommonNode(
            out string str, IReadOnlyDictionary<string, object> mappingNode)
        {
            if (TestListValidation.HasWrongIntegerNode(
                    out str, mappingNode, Tag.LocalRepeat, MinimumLocalRepeat))
            {
                return true;
            }

            if (TestListValidation.HasWrongIntegerNode(
                    out str,
                    mappingNode,
                    Tag.Timeout,
                    TestExecuter.MinimumTimeout, TestExecuter.MaximumTimeout))
            {
                return true;
            }

            if (TestUnitInfo.HasWrongCommonNode(out str, mappingNode))
            {
                return true;
            }

            str = string.Empty;

            return false;
        }

        /// <summary>
        /// WRONG_YAML に該当するテストノードかどうかを示す値を取得します。
        /// </summary>
        /// <param name="str">エラーメッセージを取得します。</param>
        /// <returns>WRONG_YAML に該当するテストノードかどうかを示す値です。</returns>
        internal bool IsWrongYaml(out string str)
        {
            if (this.MappingNode == null)
            {
                str = "Invalid node type for a test entry";

                return true;
            }

            IReadOnlyList<TestUnitInfo> unitInfos = this.GetTestUnitInfos(
                this.GetTestUnitNodesConvertedFromTestNode());

            foreach (var unitInfo in unitInfos)
            {
                if (unitInfo.IsWrongYaml(out str))
                {
                    return true;
                }
            }

            if (TestUnitInfo.HasWrongUnitsNode(out str, this.MappingNode))
            {
                return true;
            }

            if (HasWrongCommonNode(out str, this.MappingNode))
            {
                return true;
            }

            if (TestUnitInfo.HasValidUnitsNode(this.MappingNode))
            {
                unitInfos = this.GetTestUnitInfos(
                    TestUnitInfo.GetUnitsNodeValue(this.MappingNode));

                foreach (var unitInfo in unitInfos)
                {
                    if (unitInfo.IsWrongYaml(out str))
                    {
                        return true;
                    }
                }
            }

            str = string.Empty;

            return false;
        }

        /// <summary>
        /// NO_PATH に該当するテストノードかどうかを示す値を取得します。
        /// </summary>
        /// <returns>NO_PATH に該当するテストノードかどうかを示す値です。</returns>
        internal bool IsNoPath()
        {
            var errorMessage = string.Empty;

            if (this.IsWrongYaml(out errorMessage))
            {
                return true;
            }

            if (!TestUnitInfo.HasValidUnitsNode(this.MappingNode))
            {
                return this.GetTestUnitInfos(
                    this.GetTestUnitNodesConvertedFromTestNode()
                    ).Any(unitInfo => unitInfo.IsNoPath());
            }
            else
            {
                var scalarNode =
                    TestUnitInfo.GetUnitsNodeValue(this.MappingNode);

                if (scalarNode.Where(x => x != null).Count() == 0)
                {
                    return true;
                }

                return this.GetTestUnitInfos(
                    scalarNode).Any(unitInfo => unitInfo.IsNoPath());
            }
        }

        /// <summary>
        /// ローカルリピート回数を取得します。
        /// </summary>
        /// <returns>ローカルリピート回数です。</returns>
        internal int GetLocalRepeat()
        {
            if (TestListValidation.HasValidIntegerNode(
                    this.MappingNode, Tag.LocalRepeat, MinimumLocalRepeat))
            {
                return int.Parse((string)this.MappingNode[Tag.LocalRepeat]);
            }
            else
            {
                return this.LocalRepeat;
            }
        }

        /// <summary>
        /// テストコンテキストの配列を取得します。
        /// </summary>
        /// <returns>テストコンテキストの配列です。</returns>
        internal TestContext[] GetTestContexts()
        {
            return this.GetTestUnitInfos(
                this.GetEffectiveTestUnitNodes()
                ).Select(x => x.GetTestContext()).ToArray();
        }

        private IReadOnlyList<object> GetTestUnitNodesConvertedFromTestNode()
        {
            return new Dictionary<string, object>[]
            {
                TestUnitInfo.ExtractCommonElements(this.MappingNode)
            };
        }

        private IReadOnlyList<object> GetEffectiveTestUnitNodes()
        {
            if (TestUnitInfo.HasValidUnitsNode(this.MappingNode))
            {
                return TestUnitInfo.GetUnitsNodeValue(this.MappingNode);
            }

            return this.GetTestUnitNodesConvertedFromTestNode();
        }

        private IReadOnlyList<TestUnitInfo> GetTestUnitInfos(
            IReadOnlyList<object> nodes)
        {
            var unitInfos = new List<TestUnitInfo>();

            var unitCount = 0u;

            var timeout = HasValidTimeoutNode(this.MappingNode)
                ? GetTimeoutNodeValue(this.MappingNode)
                : this.Timeout;

            IReadOnlyList<string> resources =
                TestUnitInfo.HasValidResourcesNode(this.MappingNode)
                ? TestUnitInfo.GetResourcesNodeValue(this.MappingNode)
                : this.Resources;

            IReadOnlyList<string> failurePatterns =
                TestUnitInfo.HasValidFailurePatternsNode(this.MappingNode)
                ? TestUnitInfo.GetFailurePatternsNodeValue(this.MappingNode)
                : this.FailurePatterns;

            var platform = TestUnitInfo.HasValidPlatformNode(this.MappingNode)
                ? TestUnitInfo.GetPlatformNodeValue(this.MappingNode)
                : this.Platform;

            var buildType =
                TestUnitInfo.HasValidBuildTypeNode(this.MappingNode)
                ? TestUnitInfo.GetBuildTypeNodeValue(this.MappingNode)
                : this.BuildType;

            var applicationProgramFormat =
                TestUnitInfo.HasValidApplicationProgramFormatNode(
                    this.MappingNode)
                ? TestUnitInfo.GetApplicationProgramFormatNodeValue(
                    this.MappingNode)
                : this.ApplicationProgramFormat;

            var targetNamePattern =
                TestUnitInfo.HasValidTargetNamePatternNode(
                    this.MappingNode)
                ? TestUnitInfo.GetTargetNamePatternNodeValue(
                    this.MappingNode)
                : this.TargetNamePattern;

            var targetInterfacePattern =
                TestUnitInfo.HasValidTargetInterfacePatternNode(
                    this.MappingNode)
                ? TestUnitInfo.GetTargetInterfacePatternNodeValue(
                    this.MappingNode)
                : this.TargetInterfacePattern;

            foreach (var node in nodes)
            {
                ++unitCount;

                if (node == null)
                {
                    continue;
                }

                var unitInfo = new TestUnitInfo(node)
                {
                    WorkingDirectory = this.WorkingDirectory,
                    ResultDirectory = this.ResultDirectory,
                    Parallelizable = this.Parallelizable,
                    BreakLevel = BreakLevel.None,
                    RoleType = RoleType.Test,
                    TestId = this.TestId,
                    GlobalRepeatId = this.GlobalRepeatId,
                    LocalRepeatId = this.LocalRepeatId,
                    UnitCount = (uint)nodes.Count(),
                    UnitId = unitCount,
                    Resources = resources,
                    FailurePatterns = failurePatterns,
                    Timeout = timeout,
                    Platform = platform,
                    BuildType = buildType,
                    ApplicationProgramFormat = applicationProgramFormat,
                    TargetNamePattern = targetNamePattern,
                    TargetInterfacePattern = targetInterfacePattern,
                    VariableManager = this.VariableManager
                };

                unitInfos.Add(unitInfo);
            }

            if (unitInfos.Count > 0)
            {
                unitInfos.First().BreakLevel = this.BreakLevel;
            }

            return unitInfos;
        }

        private static class Tag
        {
            internal const string LocalRepeat = "LocalRepeat";

            internal const string Timeout = "Timeout";
        }
    }
}
