﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner.GenerationRule
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using Executer;
    using Executer.InsertionRule;
    using Microsoft.Build.Evaluation;
    using Microsoft.Build.Execution;
    using Microsoft.Win32;
    using static InstructionGenerator;

    /// <summary>
    /// vcxproj ファイル用のインストラクション生成ルールです。
    /// </summary>
    internal sealed class VcProjectFileRule : IRule
    {
        /// <summary>
        /// VcProjectFileRule クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal VcProjectFileRule()
        {
            this.OasisVersion = new Lazy<string>(GetOasisVersion);
        }

        private Lazy<string> OasisVersion { get; set; }

        /// <summary>
        /// サポート対象のテストコンテキストかどうかを示す値を取得します。
        /// </summary>
        /// <param name="context">テストコンテキストです。</param>
        /// <returns>サポート対象のテストコンテキストかどうかを示す値です。</returns>
        public bool Supports(TestContext context)
        {
            return context.TargetPath.EndsWith(".vcxproj");
        }

        /// <summary>
        /// ルール適用時にターゲットファイルを必要とするかどうかを示す値を取得します。
        /// </summary>
        /// <returns>ルール適用時にターゲットファイルを必要とするかどうかを示す値です。</returns>
        public bool NeedsTargetFile()
        {
            return true;
        }

        /// <summary>
        /// テストコンテキストにインストラクション生成ルールを適用します。
        /// </summary>
        /// <param name="context">テストコンテキストです。</param>
        /// <returns>生成されたテストインストラクションです。</returns>
        public Instruction Generate(TestContext context)
        {
            TestFrameworkName framework = TestFrameworkName.None;

            Instruction instruction =
                string.IsNullOrEmpty(context.TargetEpiPath)
                    ? this.GenerateFromVcProjectFile(out framework, context)
                    : this.GenerateFromEpiFile(out framework, context);

            instruction.Command = instruction.TargetPath;

            instruction.Option = context.Option;

            instruction.TargetProjectPath = context.TargetPath;

            instruction.ReportPath = context.ReportPath;

            instruction.DumpFileType = DumpFileTypeDefinition.MiniDump;

            instruction.DumpFilePath = Utility.RemoveExtension(
                context.Path, ExtensionDefinition.TestResult
                ) + DumpFileExtensionDefinition.MiniDump;

            switch (framework)
            {
                case TestFrameworkName.None:
                    break;

                case TestFrameworkName.GoogleTest:
                    AppendGtestOutput(instruction, context);
                    break;

                case TestFrameworkName.MSTest:
                    UpdateInstructionForVsTest(instruction, context);
                    break;
            }

            var nspRule = new NspFileRule();

            UpdateTestContext(context, instruction);

            if (nspRule.Supports(context))
            {
                return nspRule.Generate(context);
            }
            else
            {
                return instruction;
            }
        }

        private static string GetPlatform(string platform)
        {
            if (string.IsNullOrEmpty(platform))
            {
                platform = BuildSystem.DefaultPlatform;
            }

            switch (platform)
            {
                case PlatformDefinition.Win32:
                case PlatformDefinition.Win32Vs2013:
                case PlatformDefinition.Win32Vs2015:
                case PlatformDefinition.Win32Vs2017:
                    return PlatformDefinition.Win32;

                case PlatformDefinition.Win64:
                case PlatformDefinition.Win64Vs2013:
                case PlatformDefinition.Win64Vs2015:
                case PlatformDefinition.Win64Vs2017:
                case PlatformDefinition.X64:
                    return PlatformDefinition.X64;

                case PlatformDefinition.NxOnWin32:
                case PlatformDefinition.NxWin32:
                case PlatformDefinition.NxWin32Vs2013:
                case PlatformDefinition.NxWin32Vs2015:
                case PlatformDefinition.NxWin32Vs2017:
                    return PlatformDefinition.Win32;

                case PlatformDefinition.NxOnWin64:
                case PlatformDefinition.NxWin64:
                case PlatformDefinition.NxWin64Vs2013:
                case PlatformDefinition.NxWin64Vs2015:
                case PlatformDefinition.NxWin64Vs2017:
                    return PlatformDefinition.X64;

                case PlatformDefinition.Nx32:
                case PlatformDefinition.NxFp2:
                case PlatformDefinition.NxFp2A32:
                    return PlatformDefinition.Nx32;

                case PlatformDefinition.Nx64:
                case PlatformDefinition.NxFp2A64:
                    return PlatformDefinition.Nx64;

                default:
                    throw new TestRunnerException(
                        $"Platform '{platform}' is not supported");
            }
        }

        private static string GetConfiguration(
            string platform,
            string buildType)
        {
            if (string.IsNullOrEmpty(platform))
            {
                platform = BuildSystem.DefaultPlatform;
            }

            string value = !string.IsNullOrEmpty(buildType)
                ? buildType
                : BuildSystem.DefaultBuildType;

            if (!string.IsNullOrEmpty(platform))
            {
                if (platform.EndsWith("_VS2013"))
                {
                    value = "VS2013_" + value;
                }
                else if (platform.EndsWith("_VS2015"))
                {
                    value = "VS2015_" + value;
                }
                else if (platform.EndsWith("_VS2017"))
                {
                    value = "VS2017_" + value;
                }

                var nxPlatforms = new string[] {
                    PlatformDefinition.Nx32,
                    PlatformDefinition.Nx64,
                };

                var nxPrefixes = new string[] {
                    PlatformDefinition.NxFp2,
                    "NXOn",
                    "NX-",
                };

                if (nxPlatforms.Contains(platform) ||
                    nxPrefixes.Any(x => platform.StartsWith(x)))
                {
                    value = "NX_" + value;
                }
            }

            return value;
        }

        private static string GetApplicationProgramFormat(string format)
        {
            if (string.IsNullOrEmpty(format))
            {
                format = ApplicationProgramFormatDefinition.Nsp;
            }

            switch (format)
            {
                case ApplicationProgramFormatDefinition.Raw:
                case ApplicationProgramFormatDefinition.Nca:
                case ApplicationProgramFormatDefinition.Nsp:
                    return format;

                default:
                    throw new TestRunnerException(
                        $"ApplicationProgramFormat '{format}' is not supported");
            }
        }

        private static bool NeedsOasis(string vsPlatform)
        {
            return new[] { PlatformDefinition.Nx32,
                           PlatformDefinition.Nx64 }.Contains(vsPlatform);
        }

        private static string GetOasisVersion()
        {
            var name = @"SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall";

            var hive = RegistryHive.LocalMachine;

            var view = RegistryView.Registry64;

            using (var baseKey = RegistryKey.OpenBaseKey(hive, view))
            {
                using (RegistryKey key = baseKey.OpenSubKey(name))
                {
                    foreach (string subName in key.GetSubKeyNames())
                    {
                        using (RegistryKey subKey = key.OpenSubKey(subName))
                        {
                            var value = subKey.GetValue("DisplayName");

                            var displayName = value as string;

                            if (!string.IsNullOrEmpty(displayName))
                            {
                                if (displayName.Contains("Nintendo Target Manager"))
                                {
                                    value = subKey.GetValue("DisplayVersion");

                                    var displayVersion = value as string;

                                    return displayVersion;
                                }
                            }
                        }
                    }
                }
            }

            return null;
        }

        private static void ParseProject(
            out TestFrameworkName framework,
            Instruction instruction,
            TestContext context,
            Tuple<string, string> toolsVersion)
        {
            var platform = context.Platform;

            var buildType = context.BuildType;

            var properties = new Dictionary<string, string>()
            {
                { "Platform", GetPlatform(platform) },
                { "Configuration", GetConfiguration(platform, buildType) }
            };

            var vsPlatform = properties["Platform"];

            if (NeedsOasis(vsPlatform))
            {
                var format = context.ApplicationProgramFormat;

                format = GetApplicationProgramFormat(format);

                properties["ApplicationProgramFormat"] = format;
            }

            var project = new ProjectInstance(
                context.TargetPath,
                properties,
                toolsVersion.Item1,
                toolsVersion.Item2,
                ProjectCollection.GlobalProjectCollection);

            var buildTarget = project
                .Properties
                .FirstOrDefault(x => x.Name == "NintendoSdkBuildTarget");

            instruction.Platform =
                (buildTarget != null) ? buildTarget.EvaluatedValue
                                      : vsPlatform;

            var vsBuildType = project
                .Properties
                .FirstOrDefault(x => x.Name == "NintendoSdkBuildType");

            instruction.BuildType =
                (vsBuildType != null) ? vsBuildType.EvaluatedValue
                                      : properties["Configuration"];

            var metas = project.ItemDefinitions["Link"].Metadata;

            var targetPath = metas
                .FirstOrDefault(x => x.Name == "OutputFile")
                .EvaluatedValue;

            instruction.TargetPath = Path.GetFullPath(targetPath);

            var dependencies = metas
                .FirstOrDefault(x => x.Name == "AdditionalDependencies")
                .EvaluatedValue.Split(';');

            if (dependencies.Contains("libgtest.lib") ||
                dependencies.Contains("libnnt_gtest.a"))
            {
                framework = TestFrameworkName.GoogleTest;
            }
            else if (MsTest.Supports(project))
            {
                framework = TestFrameworkName.MSTest;
            }
            else
            {
                framework = TestFrameworkName.None;
            }
        }

        private static void AppendGtestOutput(
            Instruction instruction, TestContext context)
        {
            var reportPath = Utility.RemoveExtension(
                context.Path, ExtensionDefinition.TestResult) +
                GoogleTestXmlRule.Extention;

            if (instruction.Option != string.Empty)
            {
                instruction.Option += " ";
            }

            instruction.Option += "--gtest_output=\"xml:" + reportPath + "\"";

            if (instruction.ReportPath == string.Empty)
            {
                instruction.ReportPath = reportPath;
            }
        }

        private static void UpdateInstructionForVsTest(
            Instruction instruction, TestContext context)
        {
            var vsTestPath = string.Empty;

            try
            {
                vsTestPath = MsTest.VsTestConsolePath;
            }
            catch (Exception ex)
            {
                context.ResultCode = ResultCode.NO_TOOL;

                context.ErrorMessage = ex.Message;

                throw new TestContextException(ex.Message, context);
            }

            var runSettings = string.Empty;

            var runSettingsPath = string.Empty;

            var resultDirectoryPath = string.Empty;

            try
            {
                MsTest.GenerateRunSettings(
                    out runSettings,
                    out runSettingsPath,
                    out resultDirectoryPath,
                    context.Path,
                    context.Platform,
                    context.Option);
            }
            catch (Exception ex)
            {
                context.ResultCode = ResultCode.ERROR;

                context.ErrorMessage = ex.Message;

                throw new TestContextException(ex.Message, context);
            }

            instruction.Command = vsTestPath;

            instruction.Option = MsTest.GetOption(
                instruction.TargetPath, context.Parameter, runSettingsPath);

            instruction.RunSettings = runSettings;

            if (string.IsNullOrEmpty(instruction.ReportPath))
            {
                instruction.ReportPath =
                    MsTest.GetReportPath(resultDirectoryPath);
            }

            instruction.RunSettingsPath = runSettingsPath;
        }

        private static void UpdateTestContext(
            TestContext context, Instruction instruction)
        {
            context.Option = instruction.Option;

            context.Platform = instruction.Platform;

            context.BuildType = instruction.BuildType;

            context.TargetPath = instruction.TargetPath;

            context.TargetProjectPath = instruction.TargetProjectPath;

            context.ReportPath = instruction.ReportPath;
        }

        private Instruction GenerateFromVcProjectFile(
            out TestFrameworkName framework, TestContext context)
        {
            Tuple<string, string> toolsVersion = null;

            try
            {
                toolsVersion = new Tuple<string, string>(
                    VisualStudio.ToolsVersion,
                    VisualStudio.SubToolsetVersion);
            }
            catch (Exception ex)
            {
                context.ResultCode = ResultCode.NO_TOOL;

                context.ErrorMessage = ex.Message;

                throw new TestContextException(ex.Message, context);
            }

            if (NeedsOasis(GetPlatform(context.Platform)) &&
                this.OasisVersion.Value == null)
            {
                var message = "Oasis is not installed";

                context.ResultCode = ResultCode.NO_TOOL;

                context.ErrorMessage = message;

                throw new TestContextException(message, context);
            }

            var instruction = new Instruction();

            try
            {
                ParseProject(
                    out framework, instruction, context, toolsVersion);
            }
            catch (Exception ex)
            {
                context.ResultCode = ResultCode.WRONG_FILE;

                context.ErrorMessage = ex.Message;

                throw new TestContextException(ex.Message, context, ex);
            }

            return instruction;
        }

        private Instruction GenerateFromEpiFile(
            out TestFrameworkName framework, TestContext context)
        {
            var instruction = new Instruction();

            try
            {
                var epiManager = new EpiManager();

                var epi = epiManager.GetEpi(context.TargetEpiPath);

                var format = context.ApplicationProgramFormat;

                if (string.IsNullOrEmpty(format))
                {
                    instruction.TargetPath = epi.ExecutableFiles[0].Path;
                }
                else if (format == ApplicationProgramFormatDefinition.Raw)
                {
                    var file = epi.ExecutableFiles.FirstOrDefault(
                        x => x.Format == ExecutableFileFormat.NspdRoot);

                    if (file != null)
                    {
                        instruction.TargetPath = file.Path;
                    }
                }
                else if (format == ApplicationProgramFormatDefinition.Nsp)
                {
                    var file = epi.ExecutableFiles.FirstOrDefault(
                        x => x.Format == ExecutableFileFormat.Nsp);

                    if (file != null)
                    {
                        instruction.TargetPath = file.Path;
                    }
                }

                if (string.IsNullOrEmpty(instruction.TargetPath))
                {
                    throw new TestRunnerException(
                        $"ApplicationProgramFormat '{format}' is not supported");
                }

                instruction.Platform = epi.Platform;

                instruction.BuildType = epi.BuildType;

                if (epi.TestFrameworks.Length == 0)
                {
                    framework = TestFrameworkName.None;
                }
                else
                {
                    switch (epi.TestFrameworks[0].Name)
                    {
                        case TestFrameworkName.GoogleTest:
                            framework = TestFrameworkName.GoogleTest;
                            break;

                        case TestFrameworkName.MSTest:
                            framework = TestFrameworkName.MSTest;
                            break;

                        default:
                            framework = TestFrameworkName.None;
                            break;
                    }
                }
            }
            catch (Exception ex)
            {
                context.ResultCode = ResultCode.WRONG_FILE;

                context.ErrorMessage = ex.Message;

                throw new TestContextException(ex.Message, context, ex);
            }

            return instruction;
        }
    }
}
