﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using YamlDotNet.Core;
    using YamlDotNet.RepresentationModel;

    /// <summary>
    /// 実行情報ファイルを管理します。
    /// </summary>
    internal sealed class EpiManager
    {
        private const string EpiName = "executable product info";

        private IReadOnlyList<string> epiPaths = new List<string>();

        /// <summary>
        /// EpiManager クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal EpiManager()
        {
        }

        /// <summary>
        /// 実行情報ファイルのファイルパスを読み込みます。
        /// </summary>
        /// <param name="epiDirectory">実行情報ファイルの読み込み元となるディレクトリの絶対パスです。</param>
        internal void LoadEpiPaths(string epiDirectory)
        {
            this.epiPaths = Directory.EnumerateFiles(
                epiDirectory, "*.yml", SearchOption.TopDirectoryOnly).ToList();
        }

        /// <summary>
        /// 指定されたプログラム名と対応する実行情報ファイルの候補を返します。
        /// </summary>
        /// <param name="programName">プログラム名です。</param>
        /// <param name="platform">プラットフォームです。</param>
        /// <param name="buildType">ビルドタイプです。</param>
        /// <returns>実行情報ファイルの候補です。</returns>
        internal string[] GetEpiPaths(
            string programName, string platform, string buildType)
        {
            var paths = this.epiPaths.Where(
                x => programName == GetProgramName(Path.GetFileName(x))
                ).ToList();

            if (paths.Count == 0)
            {
                return new string[0];
            }

            var epiPlatform = EpiPlatform.Convert(
                !string.IsNullOrEmpty(platform)
                    ? platform
                    : BuildSystem.DefaultPlatform);

            var epiBuildType = !string.IsNullOrEmpty(buildType)
                ? buildType
                : BuildSystem.DefaultBuildType;

            var pcToolBuildType = GetPcToolBuildType(epiBuildType);

            var configs = new List<Tuple<string, string>>();

            if (!EpiPlatform.GetPcToolPlatforms().Contains(platform))
            {
                configs.Add(
                    new Tuple<string, string>(epiPlatform, epiBuildType));
            }
            else
            {
                if (!string.IsNullOrEmpty(buildType))
                {
                    configs.Add(
                        new Tuple<string, string>(platform, buildType));
                }

                if (buildType != pcToolBuildType)
                {
                    configs.Add(
                        new Tuple<string, string>(platform, pcToolBuildType));
                }
            }

            foreach (var pcToolPlatform in EpiPlatform.GetPcToolPlatforms())
            {
                if (pcToolPlatform == platform)
                {
                    continue;
                }

                if (!string.IsNullOrEmpty(buildType))
                {
                    configs.Add(
                        new Tuple<string, string>(pcToolPlatform, buildType));
                }

                if (buildType != pcToolBuildType)
                {
                    configs.Add(
                        new Tuple<string, string>(
                            pcToolPlatform, pcToolBuildType));
                }
            }

            foreach (var config in configs)
            {
                var extension = $".{config.Item1}.{config.Item2}.yml";

                var filteredPaths = paths.Where(
                    path => Path.GetFileName(path).EndsWith(extension)
                    ).ToArray();

                if (filteredPaths.Length > 0)
                {
                    return filteredPaths;
                }
            }

            return new string[0];
        }

        /// <summary>
        /// 実行情報を指定されたファイルから取得します。
        /// </summary>
        /// <param name="path">実行情報ファイルの絶対パスです。</param>
        /// <returns>実行情報です。</returns>
        internal Epi GetEpi(string path)
        {
            var ys = new YamlStream();

            using (var reader = File.OpenText(path))
            {
                try
                {
                    ys.Load(reader);
                }
                catch (YamlException ex)
                {
                    throw new TestRunnerException(
                        $"Failed to load '{path}': {ex.Message}", ex);
                }
            }

            if (ys.Documents.Count == 0)
            {
                throw new TestRunnerException(
                    $"The {EpiName} is empty: {path}");
            }

            if (!(ys.Documents[0].RootNode is YamlMappingNode))
            {
                throw new TestRunnerException(
                    $"The {EpiName} has invalid format: {path}");
            }

            Epi epi = ParseEpi(
                path, (YamlMappingNode)(ys.Documents[0].RootNode));

            return epi;
        }

        private static string GetProgramName(string fileName)
        {
            int index = fileName.LastIndexOf('.');

            index = fileName.LastIndexOf('.', index - 1);

            index = fileName.LastIndexOf('.', index - 1);

            return fileName.Substring(0, index);
        }

        private static string GetPcToolBuildType(string buildType)
        {
            switch (buildType)
            {
                case BuildTypeDefinition.Develop:
                    return BuildTypeDefinition.Release;

                default:
                    return buildType;
            }
        }

        private static Epi ParseEpi(string path, YamlMappingNode node)
        {
            var epi = new Epi();

            var splittedPath = path.Split('.');

            epi.Platform = splittedPath[splittedPath.Length - 3];

            var key = "BuildType";

            try
            {
                var buildTypeNode = (YamlScalarNode)node[key];

                epi.BuildType = buildTypeNode.Value;
            }
            catch (Exception ex)
            {
                throw new TestRunnerException(
                    $"The {EpiName} doesn't have valid '{key}': {path}", ex);
            }

            epi.ExecutableFiles = ParseExecutableFiles(path, node);

            epi.TestFrameworks = ParseTestFrameworks(path, node);

            epi.Resources = ParseResources(path, node);

            return epi;
        }

        private static ExecutableFile[] ParseExecutableFiles(
            string path, YamlMappingNode parent)
        {
            YamlSequenceNode nodes;

            var key = "ExecutableFiles";

            try
            {
                nodes = (YamlSequenceNode)parent[key];
            }
            catch (Exception ex)
            {
                throw new TestRunnerException(
                    $"The {EpiName} doesn't have valid '{key}': {path}", ex);
            }

            var files = nodes.Select(x => ParseExecutableFile(path, x)).ToList();

            if (files.Count == 0)
            {
                throw new TestRunnerException($"'{key}' is empty: {path}");
            }

            files.Sort((lhs, rhs) => {
                if (lhs.Format != ExecutableFileFormat.NspdRoot &&
                    rhs.Format == ExecutableFileFormat.NspdRoot)
                {
                    return -1;
                }

                if (lhs.Format == ExecutableFileFormat.NspdRoot &&
                    rhs.Format != ExecutableFileFormat.NspdRoot)
                {
                    return 1;
                }

                return 0;
            });

            return files.ToArray();
        }

        private static ExecutableFile ParseExecutableFile(
            string path, YamlNode node)
        {
            var name = "ExecutableFile";

            if (!(node is YamlMappingNode))
            {
                throw new TestRunnerException($"'{name}' is invalid: {path}");
            }

            YamlScalarNode formatNode;

            try
            {
                formatNode = (YamlScalarNode)node["Format"];
            }
            catch (Exception ex)
            {
                throw new TestRunnerException(
                    $"'{name}' doesn't have valid 'Format': {path}", ex);
            }

            YamlScalarNode pathNode;

            try
            {
                pathNode = (YamlScalarNode)node["Path"];
            }
            catch (Exception ex)
            {
                throw new TestRunnerException(
                    $"'{name}' doesn't have valid 'Path': {path}", ex);
            }

            var file = new ExecutableFile();

            switch (formatNode.Value)
            {
                case "NspdRoot":
                    file.Format = ExecutableFileFormat.NspdRoot;
                    break;

                case "Nsp":
                    file.Format = ExecutableFileFormat.Nsp;
                    break;

                case "Exe":
                    file.Format = ExecutableFileFormat.Exe;
                    break;

                case "Dll":
                    file.Format = ExecutableFileFormat.Dll;
                    break;

                default:
                    throw new TestRunnerException(
                        $"Format '{formatNode.Value}' is unsupported: {path}");
            }

            file.Path = Path.Combine(BuildSystem.RootPath, pathNode.Value);

            return file;
        }

        private static TestFramework[] ParseTestFrameworks(
            string path, YamlMappingNode parent)
        {
            YamlSequenceNode nodes;

            var key = "TestFrameworks";

            try
            {
                nodes = (YamlSequenceNode)parent[key];
            }
            catch (Exception ex)
            {
                throw new TestRunnerException(
                    $"The {EpiName} doesn't have valid '{key}': {path}", ex);
            }

            return nodes.Select(x => ParseTestFramework(path, x)).ToArray();
        }

        private static TestFramework ParseTestFramework(
            string path, YamlNode node)
        {
            var name = "TestFramework";

            if (!(node is YamlMappingNode))
            {
                throw new TestRunnerException($"'{name}' is invalid: {path}");
            }

            YamlScalarNode nameNode;

            try
            {
                nameNode = (YamlScalarNode)node["Name"];
            }
            catch (Exception ex)
            {
                throw new TestRunnerException(
                    $"'{name}' doesn't have valid 'Name': {path}", ex);
            }

            var versionKey = "RunnerInterfaceVersion";

            YamlScalarNode varsionNode;

            try
            {
                varsionNode = (YamlScalarNode)node[versionKey];
            }
            catch (Exception ex)
            {
                throw new TestRunnerException(
                    $"'{name}' doesn't have valid '{versionKey}': {path}", ex);
            }

            var framework = new TestFramework();

            switch (nameNode.Value)
            {
                case "MSTest":
                    framework.Name = TestFrameworkName.MSTest;
                    break;

                case "xUnit":
                    framework.Name = TestFrameworkName.xUnit;
                    break;

                case "GoogleTest":
                    framework.Name = TestFrameworkName.GoogleTest;
                    break;

                default:
                    throw new TestRunnerException(
                        $"TestFramework '{nameNode.Value}' is unsupported: {path}");
            }

            framework.RunnerInterfaceVersion = varsionNode.Value;

            return framework;
        }

        private static string[] ParseResources(
            string path, YamlMappingNode parent)
        {
            YamlSequenceNode nodes;

            var key = "Resources";

            try
            {
                nodes = (YamlSequenceNode)parent[key];
            }
            catch (Exception ex)
            {
                throw new TestRunnerException(
                    $"The {EpiName} doesn't have valid '{key}': {path}", ex);
            }

            return nodes.Select(x => ParseResource(path, x)).ToArray();
        }

        private static string ParseResource(
            string path, YamlNode node)
        {
            var name = "Resource";

            if (!(node is YamlScalarNode))
            {
                throw new TestRunnerException($"'{name}' is invalid: {path}");
            }

            return Path.Combine(
                BuildSystem.RootPath, ((YamlScalarNode)node).Value);
        }

        /// <summary>
        /// 実行ファイルの情報です。
        /// </summary>
        internal class ExecutableFile
        {
            /// <summary>
            /// 実行ファイルの形式です。
            /// </summary>
            internal ExecutableFileFormat Format { get; set; }

            /// <summary>
            /// 実行ファイルのファイルパスです。
            /// </summary>
            internal string Path { get; set; }
        }

        /// <summary>
        /// テストフレームワークの情報です。
        /// </summary>
        internal class TestFramework
        {
            /// <summary>
            /// テストフレームワークの名前です。
            /// </summary>
            internal TestFrameworkName Name { get;  set; }

            /// <summary>
            /// テストフレームワークの互換性を示すバージョンです。
            /// </summary>
            internal string RunnerInterfaceVersion { get; set; }
        }

        /// <summary>
        /// 実行情報です。
        /// </summary>
        internal class Epi
        {
            /// <summary>
            /// ビルドターゲットです。
            /// </summary>
            internal string Platform { get; set; }

            /// <summary>
            /// ビルドタイプです。
            /// </summary>
            internal string BuildType { get; set; }

            /// <summary>
            /// 実行ファイルの情報です。
            /// </summary>
            internal ExecutableFile[] ExecutableFiles { get; set; }

            /// <summary>
            /// テストフレームワークの情報です。
            /// </summary>
            internal TestFramework[] TestFrameworks { get; set; }

            /// <summary>
            /// 実行に必要となるファイルのパスです。
            /// </summary>
            internal string[] Resources { get; set; }
        }

        private static class EpiPlatform
        {
            private const string AnyCpu = "AnyCPU";

            private const string Win32V140 = "Win32-v140";

            private const string Win32V141 = "Win32-v141";

            private const string X64V140 = "x64-v140";

            private const string X64V141 = "x64-v141";

            private const string NxWin32V140 = "NX-Win32-v140";

            private const string NxWin32V141 = "NX-Win32-v141";

            private const string NxX64V140 = "NX-x64-v140";

            private const string NxX64V141 = "NX-x64-v141";

            private const string NxFp2A32 = "NX-NXFP2-a32";

            private const string NxFp2A64 = "NX-NXFP2-a64";

            private static readonly IReadOnlyCollection<string>
                PcToolPlatforms = new[]
            {
                AnyCpu,
                PlatformDefinition.X86,
                PlatformDefinition.X64,
            };

            private static readonly IReadOnlyDictionary<string, string> Map =
                new Dictionary<string, string>()
            {
                { PlatformDefinition.Win32Vs2015, Win32V140 },
                { PlatformDefinition.Win32Vs2017, Win32V141 },
                { PlatformDefinition.Win64Vs2015, X64V140 },
                { PlatformDefinition.Win64Vs2017, X64V141 },
                { PlatformDefinition.NxWin32Vs2015, NxWin32V140 },
                { PlatformDefinition.NxWin32Vs2017, NxWin32V141 },
                { PlatformDefinition.NxWin64Vs2015, NxX64V140 },
                { PlatformDefinition.NxWin64Vs2017, NxX64V141 },
                { PlatformDefinition.Nx32, NxFp2A32 },
                { PlatformDefinition.Nx64, NxFp2A64 },
                { PlatformDefinition.NxFp2, NxFp2A32 },
                { PlatformDefinition.NxFp2A32, NxFp2A32 },
                { PlatformDefinition.NxFp2A64, NxFp2A64 },
            };

            internal static IReadOnlyCollection<string> GetPcToolPlatforms()
            {
                return PcToolPlatforms;
            }

            internal static string Convert(string platform)
            {
                string epiPlatform;

                if (Map.TryGetValue(platform, out epiPlatform))
                {
                    return epiPlatform;
                }
                else
                {
                    return platform;
                }
            }
        }
    }
}
