﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner.Executer
{
    using System;
    using System.Diagnostics;
    using System.IO;
    using System.Runtime.InteropServices;

    /// <summary>
    /// MiniDump 形式のクラッシュダンプ用の関数を定義します。
    /// </summary>
    internal static class MiniDump
    {
        /// <summary>
        /// クラッシュダンプを作成します。
        /// </summary>
        /// <param name="filePath">ファイルパスです。</param>
        /// <param name="process">プロセスです。</param>
        internal static void Create(string filePath, Process process)
        {
            var mode = FileMode.Create;

            var fileAccess = FileAccess.Write;

            using (var fs = new FileStream(filePath, mode, fileAccess))
            {
                var isSuccess = MiniDumpWriteDump(
                    process.Handle, (uint)process.Id,
                    fs.SafeFileHandle,
                    (uint)DumpType.WithFullMemory,
                    IntPtr.Zero, IntPtr.Zero, IntPtr.Zero);

                if (!isSuccess)
                {
                    int errorCode = Marshal.GetHRForLastWin32Error();

                    Marshal.ThrowExceptionForHR(errorCode);
                }
            }
        }

        [DllImport("dbghelp.dll", SetLastError = true)]
        private static extern bool MiniDumpWriteDump(
            IntPtr hProcess, uint processId,
            SafeHandle hFile,
            uint dumpType,
            IntPtr expParam, IntPtr userStreamParam, IntPtr callbackParam);

        [Flags]
        private enum DumpType : uint
        {
            Normal = 0x00000000,
            WithDataSegs = 0x00000001,
            WithFullMemory = 0x00000002,
            WithHandleData = 0x00000004,
            FilterMemory = 0x00000008,
            ScanMemory = 0x00000010,
            WithUnloadedModules = 0x00000020,
            WithIndirectlyReferencedMemory = 0x00000040,
            FilterModulePaths = 0x00000080,
            WithProcessThreadData = 0x00000100,
            WithPrivateReadWriteMemory = 0x00000200,
            WithoutOptionalData = 0x00000400,
            WithFullMemoryInfo = 0x00000800,
            WithThreadInfo = 0x00001000,
            WithCodeSegs = 0x00002000,
            WithoutAuxiliaryState = 0x00004000,
            WithFullAuxiliaryState = 0x00008000,
            WithPrivateWriteCopyMemory = 0x00010000,
            IgnoreInaccessibleMemory = 0x00020000,
            WithTokenInformation = 0x00040000,
            WithModuleHeaders = 0x00080000,
            FilterTriage = 0x00100000,
            ValidTypeFlags = 0x001fffff,
        }
    }
}
