﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace SystemUpdaterNx.MuteAudio
{
    using System.ComponentModel.Composition;
    using System.IO;
    using System.Text;
    using System.Threading;
    using SystemUpdaterNx.Common;
    using TargetShell.Library;
    using TargetShell.CommandPluginBase;
    using TargetShell.DevicePluginBase;

    /// <summary>
    /// 音声のミュートを行うクラス
    /// </summary>
    [Export(typeof(IDevicePluginInterface<CommandParameter>))]
    public class Nx :
            BaseDevicePlugin<CommandParameter>, IDevicePluginInterface<CommandParameter>
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Nx()
        {
            this.ParallelModeFlag = true;
        }

        /// <summary>
        /// デバイスプラグイン実行メソッド
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <param name="successDeviceCount">成功デバイスの数</param>
        protected override void SpecificProcessDevicePlugin(
                CommandParameter parameter, ref int successDeviceCount)
        {
            try
            {
                // コマンドの構築
                var runOnTargetCommand = CreateRunOnTargetCommand(parameter);
                // 音声のミュートを行う
                var returnCode = TargetShellLibrary.InvokeExe(
                        Path.Combine(parameter.NintendoSdkRoot,
                                ConstantsNx.RunOnTargetExe),
                        runOnTargetCommand.ToString(), parameter.SerialDirectroy,
                        parameter.TargetEntry.SerialNumber, CommonConstants.OutputLogName);
                if (returnCode)
                {
                    Interlocked.Increment(ref successDeviceCount);
                }
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
            }
        }

        /// <summary>
        /// 音声ミュートコマンドを構築
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <returns>コマンド文字列</returns>
        private string CreateRunOnTargetCommand(CommandParameter parameter)
        {
            var runOnTargetCommandBuilder = new StringBuilder();
            // DevMenuCommand.nsp設定
            runOnTargetCommandBuilder.AppendFormat(" {0}", Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsNx.DevMenuCommandNsp));
            // 詳細ログ
            if (parameter.Verbose)
            {
                runOnTargetCommandBuilder.AppendFormat(" --verbose");
            }
            // 要求する開発機指定
            if (!string.IsNullOrEmpty(parameter.TargetEntry.SerialNumber))
            {
                runOnTargetCommandBuilder.AppendFormat(" --target {0}",
                        parameter.TargetEntry.SerialNumber);
            }
            // 音声のミュートコマンド
            runOnTargetCommandBuilder.AppendFormat(" -- audio mute");
            return runOnTargetCommandBuilder.ToString();
        }
    }
}
