﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace PublicNx.ExportSetting
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel.Composition;
    using System.IO;
    using System.Text;
    using System.Threading;
    using System.Threading.Tasks;
    using PublicNx.SettingCommon;
    using TargetShell.Library;
    using TargetShell.PluginInterface;

    /// <summary>
    /// サブコマンドプラグイン
    /// </summary>
    [Export(typeof(ITargetShellPlugin))]
    public class ExportSetting : SettingsCommon, ITargetShellPlugin
    {
        /// <summary>
        /// サブコマンドを実行する
        /// </summary>
        /// <param name="subCommandArgument">サブコマンド引数</param>
        /// <param name="deviceList">デバイスのリスト</param>
        /// <returns>true：成功  false：失敗</returns>
        public bool InvokePlugin(ISubCommandArgument subCommandArgument,
                List<TargetEntry> deviceList)
        {
            Console.WriteLine("ExportSetting start.");
            var successDeviceCount = 0;
            var totalDevices = deviceList.Count;
            try
            {
                var parameter = PrepareSettingPlugin(subCommandArgument);
                successDeviceCount = Settings(deviceList, parameter);
                Console.WriteLine(
                        "Command Result success {0}/{1}", successDeviceCount, totalDevices);
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
                return false;
            }
            Console.WriteLine("ExportSetting end.");
            return ((successDeviceCount == totalDevices) ? true : false);
        }

        /// <summary>
        /// サブコマンド引数取得
        /// <returns>サブコマンド引数</returns>
        /// </summary>
        public ISubCommandArgument GetSubCommandArgument()
        {
            var argument = new SubCommandArgument(SettingCommonConstants.ExportSubCommand);
            return argument;
        }

        /// <summary>
        /// 設定のエクスポートを行う
        /// </summary>
        /// <param name="deviceList">デバイスリストのパラメータ</param>
        /// <param name="parameter">コマンドパラメータ</param>
        /// <param name="logOutputDir">シリアルフォルダパス</param>
        /// <returns>成功したデバイスの数</returns>
        public int Settings(List<TargetEntry> deviceList, CommandParameter parameter)
        {
            // 成功したデバイスの数
            var successDeviceCount = 0;

            // SdevとEdevを取得する
            var deviceNx = deviceList.FindAll(
                    device => device.HardwareType == "Sdev" || device.HardwareType == "Edev");

            // 設定ファイル保存ディレクトリの確認
            CheckSettingOutputDirectory(parameter);

            // コマンドを並列処理で実行する
            Parallel.For(0, deviceNx.Count, id =>
            {
                var localParameter = new CommandParameter
                {
                    TargetEntry = deviceNx[id],
                    Verbose = parameter.Verbose,
                    NintendoSdkRoot = parameter.NintendoSdkRoot,
                    DirectoryPath = parameter.DirectoryPath
                };
                // ログ出力先を設定している場合のみログフォルダを作成する
                localParameter.SerialDirectroy =
                    TargetShellLibrary.CreateSerialNumberLogDirectory(parameter.LogOutputDir,
                    localParameter.TargetEntry.SerialNumber);
                SpecificProcessDevicePlugin(localParameter, ref successDeviceCount);
            });
            return successDeviceCount;
        }

        /// <summary>
        /// デバイスプラグイン実行メソッド
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <param name="successDeviceCount">成功デバイスの数</param>
        private void SpecificProcessDevicePlugin(
                CommandParameter parameter, ref int successDeviceCount)
        {
            try
            {
                // コマンドの構築
                var settingsManagerCommand = CreateSettingsManagerCommand(parameter);
                // コマンドの実行を行う
                var returnCode = TargetShellLibrary.InvokeExe(
                        Path.Combine(parameter.NintendoSdkRoot,
                        SettingCommonConstants.SettingsManager),
                        settingsManagerCommand.ToString(),
                        parameter.SerialDirectroy,
                        parameter.TargetEntry.SerialNumber,
                        CommonConstants.OutputLogName);
                if (returnCode)
                {
                    Interlocked.Increment(ref successDeviceCount);
                }
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
            }
        }

        /// <summary>
        /// コマンドを構築
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <returns>コマンド文字列</returns>
        private string CreateSettingsManagerCommand(CommandParameter parameter)
        {
            var settingsManagerCommandBuilder = new StringBuilder();
            settingsManagerCommandBuilder.AppendFormat(" Export");
            // 要求する開発機指定
            if (!string.IsNullOrEmpty(parameter.TargetEntry.SerialNumber))
            {
                settingsManagerCommandBuilder.AppendFormat(" --target {0}",
                        parameter.TargetEntry.SerialNumber);
            }
            // 出力パス設定
            var outputDirectory = string.Empty;
            if (string.IsNullOrEmpty(parameter.DirectoryPath))
            {
                // ディレクトリが指定されていなかったらSDKルートにTargetShellSettingに出力する
                outputDirectory = Path.Combine(parameter.NintendoSdkRoot,
                        SettingCommonConstants.ExportDirectory);
                settingsManagerCommandBuilder.AppendFormat(" {0}",
                        Path.Combine(
                        outputDirectory, parameter.TargetEntry.SerialNumber + ".json"));
            }
            else
            {
                // ディレクトリが設定されている場合
                outputDirectory = parameter.DirectoryPath;
                settingsManagerCommandBuilder.AppendFormat(" {0}",
                        Path.Combine(parameter.DirectoryPath,
                        parameter.TargetEntry.SerialNumber + ".json"));
            }
            // ディレクトリがなければ作成する
            if (!Directory.Exists(outputDirectory))
            {
                Directory.CreateDirectory(outputDirectory);
            }
            return settingsManagerCommandBuilder.ToString();
        }

        /// <summary>
        /// 設定ファイル出力先ディレクトリを確認する
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        private void CheckSettingOutputDirectory(CommandParameter parameter)
        {
            var outputDirectory = parameter.DirectoryPath;
            if (string.IsNullOrEmpty(outputDirectory))
            {
                // パス未設定の場合
                outputDirectory = Path.Combine(parameter.NintendoSdkRoot,
                        SettingCommonConstants.ExportDirectory);
            }
            if (Directory.Exists(outputDirectory))
            {
                BackupSettingFile(outputDirectory);
            }
            else
            {
                Directory.CreateDirectory(outputDirectory);
            }
            return;
        }

        /// <summary>
        /// 設定ファイルをバックアップする
        /// </summary>
        /// <param name="outputDirectory">出力先ディレクトリ</param>
        private void BackupSettingFile(string outputDirectory)
        {
            var files = Directory.GetFiles(outputDirectory, "*",
                    SearchOption.TopDirectoryOnly);
            if (files.Length == 0)
            {
                // ファイルが1つもなければバックアップ処理なしで終了
                return;
            }
            var backupPath = Path.Combine(outputDirectory,
                    "backup" + DateTime.Now.ToString("yyyyMMddHHmmss"));
            Directory.CreateDirectory(backupPath);
            foreach (var file in files)
            {
                var destFileName = Path.Combine(backupPath, Path.GetFileName(file));
                File.Move(file, destFileName);
            }
            return;
        }
    }
}
