﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TargetShell
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using TargetShell.PluginInterface;

    /// <summary>
    /// コマンドラインツールを使用するときの定義
    /// </summary>
    public class CommandLineToolConstants
    {
        /// <summary>
        /// ControlTargetの実行ファイル名
        /// </summary>
        public const string ControlTargetExeName = @"ControlTarget.exe";
        /// <summary>
        /// ControlTargetのパス
        /// </summary>
        public const string ControlTargetPath = @"Tools\CommandLineTools";
        /// <summary>
        /// 既定のターゲットが設定されていない場合の返却文字列
        /// </summary>
        public const string NoDefaultTarget = @"no default target.";
    }

    /// <summary>
    /// TargetShellのユーティリティクラス
    /// </summary>
    public class Utility
    {
        /// <summary>
        /// 打刻フォーマット
        /// </summary>
        public const string TimeStampFormat = @"yyyy/MM/dd HH:mm:ss.fff";
        /// <summary>
        /// サブコマンドの区切り文字
        /// </summary>
        private const string SeparationSubCommandWord = "--";
        /// <summary>
        /// 引数値の区切り文字
        /// </summary>
        private const string SeparationValueWord = "=";
        /// <summary>
        /// Registeredオプション文字
        /// </summary>
        private const string OptionRegisteredWord = "--registered";
        /// <summary>
        /// Nameオプション文字
        /// </summary>
        private const string OptionNameWord = "--name";
        /// <summary>
        /// Serialnumberオプション文字
        /// </summary>
        private const string OptionSerialnumberWord = "--serial-number";
        /// <summary>
        /// Ipaddressオプション文字
        /// </summary>
        private const string OptionIpaddressWord = "--ip-address";
        /// <summary>
        /// Hwtypeオプション文字
        /// </summary>
        private const string OptionHwtypeWord = "--hw-type";
        /// <summary>
        /// Hwconfigオプション文字
        /// </summary>
        private const string OptionHwconfigWord = "--hw-config";
        /// <summary>
        /// Connectionpathオプション文字
        /// </summary>
        private const string OptionConnectionpathWord = "--connection-path";
        /// <summary>
        /// detect-deviceオプション文字
        /// </summary>
        private const string OptionDetectDeviceWord = "--detect-device";
        /// <summary>
        /// NotSpecified引数文字列
        /// </summary>
        private const string ValueNotSpecifiedWord = "NotSpecified";
        /// <summary>
        /// Sdev引数文字列
        /// </summary>
        private const string ValueSdevWord = "Sdev";
        /// <summary>
        /// Edev引数文字列
        /// </summary>
        private const string ValueEdevWord = "Edev";
        /// <summary>
        /// HostBridge引数文字列
        /// </summary>
        private const string ValueHostBridgeWord = "HostBridge";
        /// <summary>
        /// Usb引数文字列
        /// </summary>
        private const string ValueUsbWord = "Usb";
        /// <summary>
        /// NintendoSdkRootを検索する文字列
        /// </summary>
        public const string NintendoSdkRootMark = @"NintendoSdkRootMark";
        /// <summary>
        /// NintendoSdkRoot
        /// </summary>
        public static string NintendoSdkRootPath { get; set; }
        /// <summary>
        /// DebugLogFlig
        /// </summary>
        public static bool DebugLogFlag { get; set; }

        /// <summary>
        /// コマンド分割処理
        /// </summary>
        internal class AddCommandArrayListTakeOverOption
        {
            /// <summary>
            /// registeredの引き継ぎオプション保存変数
            /// </summary>
            private string registeredString;
            /// <summary>
            /// nameの引き継ぎオプション保存変数
            /// </summary>
            private string nameString;
            /// <summary>
            /// serialnumberの引き継ぎオプション保存変数
            /// </summary>
            private string serialnumberString;
            /// <summary>
            /// ipaddressの引き継ぎオプション保存変数
            /// </summary>
            private string ipaddressString;
            /// <summary>
            /// hwtypeの引き継ぎオプション保存変数
            /// </summary>
            private string hwtypeString;
            /// <summary>
            /// hwconfigの引き継ぎオプション保存変数
            /// </summary>
            private string hwconfigString;
            /// <summary>
            /// connectionpathの引き継ぎオプション保存変数
            /// </summary>
            private string connectionpathString;

            /// <summary>
            /// コンストラクタ（メンバー変数を初期化する）
            /// </summary>
            public AddCommandArrayListTakeOverOption()
            {
                this.registeredString = string.Empty;
                this.nameString = string.Empty;
                this.serialnumberString = string.Empty;
                this.ipaddressString = string.Empty;
                this.hwtypeString = string.Empty;
                this.hwconfigString = string.Empty;
                this.connectionpathString = string.Empty;
            }

            /// <summary>
            /// コマンドリストに追加する
            /// </summary>
            /// <param name="commandList">コマンドリスト</param>
            /// <param name="oneCommand">追加するコマンド</param>
            /// <param name="takeOverOption">引き継ぎオプション</param>
            public void AddToCommandArrayList(
                    List<string[]> commandArrayList, List<string> oneCommandList)
            {
                TakeOverOptionParam(oneCommandList);
                commandArrayList.Add(oneCommandList.ToArray());
                SetTakeOverOptionParam(oneCommandList);
            }

            /// <summary>
            /// コマンドのオプションの引き継がせる
            /// -- で連結されたサブコマンドにオプションが設定されていない場合
            /// 前のサブコマンドのオプションを引き継げるようにcommandListに追加する
            /// -- で連結されたサブコマンドにオプションが設定されている場合
            /// 設定されているオプションが優先される
            /// </summary>
            /// <param name="commandList">コマンドリスト</param>
            private void TakeOverOptionParam(List<string> commandList)
            {
                if (!(string.IsNullOrEmpty(this.registeredString))
                        && (!IsIncludeOption(commandList, OptionRegisteredWord)))
                {
                    commandList.Add(this.registeredString);
                }
                if (!(string.IsNullOrEmpty(this.nameString))
                        && (!IsIncludeOption(commandList, OptionNameWord)))
                {
                    commandList.Add(this.nameString);
                }
                if (!(string.IsNullOrEmpty(this.serialnumberString))
                        && (!IsIncludeOption(commandList, OptionSerialnumberWord)))
                {
                    commandList.Add(this.serialnumberString);
                }
                if (!(string.IsNullOrEmpty(this.ipaddressString))
                        && (!IsIncludeOption(commandList, OptionIpaddressWord)))
                {
                    commandList.Add(this.ipaddressString);
                }
                if (!(string.IsNullOrEmpty(this.hwtypeString))
                        && (!IsIncludeOption(commandList, OptionHwtypeWord)))
                {
                    commandList.Add(this.hwtypeString);
                }
                if (!(string.IsNullOrEmpty(this.hwconfigString))
                        && (!IsIncludeOption(commandList, OptionHwconfigWord)))
                {
                    commandList.Add(this.hwconfigString);
                }
                if (!(string.IsNullOrEmpty(this.connectionpathString))
                        && (!IsIncludeOption(commandList, OptionConnectionpathWord)))
                {
                    commandList.Add(this.connectionpathString);
                }
            }

            /// <summary>
            /// コマンドのオプションの引き継ぎ設定する
            /// 現在のサブコマンドのオプションを引き継げるようにオプション情報を保存する
            /// </summary>
            /// <param name="commandList">コマンドリスト</param>
            private void SetTakeOverOptionParam(List<string> commandList)
            {
                foreach (var commandPart in commandList)
                {
                    if (commandPart.Contains(OptionRegisteredWord))
                    {
                        this.registeredString = commandPart.ToString();
                    }
                    else if (commandPart.Contains(OptionNameWord))
                    {
                        this.nameString = commandPart.ToString();
                    }
                    else if (commandPart.Contains(OptionSerialnumberWord))
                    {
                        this.serialnumberString = commandPart.ToString();
                    }
                    else if (commandPart.Contains(OptionIpaddressWord))
                    {
                        this.ipaddressString = commandPart.ToString();
                    }
                    else if (commandPart.Contains(OptionHwtypeWord))
                    {
                        this.hwtypeString = commandPart.ToString();
                    }
                    else if (commandPart.Contains(OptionHwconfigWord))
                    {
                        this.hwconfigString = commandPart.ToString();
                    }
                    else if (commandPart.Contains(OptionConnectionpathWord))
                    {
                        this.connectionpathString = commandPart.ToString();
                    }
                }
            }

            /// <summary>
            /// コマンドリストに既に登録されているか確認する
            /// </summary>
            /// <param name="commandList">コマンドリスト</param>
            /// <param name="option">オプション</param>
            /// <returns>true:含まれている</returns>
            /// <returns>false:含まれていない</returns>
            private bool IsIncludeOption(List<string> commandList, string option)
            {
                foreach (var oneCommand in commandList)
                {
                    if (oneCommand.Contains(option))
                    {
                        return true;
                    }
                }
                return false;
            }
        }

        /// <summary>
        /// コマンドを分割する
        /// </summary>
        /// <param name="command">コマンド</param>
        /// <returns>解析したコマンドリスト</returns>
        public static List<string[]> DivideMultiCommand(string[] command)
        {
            var commandArrayList = new List<string[]>();
            var oneCommandList = new List<string>();
            var addCommandArrayListTakeOverOption = new AddCommandArrayListTakeOverOption();
            foreach (var commandPart in command)
            {
                // コマンドの区切りで１つのコマンドにする
                if (commandPart.Equals(SeparationSubCommandWord))
                {
                    if (oneCommandList.Count != 0)
                    {
                        addCommandArrayListTakeOverOption.AddToCommandArrayList(
                                commandArrayList, oneCommandList);
                        oneCommandList = new List<string>();
                    }
                    continue;
                }
                oneCommandList.Add(commandPart);
            }
            // コマンドの終端を確認して１つのコマンドにする
            if (oneCommandList.Count != 0)
            {
                addCommandArrayListTakeOverOption.AddToCommandArrayList(
                        commandArrayList, oneCommandList);
            }
            return commandArrayList;
        }

        /// <summary>
        /// 共通コマンド解析して開発機検索条件を取得する
        /// </summary>
        /// <param name="args">引数</param>
        /// <param name="devicePluginInfo">デバイスプラグイン情報</param>
        /// <returns>開発機検索の条件</returns>
        public static SearchCondition SetSearchCondition(ISubCommandArgument subCommandArgument,
                List<DevicePluginInfo> devicePluginInfo)
        {
            // 検索条件を設定
            var searchCondition = new SearchCondition();
            searchCondition.TargetEntry = new SearchTargetEntry();
            searchCondition.DetectDevice = false;
            searchCondition.DevicePluginInfo = devicePluginInfo;
            searchCondition.DetectDevice = subCommandArgument.OptionArgs.CommonArgs.DetectDevice;
            searchCondition.TargetEntry.DeviceName = subCommandArgument.OptionArgs.CommonArgs.Name;
            searchCondition.TargetEntry.SerialNumber = subCommandArgument.OptionArgs.CommonArgs.SerialNumber;
            searchCondition.TargetEntry.IpAddress = subCommandArgument.OptionArgs.CommonArgs.Ipaddress;
            if (!string.IsNullOrEmpty(subCommandArgument.OptionArgs.CommonArgs.HardwareType))
            {
                searchCondition.TargetEntry.HardwareType =
                        subCommandArgument.OptionArgs.CommonArgs.HardwareType;
            }
            searchCondition.TargetEntry.HardwareConfig =
                    subCommandArgument.OptionArgs.CommonArgs.HardwareConfig;
            searchCondition.TargetEntry.ConnectionPath =
                    subCommandArgument.OptionArgs.CommonArgs.ConnectionPath;
            // 検索条件をログ出力
            Console.WriteLine("--Search Condition--");
            Console.WriteLine("  DetectDevice: {0}", searchCondition.DetectDevice);
            Console.WriteLine("  Name: {0}", searchCondition.TargetEntry.DeviceName);
            Console.WriteLine("  SerialNumber: {0}", searchCondition.TargetEntry.SerialNumber);
            Console.WriteLine("  Ipaddress: {0}",
                    string.Join(",", searchCondition.TargetEntry.IpAddress));
            Console.WriteLine("  HardwareType: {0}", searchCondition.TargetEntry.HardwareType);
            Console.WriteLine("  HardwareConfig: {0}",
                searchCondition.TargetEntry.HardwareConfig);
            Console.WriteLine("  ConnectionPath: {0}",
                searchCondition.TargetEntry.ConnectionPath);
            return searchCondition;
        }

        /// <summary>
        /// 共通コマンド解析して開発機検索条件を取得する
        /// </summary>
        /// <param name="args">引数</param>
        /// <param name="devicePluginInfo">デバイスプラグイン情報</param>
        /// <returns>開発機のリスト</returns>
        public static List<TargetEntry> SearchDevice(ISubCommandArgument subCommandArgument,
                List<DevicePluginInfo> devicePluginInfo)
        {
            var searchCondition = SetSearchCondition(subCommandArgument, devicePluginInfo);
            var searchLib = new SearchLib();
            // SearchLibにデバイスプラグインの情報を設定する
            searchLib.SetSearchCondition(searchCondition);
            var searchResult = Result.Fail;
            try
            {
                searchResult = searchLib.Run();
            }
            catch (SearchDeviceException exception)
            {
                Console.WriteLine(exception.Message);
                Environment.Exit(1);
            }
            if (searchResult == Result.Success)
            {
                SearchLib.DumpList(searchLib.DeviceList);
            }
            return searchLib.DeviceList;
        }

        /// <summary>
        /// TargetManagerに登録されているターゲット取得
        /// <param name="devicePluginInfo">デバイスプラグイン情報</param>
        /// <returns>デバイスリスト</returns>
        /// </summary>
        public static List<TargetEntry> GetRegisteredTarget(
                List<DevicePluginInfo> devicePluginInfo)
        {
            var searchLib = new SearchLib();
            // 検索条件指定（TargetManagerに登録されているものなので何も指定しない）
            var searchCondition = new SearchCondition();
            searchCondition.TargetEntry = new SearchTargetEntry();
            searchCondition.DetectDevice = false;
            searchCondition.DevicePluginInfo = devicePluginInfo;
            searchLib.SetSearchCondition(searchCondition);
            var searchResult = Result.Fail;
            try
            {
                // デバイス検索実行
                searchResult = searchLib.Run();
            }
            catch (SearchDeviceException exception)
            {
                // デバイスの検索でエクセプションが発生したらTargetShell終了
                Console.WriteLine(exception.Message);
                Environment.Exit(1);
            }
            return searchLib.DeviceList;
        }

        /// <summary>
        /// TargetManagerへターゲットを再登録
        /// <param name="previousDeviceList">処理実行前のデバイスリスト</param>
        /// <param name="requestedDeviceList">処理要求のデバイスリスト</param>
        /// <param name="devicePluginInfo">デバイスプラグイン情報</param>
        /// <returns>成功:true 失敗:false</returns>
        /// </summary>
        public static bool ReregisterTarget(List<TargetEntry> previousDeviceList,
                List<TargetEntry> requestedDeviceList, List<DevicePluginInfo> devicePluginInfo)
        {
            var result = true;
            // 登録するべきデバイスリスト
            var registDeviceList = new List<TargetEntry>();
            // TargetShell開始時に登録されていたデバイスのリスト取得
            registDeviceList.AddRange(previousDeviceList);
            // プラグイン処理を要求したデバイスのリストを取得
            registDeviceList.AddRange(requestedDeviceList);
            registDeviceList = registDeviceList.Distinct().ToList();
            // 現在の登録状態を確認する
            var nowDeviceList = GetRegisteredTarget(devicePluginInfo);
            // 登録すべきデバイス取得する
            foreach (var oneNowDevice in nowDeviceList)
            {
                // 重複登録にならないようにすでに登録されているものは除く
                registDeviceList.RemoveAll(
                        deviceList => deviceList.SerialNumber.Equals(oneNowDevice.SerialNumber));
            }
            // 登録する
            foreach (var oneRegistDevice in registDeviceList)
            {
                if (!RegisterTarget(oneRegistDevice.SerialNumber))
                {
                    // エラーが発生したらログだけ残しておく
                    Console.WriteLine("ReregisterTarget: RegisterTarget err " +
                            oneRegistDevice.SerialNumber);
                    result = false;
                }
            }
            return result;
        }

        /// <summary>
        /// NintendoSdkRootのパスを取得する
        /// </summary>
        /// <returns>NintendoSdkRootのパス</returns>
        public static string GetNintendoSdkRootPath()
        {
            var nintendoSdkRootDirectory = new DirectoryInfo(Path.GetDirectoryName(
                    AppDomain.CurrentDomain.BaseDirectory));
            while (!File.Exists(Path.Combine(nintendoSdkRootDirectory.FullName,
                    NintendoSdkRootMark)))
            {
                if (nintendoSdkRootDirectory.Parent == null)
                {
                    throw new UtilityException("Not found NintendoSdkRootMark file.");
                }
                nintendoSdkRootDirectory = nintendoSdkRootDirectory.Parent;
            }
            return nintendoSdkRootDirectory.FullName;
        }

        /// <summary>
        /// TargetManagerへターゲットを登録
        /// <param name="deviceName">登録するデバイスネーム</param>
        /// <param name="mode">登録・削除</param>
        /// <returns>成功:true 失敗:false</returns>
        /// </summary>
        private static bool RegisterTarget(string deviceName)
        {
            var result = false;
            // ControlTargetのパス生成
            var controlTargetExe = CreateControlTargetPath();
            // コマンド生成
            var controlTargetCommandBuilder = new StringBuilder();
            controlTargetCommandBuilder.AppendFormat(
                    " register --verbose");
            controlTargetCommandBuilder.AppendFormat(
                    " --target {0}", deviceName);
            // ControlTargetを実行
            result = InvokeCommandLineTool(controlTargetExe,
                    controlTargetCommandBuilder.ToString());
            return result;
        }

        /// <summary>
        /// 既定の開発機を取得する
        /// <param name="defalutTarget">既定の開発機</param>
        /// <returns>成功:true 失敗:false</returns>
        /// </summary>
        public static bool GetDefaultTarget(out string defalutTarget)
        {
            var result = false;
            defalutTarget = string.Empty;
            // ControlTargetのパス生成
            var controlTargetExe = CreateControlTargetPath();
            // コマンド生成
            var controlTargetCommandBuilder = new StringBuilder();
            controlTargetCommandBuilder.AppendFormat(
                    " get-default");
            // ControlTargetを実行
            var outputString = string.Empty;
            result = InvokeCommandLineTool(controlTargetExe,
                    controlTargetCommandBuilder.ToString(), out outputString);
            // エラーとなる場合、出力が空白、null、スペースのみの場合
            // 既定の開発機が設定されていない場合は、no default target.とする
            if ((!result) || string.IsNullOrWhiteSpace(outputString) ||
                    (outputString.Contains(CommandLineToolConstants.NoDefaultTarget)))
            {
                defalutTarget = CommandLineToolConstants.NoDefaultTarget;
                Console.WriteLine("GetDefaultTarget no default target result = {0}",
                        result.ToString());
                return result;
            }
            // 既定の開発機のシリアル番号のみを取得する
            // ABC00000000\t192.168.XXX.XXX の形式で出力されるので
            // 空白区切りの1つ目を取り出し、さらにタブ区切りの１つ目を取り出す。
            var arrayString = outputString.Split(' ');
            // 出力形式チェック
            if (arrayString.Length == 1)
            {
                // 意図した形式で出力が取得できた場合
                var controlTargetOutputString = arrayString[0];
                if (!string.IsNullOrWhiteSpace(controlTargetOutputString))
                {
                    var controlTargetOutputArrayString = controlTargetOutputString.Split('\t');
                    defalutTarget = controlTargetOutputArrayString[0];
                }
                else
                {
                    // 意図しない文字列が返却されたので失敗とする
                    Console.WriteLine("Invalit DeviceName From ControlTarget");
                    result = false;
                }
            }
            else
            {
                // 意図しない形式で返却されたので失敗とする
                Console.WriteLine("Invalit Output From ontrolTarget");
                result = false;
            }
            return result;
        }

        /// <summary>
        /// 既定の開発機を設定する
        /// <param name="defalutTarget">既定の開発機</param>
        /// <returns>成功:true 失敗:false</returns>
        /// </summary>
        public static bool SetDefaultTarget(string defalutTarget)
        {
            var result = false;
            var nowDefaultTarget = string.Empty;
            // 現在設定されている既定のデバイスを取得する
            if (!GetDefaultTarget(out nowDefaultTarget))
            {
                // 取得に失敗してもログだけ出力し設定の処理は継続させる
                Console.WriteLine("SetDefaultTarget GetDefaultTarget error");
            }
            // 設定できないものが取得された場合はエラーとする
            if (string.IsNullOrWhiteSpace(nowDefaultTarget) ||
                    nowDefaultTarget.Contains(CommandLineToolConstants.NoDefaultTarget))
            {
                Console.WriteLine("SetDefaultTarget GetDefaultTarget InvalitTarget");
                return false;
            }
            // 既定の開発機が変更されていなければ成功で終了する
            if (nowDefaultTarget.Equals(defalutTarget))
            {
                Console.WriteLine("SetDefaultTarget Already Setting DefaultTarget");
                return true;
            }
            // ControlTargetのパス生成
            var controlTargetExe = CreateControlTargetPath();
            // コマンド生成
            var controlTargetCommandBuilder = new StringBuilder();
            controlTargetCommandBuilder.AppendFormat(
                    " set-default");
            controlTargetCommandBuilder.AppendFormat(
                    " --target {0}", defalutTarget);
            // ControlTargetを実行
            result = InvokeCommandLineTool(controlTargetExe,
                    controlTargetCommandBuilder.ToString());
            return result;
        }

        /// <summary>
        /// 接続されているデバイスの情報を取得する
        /// <param name="isDetect">PCに接続されているものすべてを取得するか</param>
        /// <returns>xml形式のデバイス情報</returns>
        /// </summary>
        public static string GetTargetInfo(bool isDetect)
        {
            var targetsInfo = string.Empty;
            var controlTarget = CreateControlTargetPath();
            var controlTargetCommandBuilder = new StringBuilder();
            if (isDetect)
            {
                // PCに接続している全てのデバイスの情報をXML形式で取得
                controlTargetCommandBuilder.AppendFormat(" detect-target --xml");
            }
            else
            {
                // TargetManagerに登録しているデバイスの情報をXML形式で取得
                controlTargetCommandBuilder.AppendFormat(" list-target --xml");
            }
            var result = InvokeCommandLineTool(controlTarget,
                    controlTargetCommandBuilder.ToString(), out targetsInfo);
            return targetsInfo;
        }

        /// <summary>
        /// ControlTargetのパスを生成する
        /// <returns>ControlTargetのパス</returns>
        /// </summary>
        public static string CreateControlTargetPath()
        {
            var controlTargetExe = CommandLineToolConstants.ControlTargetExeName;
            controlTargetExe = Path.Combine(NintendoSdkRootPath,
                    CommandLineToolConstants.ControlTargetPath,
                    controlTargetExe);
            return controlTargetExe;
        }

        /// <summary>
        /// 打刻文字列を取得する
        /// <returns>打刻文字列</returns>
        /// </summary>
        private static string GetTimeStamp()
        {
            return DateTime.Now.ToString(TimeStampFormat);
        }

        /// <summary>
        /// コンソールへのログ出力用のイベントハンドラ取得
        /// <param name="stringBuilder">ストリングビルダ</param>
        /// <returns>イベントハンドラ</returns>
        /// </summary>
        private static DataReceivedEventHandler GetDataReceivedEventHandler(
                StringBuilder stringBuilder)
        {
            var handler = new DataReceivedEventHandler(
                (object obj, DataReceivedEventArgs args) =>
                {
                    if (args.Data == null)
                    {
                        return;
                    }
                    var stringNow = GetTimeStamp();
                    var data = stringNow + " " + args.Data;
                    if (DebugLogFlag)
                    {
                        Console.WriteLine(data);
                    }
                    stringBuilder.AppendLine(args.Data);
                });
            return handler;
        }

        /// <summary>
        /// コマンドラインツールの実行
        /// <param name="toolPath">実行するツールのパス</param>
        /// <param name="argument">実行するツールに渡す引数</param>
        /// <returns>成功:true 失敗:false</returns>
        /// </summary>
        private static bool InvokeCommandLineTool(string toolPath, string argument)
        {
            var result = false;
            var redirectString = string.Empty;
            result = InvokeCommandLineTool(toolPath, argument, out redirectString);
            return result;
        }

        /// <summary>
        /// コマンドラインツールの実行
        /// <param name="toolPath">実行するツールのパス</param>
        /// <param name="argument">実行するツールに渡す引数</param>
        /// <param name="redirectString">実行時のコンソール出力</param>
        /// <returns>成功:true 失敗:false</returns>
        /// </summary>
        private static bool InvokeCommandLineTool(string toolPath, string argument,
            out string redirectString)
        {
            var result = false;
            // プロセスを生成してツールを実行する
            using (var process = new Process())
            {
                // 使用する実行ファイルと引数の出力
                var fileNameLog = GetTimeStamp() + " FileName: " + toolPath;
                Console.WriteLine(fileNameLog);
                var argumentLog = GetTimeStamp() + " Argument: " + argument;
                Console.WriteLine(argumentLog);
                // コンソールに実行ログを出力するためのイベントハンドラ取得
                var logStringBuilder = new StringBuilder();
                var handler = GetDataReceivedEventHandler(logStringBuilder);
                // コンソールに実行ログを出力するためのイベントハンドラ設定
                process.OutputDataReceived += handler;
                process.ErrorDataReceived += handler;
                // ツールのパス設定
                process.StartInfo.FileName = toolPath;
                // ツールに引数設定
                process.StartInfo.Arguments = argument;
                // コンソールの出力をリダイレクト
                process.StartInfo.RedirectStandardOutput = true;
                // コンソールのエラー出力をリダイレクト
                process.StartInfo.RedirectStandardError = true;
                // プロセスを実行可能ファイルから直接作成
                process.StartInfo.UseShellExecute = false;
                // 実行開始
                process.Start();
                process.BeginOutputReadLine();
                process.BeginErrorReadLine();
                // 実行完了まで待つ
                process.WaitForExit();
                Console.WriteLine("InvokeCommandLineTool: exit code = {0}", process.ExitCode);
                redirectString = logStringBuilder.ToString();
                result = (process.ExitCode == 0) ? true : false;
            }
            return result;
        }

        /// <summary>
        /// TargetShellのユーティリティ用のエクセプション
        /// </summary>
        public class UtilityException : Exception
        {
            /// <summary>
            /// Utilityのエクセプション処理
            /// </summary>
            public UtilityException(string message) : base(message)
            {
            }
        }

        /// <summary>
        /// 例外が発生したときに例外を画面に表示する
        /// </summary>
        /// <param name="exception">発生した例外</param>
        public static void PrintException(Exception exception)
        {
            Console.WriteLine("[ERROR] {0}", exception.Message);
            Console.WriteLine("StackTrace: {0}", exception.StackTrace);
        }
    }
}
