﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TargetShell.CommandPluginBase
{
    using System;
    using System.Collections.Generic;
    using System.Globalization;
    using System.Threading;
    using TargetShell.Library;
    using TargetShell.PluginInterface;

    /// <summary>
    /// 共通のデバイスプラグイン呼び出しクラス
    /// </summary>
    public class BaseCommandPlugin<IDevicePluginInterface,
            CommandParameter, SubCommandArgument> :
            ITargetShellPlugin where SubCommandArgument : new()
    {
        /// <summary>
        /// デバイスプラグイン
        /// </summary>
        public DevicePluginCaller
                <IDevicePluginInterface<CommandParameter>, CommandParameter> DevicePluginCaller
                { get; set; }
        /// <summary>
        /// ログ表示モード
        /// </summary>
        protected bool LogHideModeFlag { get; set; }

        /// <summary>
        /// 初期化を実行する
        /// </summary>
        /// <param name="subCommandArgument">サブコマンド引数</param>
        /// <param name="deviceList">デバイスのリスト</param>
        /// <returns>true：成功  false：失敗</returns>
        public bool InvokePlugin(ISubCommandArgument subCommandArgument,
                List<TargetEntry> deviceList)
        {
            var totalDevices = 0;
            var successDevices = 0;
            try
            {
                RunSubCommand(subCommandArgument, deviceList,
                        out totalDevices, out successDevices);
            }
            catch (TargetShellLibraryException exception)
            {
                Console.WriteLine(
                        "Command Result success {0}/{1}", successDevices, totalDevices);
                TargetShellLibrary.PrintException(exception);
                return false;
            }
            return ((successDevices == totalDevices) ? true : false);
        }

        /// <summary>
        /// デバイスプラグインを読み込む
        /// </summary>
        public void LoadDevicePlugin()
        {
            this.DevicePluginCaller =
                    new DevicePluginCaller
                    <IDevicePluginInterface<CommandParameter>, CommandParameter>();
            this.DevicePluginCaller.Load();
        }

        /// <summary>
        /// デバイスの情報を取得する
        /// <returns>デバイスプラグイン情報</returns>
        /// </summary>
        public List<DevicePluginInfo> GetDevicePluginInfo()
        {
            var devicePluginInfo = this.DevicePluginCaller.GetDevicePluginInfo();
            return devicePluginInfo;
        }

        /// <summary>
        /// サブコマンド引数取得
        /// <returns>サブコマンド引数</returns>
        /// </summary>
        public ISubCommandArgument GetSubCommandArgument()
        {
            var argument = new SubCommandArgument();
            return (ISubCommandArgument)argument;
        }

        /// <summary>
        /// サブコマンド実行準備
        /// </summary>
        /// <param name="subCommandArgument">サブコマンド引数</param>
        public void PrepareRunSubCommand(ISubCommandArgument subCommandArgument)
        {
            // 日付ログディレクトリ生成
            subCommandArgument.OptionArgs.DevicePluginArgs.LogOutputDir =
                    TargetShellLibrary.CreateDateLogDirectory(
                    subCommandArgument.OptionArgs.CommonArgs.OutputDirectory);
            // Verboseコマンド受け渡し
            subCommandArgument.OptionArgs.DevicePluginArgs.Verbose =
                    subCommandArgument.OptionArgs.CommonArgs.Verbose;
        }

        /// <summary>
        /// サブコマンドを実行する
        /// </summary>
        /// <param name="subCommandArgument">引数の情報</param>
        /// <param name="deviceList">サブコマンドを実行するデバイスのリスト</param>
        /// <param name="totalDevices">検知した開発機の数</param>
        /// <param name="successDevices">処理に成功した開発機の数</param>
        public virtual void RunSubCommand(ISubCommandArgument subCommandArgument,
                List<TargetEntry> deviceList, out int totalDevices, out int successDevices)
        {
            var localCommandParameter =
                    TargetShellLibrary.CastParameter<CommandParameter>(
                    subCommandArgument.OptionArgs.DevicePluginArgs);
            Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", true);
            if (!IsRequiredOption(localCommandParameter))
            {
                totalDevices = 0;
                successDevices = 0;
                Console.WriteLine("Required option is not set.");
                return;
            }
            PrepareRunSubCommand(subCommandArgument);
            // デバイスプラグインの対応する開発機のみ取得する
            var sportDevicesInfo = this.DevicePluginCaller.GetDevicePluginInfo();
            var sportDeviceList = new List<TargetEntry>();
            foreach (var sportDeviceInfo in sportDevicesInfo)
            {
                sportDeviceList.AddRange(deviceList.FindAll(device => device.HardwareType ==
                        sportDeviceInfo.SupportHardwareType));
            }
            totalDevices = sportDeviceList.Count;
            if (this.LogHideModeFlag)
            {
                // デバイス数＋ログ表示名の配列を確保する
                subCommandArgument.OptionArgs.DevicePluginArgs.LogHideString =
                        new string[totalDevices];
            }
            var successDeviceCount = this.DevicePluginCaller.RunDevicePlugin(deviceList,
                    localCommandParameter);
            Console.WriteLine(
                    "Command Result success {0}/{1}", successDeviceCount, totalDevices);
            successDevices = successDeviceCount;
            if (this.LogHideModeFlag)
            {
                Console.WriteLine("=============================================================");
                Console.WriteLine(
                        subCommandArgument.OptionArgs.DevicePluginArgs.OutputTitleString);
                foreach (var logString in
                        subCommandArgument.OptionArgs.DevicePluginArgs.LogHideString)
                {
                    Console.WriteLine(logString);
                }
                Console.WriteLine("=============================================================");
            }
        }

        /// <summary>
        /// 必須オプションが設定されているか確認する（固有のチェックを行う場合はオーバーライドすること）
        /// </summary>
        /// <param name="parameter">コマンドパラメータ</param>
        /// <returns>true:設定されている false:設定されていない</returns>
        public virtual bool IsRequiredOption(CommandParameter parameter)
        {
            return true;
        }
    }
}
